# encoding=utf-8
"""
功 能：定制管理面备份目录
版权信息：华为技术有限公司，版本所有(C) 2019-2029
"""
import json
import logging
import os
import shlex
import subprocess
import sys

LOG = logging.getLogger(__name__)
INTALL_PATH = os.getenv("INSTALL_ROOT", '/opt/oss')
NODELISTS_JSON = os.path.join(INTALL_PATH, "manager/etc/sysconf/nodelists.json")
CLOUDSOP_BACKUP_PATH = os.path.join(INTALL_PATH, "manager", "var", "etc", "backuprestore")
ES_BACKUP_PATH = "/opt/upgrade/backup/backup_uniep_config"
ES_LOG_PATH = os.path.join(INTALL_PATH, "log/manager/easysuite_upgrade")
CLOUSOP_OMP_NAME = "management"
ES_MODIFY_FALG = os.path.join(ES_BACKUP_PATH, 'modify_backup_uniep.flag')


def init_log():
    """
    功能说明:初始化日志
    :return:
    """
    # 初始化日志
    LOG.setLevel(logging.DEBUG)
    log_path = os.path.join(ES_LOG_PATH)
    if not os.path.exists(log_path):
        os.mkdir(log_path)
    log_file = os.path.join(log_path, "pre_backup_uniep.py_c_.log")
    fh = logging.FileHandler(log_file, mode="a")
    log_formatter = "[%(asctime)s][%(filename)s][line:%(lineno)d][%(levelname)s] %(message)s"
    formatter = logging.Formatter(log_formatter)
    fh.setFormatter(formatter)
    fh.setLevel(logging.DEBUG)
    ch = logging.StreamHandler()
    ch.setLevel(logging.WARNING)
    ch.setFormatter(formatter)
    LOG.addHandler(fh)
    LOG.addHandler(ch)


class LOGGER():
    @staticmethod
    def replace_key(msg, *args):
        """
        功能说明:屏蔽关键信息打印
        :param msg:
        :param args:
        :return:
        """
        # 非str类型直接返回
        if not isinstance(msg, str):
            return msg
        if args:
            msg = msg % args

        # 关键字段
        replace_str = {"/opt/oss": "{OSS}"}
        for key in replace_str:
            if key in msg:
                msg = msg.replace('/opt/oss', '{OSS}')
        return msg

    @staticmethod
    def info(msg, *args):
        """
        功能说明:统一处理日志打印
        :param msg:
        :param args:
        :return:
        """
        LOG.info(LOGGER.replace_key(msg, *args))

    @staticmethod
    def error(msg, *args):
        """
        功能说明:统一处理日志打印
        :param msg:
        :param args:
        :return:
        """
        LOG.error(LOGGER.replace_key(msg, *args))


def run_cmd(cmd):
    """
    功能处理:下发命令
    :return:
    """
    LOG.info(f"Start to run cmd:{cmd}")
    p_thread = subprocess.Popen(shlex.split(cmd), shell=False)
    ret_code = p_thread.wait()
    LOG.info(f"Finished to run cmd:{cmd}.result:{ret_code}")
    return ret_code


def get_products():
    """
    功能说明:获取产品列表
    :return: product_list -> list: ["NCE"]
    """
    product_list = []
    with open(NODELISTS_JSON, 'r') as file_obj:
        file_data = json.load(file_obj)
        for node in file_data.get("nodeList").values():
            if node.get("assignedToTenancy") not in product_list:
                product_list.append(node.get("assignedToTenancy"))
    if "manager" in product_list:
        product_list.remove("manager")
    return product_list


def backup_config():
    """
    功能说明:备份0号节点管理面备份配置文件
    :return: True 成功 False 失败
    """
    # 目录不存在,直接创建,目录和文件只需要保证ossadm可读写
    run_cmd(f'mkdir -p {ES_BACKUP_PATH}')

    # 删除旧的配置
    run_cmd(f'rm -f {ES_BACKUP_PATH}/backup_management_*.json')

    # 遍历产品和平台
    all_products = get_products()
    all_products.append(CLOUSOP_OMP_NAME)
    # 备份管理面备份参数配置文件
    for product in all_products:
        file_name = f"backup_management_{product}.json"
        src_file = os.path.join(CLOUDSOP_BACKUP_PATH, file_name)
        des_file = os.path.join(ES_BACKUP_PATH, file_name)
        des_bak_file = os.path.join(ES_BACKUP_PATH, f'{file_name}.bak')
        if os.path.isfile(src_file):
            run_cmd(f'cp -fp {src_file} {des_file}')
            run_cmd(f'cp -fp {src_file} {des_bak_file}')
    return True


class PreBackupUniep(object):
    """
    升级框架定制管理面备份排除目录
    """

    @staticmethod
    def config_backup_file(exclude_paths, config_file):
        """
        根据子域配置修改所有管理节点
        """
        # 备份管理面
        backup_file = os.path.join(ES_BACKUP_PATH, config_file)
        try:
            with open(backup_file, "r", encoding="utf-8") as f:
                backup_info = json.load(f)
        except Exception as e:
            print("[Error]open file  error")

        # 记录修改前配置
        LOG.info(f"backup_file data:{backup_info}")
        exclude_path_list = backup_info.get("excludePathList")
        if exclude_paths:
            for exclude_path in exclude_paths:
                exclude_path_dict = {"path": exclude_path}
                if exclude_path_dict not in exclude_path_list:
                    exclude_path_list.append({"path": exclude_path})
        backup_info.update({
            "excludePathList": exclude_path_list
        })

        # 记录修改后配置
        LOG.info(f"backup_file data:{backup_info}")
        try:
            with os.fdopen(os.open(backup_file, os.O_CREAT | os.O_WRONLY | os.O_TRUNC, mode=0o660),
                           'w', encoding='utf-8') as f:
                f.write(json.dumps(backup_info, indent=4))
                f.close()
        except Exception as e:
            print("[Error]open file %s error: %s" % (backup_file, e))

        # 同步定制后的备份文件到所有管理节点上
        PreBackupUniep.scp_sync_file(os.path.join(ES_BACKUP_PATH, config_file),
                                     os.path.join(CLOUDSOP_BACKUP_PATH, config_file))

    @staticmethod
    def query_mgr_ip():
        nodelist_file = os.path.join(os.getenv("INSTALL_ROOT", "/opt/oss"), "manager", "etc",
                                     "sysconf", "nodelists.json")
        try:
            with open(nodelist_file, "r", encoding="utf-8") as f:
                nodes_info = json.load(f)
        except Exception as e:
            print("[Error]open file %s error: %s" % (nodelist_file, e))

        # 过滤出产品节点信息
        nodes_ip_list = []
        for _, node in nodes_info.get("nodeList").items():
            if "MGR" not in node.get("role"):
                continue
            for ip_info in node.get("IPAddresses"):
                if "maintenance" in ip_info.get("usage"):
                    nodes_ip_list.append(ip_info.get("IP"))
                    break
        return nodes_ip_list

    @staticmethod
    def scp_sync_file(src_file, des_file):
        """
        功能说明:同步文件到所有OMP节点
        :param src_file
        :param des_file
        :return:
        """
        # 同步定制后的备份文件到所有管理节点上
        nodes_ip_list = PreBackupUniep.query_mgr_ip()
        for ip in nodes_ip_list:
            run_cmd(f"scp -o StrictHostKeyChecking=no {src_file} ossadm@{ip}:{des_file}")
        return True

    @staticmethod
    def modify_backup_config(exclude_paths):
        """
        功能说明:修改备份配置
        :param exclude_paths:
        :return:
        """
        if os.path.isfile(ES_MODIFY_FALG):
            LOG.info(f"{ES_MODIFY_FALG} is existed.Skip modify")
            # 修改过一次不再修改
            return True

        # 备份修改前的配置
        backup_config()

        for file_name in os.listdir(ES_BACKUP_PATH):
            if not (file_name.startswith("backup_management_") and file_name.endswith(".json.bak")):
                continue
            # 修改备份文件
            PreBackupUniep.config_backup_file(exclude_paths, file_name.replace('.bak', ''))

        # 修改完成生成标志
        os.mknod(ES_MODIFY_FALG)
        return True

    @staticmethod
    def restore_backup_config():
        """
        功能说明:恢复备份配置
        :return:
        """
        if not os.path.isfile(ES_MODIFY_FALG):
            # 没有修改过不恢复
            return True

        # 按备份目录实际文件清单处理
        for file_name in os.listdir(ES_BACKUP_PATH):
            if not (file_name.startswith("backup_management_") and file_name.endswith(".json.bak")):
                continue
            # 恢复对应配置文件
            PreBackupUniep.scp_sync_file(os.path.join(ES_BACKUP_PATH, file_name),
                                         os.path.join(CLOUDSOP_BACKUP_PATH,
                                                      file_name.replace('.bak', '')))

        # 恢复一次删除对应文件
        os.remove(ES_MODIFY_FALG)
        return True


def main(argv):
    """
    定制管理面备份文件
    """
    # argv: exclude_paths, action(modify,restore)
    exclude_paths = argv[1].split(",")

    action = 'modify'
    if len(argv) >= 3:
        action = argv[2]

    # 初始化日志
    init_log()

    # 一次修改对应一次恢复
    if action == 'modify':
        PreBackupUniep.modify_backup_config(exclude_paths)
    else:
        PreBackupUniep.restore_backup_config()
    return True


if __name__ == '__main__':
    main(sys.argv)
