# -*- coding:utf-8 -*-
"""
功 能：检查指定微服务是否正在运行
版权信息：华为技术有限公司，版本所有(C) 2021-2029
修改记录：2021-07-20 17:00 创建
"""
import json
import time
import sys

from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr

logger = Logger().getinstance(sys.argv[0])

class CheckTools:
    @staticmethod
    def fliter_list_status(process_list, status):
        """
        返回待过滤列表的状态是否都符合要求
        :param process_list:待确认列表
        :param status:目标状态列表
        :return:False,不匹配的列表。True，空列表
        """
        nonconformance_list = []
        for one_process in process_list:
            if one_process.get("status") not in status:
                logger.info("The process is not %s %s" % (status, one_process))
                nonconformance_list.append(one_process.get("appName"))
        if nonconformance_list:
            return False, set(nonconformance_list)
        return True, []


    @staticmethod
    def get_exist_services(app_service_list, product_name):
        """
        获取指定产品，指定app列表中的服务状态
        :param app_service_list: 服务查询列表
        :param product_name: 产品名
        :return: True/False: 查询是否成功，process_list指定app状态的列表（不存在的app自动跳过）
        """
        request_function = Unieptaskmgr()
        query_app_process_url = '/rest/plat/productmonitorservice/v1/main/service/expstatus' \
                                '?productName=%s&appName=%s'
        process_list = []
        for service in app_service_list:
            status, response = request_function.send_get_request(
                query_app_process_url % (product_name, service))
            if not status:
                return False, []
            if status and not CheckTools.check_hiro_status(service, response):
                result_dict = json.loads(response).get("result")
                status_dict = json.loads(result_dict).get("statuslist")
                process_list = process_list + status_dict
        return True, process_list

    @staticmethod
    def check_hiro_status(service, response):
        """
        HIROBERService 做特殊处理，有running 状态即可
        :service: 服务名称
        :response:接口回显
        :return: True：忽略检查， False: 不忽略检查
        """
        if service == "HIROBERService":
            result_dict = json.loads(response).get("result")
            status_list = json.loads(result_dict).get("statuslist")
            for status_index in status_list:
                if status_index.get("status") == "RUNNING":
                    return True
        return False

    @staticmethod
    def check_app_status(app_service_list, product_name, try_count=3, action="start"):
        """
        检查指定产品下，指定app列表的状态是否符合要求
        :param app_service_list: 待检查app列表
        :param product_name:产品名
        :param try_count:重试次数
        :param action:启动或失败
        :return: True/False,不符合要求的列表。
        """
        status = ["STOPPED"]
        if action == "start":
            status = ["RUNNING", "STOPPED(STANDBY)"]
        out_list = []
        process_list = []
        for time_count in range(0, try_count):
            result, process_list = CheckTools.get_exist_services(app_service_list, product_name)
            if not result:
                logger.info("Failed to obtain the service list.")
                time.sleep(10)
                continue
            result, out_list = CheckTools.fliter_list_status(process_list, status)
            if result:
                logger.info("True;Process_list:%s" % process_list)
                return True, []
        logger.info("False;Process_list:%s" % process_list)
        return False, out_list


    @staticmethod
    def check_necessary_services(product_name):
        """
        查询产品必要的服务是否在运行，如果服务不存在，会自动跳过
        :param product_name: 产品名称
        :return: True/False
        """
        app_service_list = ['RouterAgent', 'ServiceAwareWatchAgent', 'DRMgrAgent',
                            'DBAgent', 'CloudbProxyService', 'Etcd', 'HIROIRService',
                            'SSAMgrWebsite', 'DBProxyService', 'ServiceCenter', 'OMMHAService',
                            'HIROBERService', 'BusService', 'UniEPAgent', 'NodeAgent']
        return CheckTools.check_app_status(app_service_list, product_name, try_count=3, action="start")

def main(argv):
    if len(argv) < 2:
        print("Please enter a product name.")
        sys.exit(1)
    productname = argv[1]
    result, out_list = CheckTools.check_necessary_services(productname)
    if not result:
        print(out_list)
        sys.exit(1)
    sys.exit(0)

if __name__ == '__main__':
    main(sys.argv)