# encoding=utf-8
"""
功 能：上传备份产品数据包
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2022-05-21 23:17 创建
"""
import sys
import os
import json
import time

from datetime import datetime
from commonlog import Logger
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr

# 日志级别枚举
INFO = "INFO"
ERROR = "ERROR"
LOGGER = Logger().getinstance(sys.argv[0])

# 上传备份包接口
UPLOAD_BACKUP_PACKAGE_URL = f"/rest/plat/brmgr/v1/backupforspec/backupdataaction?action=upload"
# 备份时间戳文件
TIMESTAMP_FILE = '/opt/upgrade/backup/backup_timestamp.properties'
# 升级步骤名
STEP_NAME = "upload_backup_package"

# 任务管理状态
FAILURE_STATE = ["FAILURE", "PARTIAL_SUCCESS", "TERMINATING", "TERMINATED"]
RUNNING_STATE = ["INIT", "RUNNING", "WAITING"]

# 查询进度状态枚举
FINISH = 0
RUNNING = 1
FAULT = 2


def build_upload_param(product_name, timestamp):
    """
    上传参数
    :param product_name：产品名
    :param timestamp:备份数据时间戳
    :return param:备份上传接口参数
    """
    param = dict(
        action="upload",
        mode="upgrade",
        timestamp=timestamp,
        productName=product_name,
        mutexLevel="none",
        backupPath="/opt/oss/backuptmp",
        resource={
            "mode": {
                "en_US": "Upgrade",
                "zh_CN": "升级"
            }
        }
    )
    return param


class UploadBackupPackage:
    """
    上传备份产品数据包
    """

    def __init__(self, args):
        LOGGER.info("Init upload backup product package.")
        self.script_id = args.get("script_id", "")
        self.task_id = args.get("task_id", "")
        self.site = args.get("site", "")
        self.product_name = args.get("product_name", "")
        self.action = args.get("action", "")

        e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
        self.path = os.path.join(e_taskmgr_path, self.script_id)
        self.rest = Unieptaskmgr()
        self.taskmgr = Taskmgrutil()
        self.taskmgr.init_e_taskmgr(self.path)
        self.msg_list = []

    def record_log(self, msg, level=""):
        """
        更新任务process、status、和日志
        :param msg:
        :param level:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        str_msg = "[%s] [%s] %s | %s" % (now_time, os.getpid(), level, msg)
        self.msg_list.append(str_msg)
        self.taskmgr.set_e_taskmsg(self.path, "\n".join(self.msg_list))
        if level == ERROR:
            self.taskmgr.set_e_taskstatus(self.path, "error")
            self.taskmgr.set_e_taskprogress(self.path, "100")
        return True

    def check_progress(self, url):
        """
        检查上传进度
        :param url:查询接口
        :return:
        """
        try:
            status, result = self.rest.send_get_request(url)
            if status:
                result = json.loads(result.decode())[0]
                execute_result = result.get("currentState")
                progress = result.get("progress", "0")
                details = result.get("details", "")
                if not execute_result or execute_result in FAILURE_STATE:
                    return FAULT, progress, details
                if execute_result in RUNNING_STATE:
                    return RUNNING, progress, details
                return FINISH, progress, details
            return FAULT, "0", ""
        except (TypeError, ValueError) as e_msg:
            self.record_log("Failed to query task msg. Exception:{}".format(e_msg), ERROR)
            LOGGER.error("Failed to query task msg. Exception:{}".format(e_msg))
            return FAULT, "0", ""

    def query_progress(self, url):
        """
        查询上传任务进度
        :param url:查询接口
        :return:
        """
        # 升级只创建任务，不查询任务结果
        if self.action == "upgrade":
            self.taskmgr.set_e_taskstatus(self.path, "finish")
            self.taskmgr.set_e_taskprogress(self.path, "100")
            self.record_log("Finish to create upload task. "
                            "Tips: this step does not affect the upgrade process.", INFO)
            LOGGER.info("Finish to create upload task. script_id:{}.".format(self.script_id))
            return True

        # 查询上传任务进度
        self.record_log("Start to query upload task progress.", INFO)
        LOGGER.info("Start to query upload task progress.")
        max_wait_seconds = 14400
        delay_seconds = 10
        end_time = int(time.monotonic()) + max_wait_seconds
        status = RUNNING
        while status != FINISH:
            status, progress, details = self.check_progress(url)
            if status == FAULT:
                self.record_log("Failed to query upload task progress.Execution details:\n%s" % details, ERROR)
                LOGGER.error("Failed to query upload task progress.")
                return False
            self.record_log(
                "Check upload process is running, execution progress is %s%%" % progress, INFO)
            if status == FINISH:
                self.taskmgr.set_e_taskstatus(self.path, "finish")
                self.taskmgr.set_e_taskprogress(self.path, "100")
                self.record_log("Finish to execute upload task.Execution details:\n%s" % details, INFO)
                LOGGER.info("Finish to execute upload task. script_id:{}.".format(self.script_id))
                return True
            time.sleep(delay_seconds)
            if int(time.monotonic()) >= end_time:
                self.record_log(
                    "Check upload progress timeout, waiting for %d seconds" % max_wait_seconds, ERROR)
                LOGGER.error("Check upload progress timeout.")
                return False
        return True

    def upload_backup_package(self):
        """
        上传备份包
        :return:
        """
        self.record_log("Start to upload backup product package.", INFO)
        LOGGER.info("Start to upload backup product package")

        # 查最近的一条备份完成的时间戳
        timestamp = self.query_timestamp()

        # 构造上传接口参数
        upload_param = build_upload_param(self.product_name, timestamp)

        try:
            status, result = self.rest.send_post_request(UPLOAD_BACKUP_PACKAGE_URL, upload_param)
            if status:
                result = json.loads(result.decode())
                task_url = result.get("url", "")
                if task_url:
                    return self.query_progress(task_url)
                task_result = result.get("result", "")
                task_entity = result.get("entity", None)
                if task_result == "ERROR" and task_entity is not None:
                    self.record_log(
                        "Failed to create task. Error msg:{}".format(task_entity.get("details", "None")), ERROR)
                    LOGGER.error(
                        "Failed to create task. Error msg:{}".format(task_entity.get("details", "None")))
                else:
                    self.record_log("Failed to upload backup product package.", ERROR)
                    LOGGER.error("Failed to upload backup product package.")
                return False
            self.record_log("Failed to upload backup product package.", ERROR)
            LOGGER.error("Failed to upload backup product package.")
            return False
        except (TypeError, ValueError) as e_msg:
            self.record_log("Failed to upload backup product package. Exception:{}".format(e_msg), ERROR)
            LOGGER.error("Failed to upload backup product package. Exception:{}".format(e_msg))
            return False

    def query_timestamp(self):
        """
        查询备份时间戳
        """
        timestamp = ""
        timestamp_str = f"backup_product_{self.product_name}_{self.site}_{self.task_id}"
        timestamp_file_content = open(TIMESTAMP_FILE, "r", encoding="utf-8")
        for line in timestamp_file_content:
            if line.find(timestamp_str) > -1:
                timestamp = line.split(":")[1].strip()
        self.record_log(f"timestamp is {timestamp}", INFO)
        LOGGER.info(f"timestamp is {timestamp}")
        return timestamp

    def check_finish(self):
        """
        根据script_id检查上次任务是否成功
        :return:
        """
        if STEP_NAME not in self.script_id:
            return False
        cmd = "grep %s %s" % (self.script_id, LOGGER.logfile)
        ret_code = os.system(cmd)
        if ret_code == 0:
            self.record_log("Finish to execute upload backup product package task.", INFO)
            self.taskmgr.set_e_taskstatus(self.path, "finish")
            self.taskmgr.set_e_taskprogress(self.path, "100")
            return True
        return False


def main(argv):
    """
    上传产品数据备份包
    :return:
    """
    script_id = argv[1]
    task_id = argv[2]
    site = argv[3]
    product_name = argv[4]
    action = argv[5]
    args = {}
    args.update({
        "script_id": script_id,
        "task_id": task_id,
        "site": site,
        "product_name": product_name,
        "action": action
    })

    upload_function = UploadBackupPackage(args)
    if not upload_function.check_finish():
        result = upload_function.upload_backup_package()
        return result
    return True


if __name__ == '__main__':
    main(sys.argv)
