#!/bin/bash

##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

##################################################
# 初始化日志路径和日志文件名
##################################################
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"

# easysuite临时路径
EASYSUITE_TMP_PATH="/opt/oss/easysuite_tmp"

# 平台ssh连接工具
SSHTOOL="/opt/oss/manager/tools/common/sshmgr/sshmgr.sh"
BACKUP_TIMESTAMP="/opt/upgrade/backup/backup_timestamp.properties"
OSS_USER=$(id -nu 3001)

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "${OSS_USER}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 初始化
##################################################
function init_param()
{
    # 任务ID
    TASK_ID="$1"
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASK_ID}"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"
    NEW_TASK="TRUE"
    [ -d ${TASK_ID_PATH} ] && NEW_TASK="FALSE"
    # 软件包路径
    upload_pkg_path="${2}"
    # 签名工具包名
    sign_tool_pkg="${3}"
    # 内层签名工具包名
    internal_sign_tool_pkg="${4}"
    # 恢复是否使用BKSignTool,老版本使用BKSignTool,新版本使用osconfig
    USE_BKSIGN_TOOL="false"
}

# 记录Easysuite框架日志及前台显示日志
function DOUBLE_LOG()
{
    TASK_LOG "$1" "$2"
}

# 刷新进度
function fresh_result()
{
    echo "Progress=$1" >${TASK_PROGRESS_FILE} || DOUBLE_LOG "ERROR" "Failed to echo 'Progress=$1' to ${TASK_PROGRESS_FILE}"
    echo "Status=$2" >${TASK_STATUS_FILE} || DOUBLE_LOG "ERROR" "Failed to echo 'Status=$2' to ${TASK_STATUS_FILE}"
}

# 任务失败刷新标志
function refresh_fail()
{
    fresh_result "100" "fail"
    DOUBLE_LOG "ERROR" "Failed to excute task:${TASK_ID}."
}

# 任务完成刷新标志
function fresh_finish()
{
    fresh_result "100" "finish"
    DOUBLE_LOG "INFO" "Finish to excute task:${TASK_ID}."
}

# 刷新进度
function fresh_progress()
{
    local addnum=$1
    local base=$(expr 100 - ${addnum})
    if [ ${PROGRESS} -le ${base} ]
    then
        PROGRESS=$(expr ${PROGRESS} + ${addnum})
        echo "Progress=${PROGRESS}">${TASK_PROGRESS_FILE}
    fi
}

##################################################
# 恢复OMP节点数据库软件
##################################################
function restore_db_software()
{
    # 执行恢复参数
    local src_or_dst=${1}
    local site=${2}
    local node_ip=${3}
    local management=${4}

    if [ ${USE_BKSIGN_TOOL} == "true" ]
    then
      local bksigntool_pkg=${5}
      local sudo_scripts="rebuildDb.sh"
      if [ "${site}" == "master" ]
      then
          local sudo_scripts="restoreDb.sh"
      fi
      local sudu_tmp_path="/opt/backupManagement"

      # 加载平台环境变量获取SUDOBIN路径
      . /opt/oss/manager/bin/engr_profile.sh &>/dev/null
      if [ -z "${SUPER_ROOT}" ]
      then
          SUPER_ROOT="/usr/local/uniepsudobin"
      fi
      local sudu_execute="${SUPER_ROOT}/execute.sh"

      DOUBLE_LOG "INFO" "Start restoring the management plane database software, node:${node_ip}."

      # 执行恢复OMP
      exec_restore_db_cmd="sudo ${sudu_execute} ${bksigntool_pkg} ${sudu_tmp_path} ${sudo_scripts} ${management}"
    else
      local restore_scripts="restoreManagementDB.sh"
      local start_path="/usr/local/osconfig/os/bin/startup.sh"

      local site_input="MASTER"
      if [ "${site}" == "slave" ]
      then
          local site_input="SLAVE"
      fi

      DOUBLE_LOG "INFO" "Start restoring the management plane database software, node:${node_ip}."

      # 执行恢复OMP
      exec_restore_db_cmd="${start_path} -pkg ${management}  -signtype cms -cmd ${restore_scripts}  -params -role ${site_input}"
    fi

    if [ "${src_or_dst}" == "src" ]
    then
        ${exec_restore_db_cmd} &>>${LOG_FILE}
        local res=$?
    else
        # 3600,平台sshmgr BUG内部超时600秒不可修改
        ssh_cmd="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 ${node_ip} "
        ${ssh_cmd} "${exec_restore_db_cmd}" &>>${LOG_FILE}
        local res=$?
        LOG "INFO" "restore_db_software.${node_ip} ${exec_restore_db_cmd} result:${res}"
        bash ${SSHTOOL} -exectype cmd -ip ${node_ip} -cmd "rm -f ${management}*" -u ${OSS_USER} -timeout 60 -encrypt N &>>${LOG_FILE}
    fi
    if [ ${res} -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to restore the management plane database software, node:${node_ip}."
        return 1
    fi
    DOUBLE_LOG "INFO" "Succeeded in restoring the management plane database software, node:${node_ip}."
}

##################################################
# 停止管理面数据库实例
##################################################
function stop_db_instance()
{
    local exec_node_ip=$1
    local stop_db_instance_cmd=". /opt/oss/manager/bin/engr_profile.sh;ipmc_adm -cmd stopdc -tenant manager"
    DOUBLE_LOG "INFO" "Start to stop the management plane database instance, node:${exec_node_ip}."

    for retry in $(seq 3)
    do
        ssh_cmd="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 ${exec_node_ip} "
        ${ssh_cmd} "${stop_db_instance_cmd}" &>>${LOG_FILE}
        local res=$?
        LOG "INFO" "stop_db_instance.${exec_node_ip} ${stop_db_instance_cmd} result:${res}"
        if [ ${res} -eq 0 ]
        then
            DOUBLE_LOG "INFO" "Succeeded in stopping the management plane database instance, node:${exec_node_ip}"
            return 0
        fi
    done
    DOUBLE_LOG "ERROR" "Failed to stop the management plane database instance, node:${exec_node_ip}"
    return 1
}

##################################################
# 下载备份的管理面包
##################################################
function download_backup_pkg()
{
    if [ -f "${BACKUP_TIMESTAMP}" ]
    then
        local backup_data="$(grep 'backup_single_uniep_' ${BACKUP_TIMESTAMP} | tail -n1 | sed 's/ //g')"
        local latest_path="$(echo ${backup_data} | awk -F: '{print $2}')"
        DOUBLE_LOG "INFO" "Start to download the backup file:${latest_path}."
    else
        DOUBLE_LOG "INFO" "Start to download the backup files."
    fi
    bash ${SCRIPT_PATH}/restore_db_tool.sh download_backup_pkg

    # 检查下载的备份包结果
    for NODE in ${OMP_NODE_NAME//,/ }
    do
        ls /opt/upgrade/${NODE}/management.tar.gz &>/dev/null
        local pakg_res=$?
        ls /opt/upgrade/${NODE}/management.tar.gz.sign &>/dev/null
        local sign_res=$?
        if [ ${pakg_res} -ne 0 -o ${sign_res} -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to download the backup file of the management plane."
            return 1
        fi

        sign_key=$(cat /opt/upgrade/${NODE}/management.tar.gz.sign | grep "SIGN_MODE" )
        # 无标志字段表示需要使用BKSignTool工具进行恢复
        [ $? -ne 0 ] && USE_BKSIGN_TOOL="true"
        LOG "INFO" "The flag field in the software package is ${sign_key}"
    done
    DOUBLE_LOG "INFO" "Finished to download the backup file of the management plane."
    fresh_progress 10
}

##################################################
# 确定恢复工具包
##################################################
function query_bk_tool()
{
  if [ ${USE_BKSIGN_TOOL} == "true" ]
  then
    if [ -n "${internal_sign_tool_pkg}" ]
    then
        unzip -oq ${upload_pkg_path}/${sign_tool_pkg} -d ${upload_pkg_path} 2>/dev/null
        sign_tool_pkg=${internal_sign_tool_pkg}
    fi
    BKSIGNTOOL_PKG="$(ls ${upload_pkg_path}/${sign_tool_pkg} 2>/dev/null)"
    if [ -z "${BKSIGNTOOL_PKG}" -o ! -f "${BKSIGNTOOL_PKG}" ]
    then
        DOUBLE_LOG "ERROR" "The ${BKSIGNTOOL_PKG} package cannot be found in the ${upload_pkg_path} directory."
        return 1
    fi
  else
    BKSIGNTOOL_PKG=""
  fi
  fresh_progress 10
}

##################################################
# 恢复&小型化场景数据库
##################################################
function restore_single_db()
{
    DOUBLE_LOG "INFO" "Restore node info: ${OMP_NODE_LIST}."
    first_manager_node=$(echo ${OMP_NODE_NAME//,/ } | awk '{print $1}')
    local management=$(ls /opt/upgrade/${first_manager_node}/management.tar.gz 2>/dev/null)
    local master_node_ip=$(echo ${OMP_NODE_LIST} | awk -F',' '{print $2}')
    restore_db_software "src" "master" ${master_node_ip} ${management} ${BKSIGNTOOL_PKG}
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to restore db:${OMP_NODE_LIST}."
        return 1
    fi
    DOUBLE_LOG "INFO" "Finished to restore db:${OMP_NODE_LIST}."
    fresh_progress 100
}

##################################################
# 恢复分布式场景数据库
##################################################
function restore_multi_db()
{
    DOUBLE_LOG "INFO" "Restore node info: ${OMP_NODE_LIST}."

    local master_node_info=$(echo ${OMP_NODE_LIST} | awk -F';' '{print $2}')
    local standby_node_info=$(echo ${OMP_NODE_LIST} | awk -F';' '{print $4}')
    
    local master_node_id=$(echo ${master_node_info} | awk -F',' '{print $1}')
    local standby_node_id=$(echo ${standby_node_info} | awk -F',' '{print $1}')
    
    local master_node_ip=$(echo ${master_node_info} | awk -F',' '{print $2}')
    local standby_node_ip=$(echo ${standby_node_info} | awk -F',' '{print $2}')
    for NODE in ${OMP_NODE_NAME//,/ }
    do
        ls /opt/upgrade/${NODE}/management.tar.gz &>/dev/null
        local pakg_res=$?
        ls /opt/upgrade/${NODE}/management.tar.gz.sign &>/dev/null
        local sign_res=$?
        if [ ${pakg_res} -ne 0 -o ${sign_res} -ne 0 ]
        then
            DOUBLE_LOG "ERROR" "Failed to download the backup file of the management plane."
            return 1
        fi
    done


    ifconfig | grep -w "${master_node_ip}" &>/dev/null
    if [ $? -eq 0 ]
    then
        local master_management=$(ls /opt/upgrade/$(echo ${OMP_NODE_NAME//,/ } | awk '{print $1}')/management.tar.gz)
        local standby_management=$(ls /opt/upgrade/$(echo ${OMP_NODE_NAME//,/ } | awk '{print $2}')/management.tar.gz)
    else
        local master_management=$(ls /opt/upgrade/$(echo ${OMP_NODE_NAME//,/ } | awk '{print $2}')/management.tar.gz)
        local standby_management=$(ls /opt/upgrade/$(echo ${OMP_NODE_NAME//,/ } | awk '{print $1}')/management.tar.gz)
    fi

    
    local tmp_management_pkg="${EASYSUITE_TMP_PATH}/management.tar.gz"
    if [ ${USE_BKSIGN_TOOL} == "false" ]
    then
      local tmp_bksigntool_pkg=""
      BKSIGNTOOL_PKG=""
    else
      local tmp_bksigntool_pkg="${EASYSUITE_TMP_PATH}/$(basename ${BKSIGNTOOL_PKG})"
    fi

    # 根据备份时的主备顺序,停备,再停主
    stop_db_instance ${standby_node_ip}
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to Stop the Database Instance at the Standby Site of the Management Plane."
        return 1
    fi
    fresh_progress 10

    # DBM确认60s主备发生倒换,为保证备份后发生过倒换的场景,正常恢复管理面数据库软件;
    sleep 60

    stop_db_instance ${master_node_ip}
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to Stop the Database Instance at the Standby Site of the Management Plane."
        return 1
    fi

    fresh_progress 10
    local restore_result=0
    # 当前是主节点
    ifconfig | grep -w "${master_node_ip}" &>/dev/null
    if [ $? -eq 0 ]
    then
        # 本地运行主节点恢复
        restore_db_software "src" "master" ${master_node_ip} ${master_management} ${BKSIGNTOOL_PKG}
        restore_result=$?
        if [ ${restore_result} -eq 0 ]
        then
            fresh_progress 30
            # 拷贝包到备节点
            bash ${SSHTOOL} -exectype cmd -ip ${standby_node_ip} -cmd "mkdir -p ${EASYSUITE_TMP_PATH}" -u ${OSS_USER} -timeout 60 -encrypt N &>>${LOG_FILE}
            ipv6_flag=":"
            if [[ "$standby_node_ip" =~ $ipv6_flag ]]
            then
              echo "standby_node_ip is ipv6"
              scp ${standby_management} ${standby_management}.sign ${BKSIGNTOOL_PKG} [${standby_node_ip}]:${EASYSUITE_TMP_PATH}
            else
              echo "standby_node_ip is ipv4"
              scp ${standby_management} ${standby_management}.sign ${BKSIGNTOOL_PKG} ${standby_node_ip}:${EASYSUITE_TMP_PATH}
            fi
            restore_result=$?
            if [ ${restore_result} -eq 0 ]
            then
                # 远程运行备节点恢复
                restore_db_software "dst" "standby" ${standby_node_ip} ${tmp_management_pkg} ${tmp_bksigntool_pkg}
                restore_result=$?
            fi
        fi
    # 当前是备节点
    else
        # 拷贝包到主节点
        bash ${SSHTOOL} -exectype cmd -ip ${master_node_ip} -cmd "mkdir -p ${EASYSUITE_TMP_PATH}" -u ${OSS_USER} -timeout 60 -encrypt N &>>${LOG_FILE}
        ipv6_flag=":"
        if [[ "$standby_node_ip" =~ $ipv6_flag ]]
        then
          echo "standby_node_ip is ipv6"
          scp ${master_management} ${master_management}.sign ${BKSIGNTOOL_PKG} [${master_node_ip}]:${EASYSUITE_TMP_PATH}
        else
          echo "standby_node_ip is ipv4"
          scp ${master_management} ${master_management}.sign ${BKSIGNTOOL_PKG} ${master_node_ip}:${EASYSUITE_TMP_PATH}
        fi
        restore_result=$?
        if [ ${restore_result} -eq 0 ]
        then
            fresh_progress 30
            # 远程运行主节点恢复
            restore_db_software "dst" "master" ${master_node_ip} ${tmp_management_pkg} ${tmp_bksigntool_pkg}
            restore_result=$?
            if [ ${restore_result} -eq 0 ]
            then
                # 本地备节点恢复
                restore_db_software "src" "standby" ${standby_node_ip} ${standby_management} ${BKSIGNTOOL_PKG}
                restore_result=$?
            fi
        fi
    fi
    if [ ${restore_result} -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to restore db:${OMP_NODE_LIST}."
        return 1
    fi
    fresh_progress 30
    DOUBLE_LOG "INFO" "Finished to restore db:${OMP_NODE_LIST}."
}


function clean_backup_file()
{
    # 恢复成功,清理BKSigntool工具包和下载的管理面备份包
    if [ -n "${BKSIGNTOOL_PKG}" ]
    then
        find ${BKSIGNTOOL_PKG}* >/dev/null 2>&1
        [ $? -eq 0 ] && find ${BKSIGNTOOL_PKG}* | xargs rm -rf >/dev/null 2>&1
    fi

    # 检查下载的备份包结果
    for NODE in ${OMP_NODE_NAME//,/ }
    do
        rm -rf /opt/upgrade/${NODE} >/dev/null 2>&1
    done

    # 固定等待180秒
    sleep 180

    if [ -f "${BACKUP_TIMESTAMP}" ]
    then
        local backup_data="$(grep 'backup_single_uniep_' ${BACKUP_TIMESTAMP} | tail -n1)"
        sed -i "/${backup_data}/d" "${BACKUP_TIMESTAMP}" 2>/dev/null
    fi

    DOUBLE_LOG "INFO" "OMP node database software rolledback successfully."
}

##################################################
# 恢复管理面数据库
##################################################
function restore_omp_software()
{
    PROGRESS=0
    OMP_NODE_NAME=$(bash ${SCRIPT_PATH}/restore_db_tool.sh get_manage_node)
    download_backup_pkg
    if [ $? -ne 0 ]
    then
        refresh_fail
        return 1
    fi

    # 获取管理面节点IP
    OMP_NODE_LIST=$(bash ${SCRIPT_PATH}/restore_db_tool.sh 'get_omp_node_ip')
    if [ -z "${OMP_NODE_LIST}" ]
    then
        DOUBLE_LOG "ERROR" "Failed to obtain OMP node information."
        refresh_fail
        return 1
    fi

    # 获取恢复工具包
    query_bk_tool
    if [ $? -ne 0 ]
    then
        refresh_fail
        return 1
    fi
    
    # 恢复数据库
    echo "${OMP_NODE_LIST}" | grep 'standby' &>/dev/null
    if [ $? -ne 0 ]
    then
        restore_single_db
        if [ $? -ne 0 ]
        then
            refresh_fail
            return 1
        fi
    else
        restore_multi_db
        if [ $? -ne 0 ]
        then
            refresh_fail
            return 1
        fi
    fi

    # 清理标记文件
    clean_backup_file

    fresh_finish
}

# 检查上次任务是否完成
function check_finish()
{
    if [ "${NEW_TASK}" == "FALSE" ]
    then
        grep -q "Finish to excute task:${TASK_ID}." ${LOG_FILE}
        if [ $? -eq 0 ]
        then
            fresh_finish
            return 0
        fi
    fi
    return 1
}

function init()
{
    check_user || return 1
    # 校验入参，初始化参数
    init_param "$@"
    # 加载公共方法
    . ${SCRIPT_PATH}/common.sh
    LOG "INFO" "Start to init ${LOG_FILE}."
}

##################################################
# 脚本主入口
##################################################
function main()
{
    init "$@" || return 1
    
    init_taskmgr ${TASK_ID}
    check_finish && return 0

    restore_omp_software
    return $?
}

main "$@"
exit $?