#!/bin/bash
##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

##################################################
# 初始化日志路径和日志文件名
##################################################
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
OSS_USER=$(id -nu 3001)

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "${OSS_USER}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

# 定义全局变量
function init_path()
{
    # 平台升级数据库脚本
    DB_UPGRADE="${OSS_ROOT}/agent/tools/shscript/upgradedb.sh"

    # 平台ssh命令工具
    SSHTOOL="${OSS_ROOT}/tools/common/sshmgr/sshmgr.sh"
}


##################################################
# 日志记录进task.log
##################################################
function TASK_LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a ${TASK_LOG_FILE}
}

# 记录双份日志，防止task.log被清理
function DOUBLE_LOG()
{
    TASK_LOG "$1" "$2"
    LOG "$1" "$2"
}

# 入参校验
function check_param()
{
    local check_tmp=$1
    [ -z "${check_tmp}" ] && return 1
    echo "${check_tmp}" | fgrep -q "../"
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "ERROR" "The input param \"${check_tmp}\" is invalid."
        return 1
    fi
}

# 获取输入参数
function init_params()
{
    local num=$#
    if [ $((${num}%2)) -eq 1 ]
    then
        num=$((${num}/2+1))
    else
        num=$((${num}/2))
    fi
    local count=1
    for((i=1;i<=${num};i++))
    do
        [[ ${1#-} == "productname" ]] && { PRODUCTNAME=$2;shift 2;check_param "${PRODUCTNAME}" || return 1; continue; }
        [[ ${1#-} == "taskid" ]] && { TASKID=$2;shift 2;check_param "${TASKID}" || return 1; continue; }
        [[ ${1#-} == "postStartService" ]] && { POSTSTARTSERVICE=$2;shift 2;check_param "${POSTSTARTSERVICE}" || return 1; continue; }
        [[ ${1#-} == "stop_app" ]] && { STOPAPP=$2;shift 2;check_param "${STOPAPP}" || return 1; continue; }
        [[ ${1#-} == "start_app" ]] && { STARTAPP=$2;shift 2;check_param "${STARTAPP}" || return 1; continue; }
    done
}

##################################################
# 校验入参
# @param
#   action              --upgrade/retry
#   productname         --NCE
#   domainconfig        --NCE-IP:config,NCE-Analysis:test
#   taskid              --c6bfc6d0a769c2b319f9
##################################################
function check_input()
{
    init_params "$@"
    if [ $? -ne 0 ]
    then
        echo "Example: bash ${SELF_FILE} -productname NCE -taskid c6bfc6d0a769c2b319f9"
        DOUBLE_LOG "ERROR" "The input param is invalid."
        return 1
    fi
}

# 检查数据库升级脚本是否存在
function precheck()
{
    DOUBLE_LOG "INFO" "Start to check necessary files."
    if [ ! -f "${DB_UPGRADE}" ]
    then
        DOUBLE_LOG "ERROR" "The database upgrade script does not exist. Path:${DB_UPGRADE}"
        return 1
    fi
    DOUBLE_LOG "INFO" "Finish to check necessary files."
}


# 刷新进度
function fresh_result()
{
    echo "Progress=$1" >${TASK_PROGRESS_FILE} || DOUBLE_LOG "ERROR" "Failed to echo 'Progress=$1' to ${TASK_PROGRESS_FILE}"
    echo "Status=$2" >${TASK_STATUS_FILE} || DOUBLE_LOG "ERROR" "Failed to echo 'Status=$2' to ${TASK_STATUS_FILE}"
}

# 任务失败刷新标志
function refresh_fail()
{
    fresh_result "100" "fail"
    DOUBLE_LOG "ERROR" "Failed to excute task:${TASKID}."
}

# 任务完成刷新标志
function fresh_finish()
{
    fresh_result "100" "finish"
    DOUBLE_LOG "INFO" "Finish to excute task:${TASKID}."
}

# 刷新进度
function fresh_progress()
{
    local addnum=$1
    local base=$(expr 100 - ${addnum})
    if [ ${PROGRESS} -le ${base} ]
    then
        PROGRESS=$(expr ${PROGRESS} + ${addnum})
        echo "Progress=${PROGRESS}">${TASK_PROGRESS_FILE}
    fi
}

# 查询产品信息
function query_info()
{
    DOUBLE_LOG "INFO" "Start to query ${PRODUCTNAME} infomation."
    TMP_INFO="${LOG_PATH}/${PRODUCTNAME}_infomation"
    if [ -d "${TMP_INFO}" ]
    then
        rm -rf "${TMP_INFO}"
    fi
    mkdir -p "${TMP_INFO}"
    bash ${OSS_ROOT}/tools/resmgr/queryproduct.sh -pn ${PRODUCTNAME} -output ${TMP_INFO} >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finish to query ${PRODUCTNAME} infomation."
        fresh_progress 3
    else
        DOUBLE_LOG "ERROR" "Failed to query ${PRODUCTNAME} infomation."
        return 1
    fi
}


##################################################
# 当ipmc返回值为110时，代表管理面服务可能暂不可用
# 等待120s后重试一下，失败后不再重试
##################################################
function stop_app()
{
    NODEIP=$(cat "${TMP_INFO}"/nodes_${PRODUCTNAME}.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['hostlist'][0]['nodemgrip'])" 2>/dev/null)
    if [ -z "${NODEIP}" ]
    then
        DOUBLE_LOG "ERROR" "Failed to query product IP."
        return 1
    fi
    local count=0
    while [ ${count} -lt 2 ]
    do
        let count++
        bash ${SSHTOOL} -exectype cmd -ip ${NODEIP} -cmd "bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd stopnodes -tenant ${PRODUCTNAME} -type app" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
        result=$?
        if [ ${result} -eq 110 ]
        then
            sleep 120
            continue
        elif [ ${result} -eq 0 ]
        then
            return 0
        else
            return 1
        fi
    done
    return 1
}

##################################################
# 停止对应产品服务
##################################################
function stop_app_with_retry()
{
    DOUBLE_LOG "INFO" "Start to stop ${PRODUCTNAME} application."
    stop_app
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finished to stop ${PRODUCTNAME} application."
        fresh_progress 10
    else
        DOUBLE_LOG "ERROR" "Failed to stop ${PRODUCTNAME} application."
        return 1
    fi
}

##################################################
# 当ipmc返回值为110时，代表管理面服务可能暂不可用
# 等待120s后重试一下，失败后不再重试
##################################################
function start_app()
{
    NODEIP=$(cat "${TMP_INFO}"/nodes_${PRODUCTNAME}.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['hostlist'][0]['nodemgrip'])" 2>/dev/null)
    if [ -z "${NODEIP}" ]
    then
        DOUBLE_LOG "ERROR" "Failed to query product IP."
        return 1
    fi
    local count=0
    while [ ${count} -lt 2 ]
    do
        let count++
        bash ${SSHTOOL} -exectype cmd -ip ${NODEIP} -cmd "bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd startnodes -tenant ${PRODUCTNAME} -type app" -u ${OSS_USER} -timeout 3600 -encrypt N >>"${LOG_FILE}" 2>&1
        result=$?
        if [ ${result} -eq 110 ]
        then
            sleep 120
            continue
        elif [ ${result} -eq 0 ]
        then
            return 0
        else
            return 1
        fi
    done
    return 1
}

##################################################
# 启动对应产品服务
##################################################
function start_app_with_retry()
{
    if [ "${POSTSTARTSERVICE}" != "yes" ]
    then
        return 0
    fi
    DOUBLE_LOG "INFO" "Start to start ${PRODUCTNAME} application."
    start_app
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finished to start ${PRODUCTNAME} application."
        fresh_progress 10
    else
        DOUBLE_LOG "ERROR" "Failed to start ${PRODUCTNAME} application."
        return 1
    fi
}

# 检查上次任务是否完成
function check_finish()
{
    if [ "${NEW_TASK}"="FALSE" ]
    then
        grep -q "Finish to excute task:${TASKID}." ${LOG_FILE}
        if [ $? -eq 0 ]
        then
            fresh_finish
            return 0
        fi
    fi
    return 1
}

# 调用平台升级数据库脚本
function invoke_upgrade_db()
{
    DOUBLE_LOG "INFO" "Start to upgrade database software."
    cd ${OSS_ROOT}/agent/tools/shscript/
    bash ${DB_UPGRADE} >>"${TASK_LOG_FILE}" 2>&1
    if [ $? -eq 0 ]
    then
        DOUBLE_LOG "INFO" "Finish to upgrade database software."
        fresh_progress 30
    else
        DOUBLE_LOG "ERROR" "Failed to upgrade database software."
        return 1
    fi
}


# 合设场景数据库升级
function update_mgr_db()
{
    PROGRESS=0

    # 查询产品信息
    query_info
    if [ $? -ne 0 ]
    then
        refresh_fail
        return 1
    fi

    # 停止产品服务
    if [ "${STOPAPP}" == "true" ]
    then
      stop_app_with_retry
      if [ $? -ne 0 ]
      then
          refresh_fail
          return 1
      fi
    fi

    # 启动数据库升级
    invoke_upgrade_db
    if [ $? -ne 0 ]
    then
        refresh_fail
        return 1
    fi

    # 启动产品服务(单管且R20C10及之后版本)
    if [ "${STARTAPP}" == "true" ]
    then
      start_app_with_retry
      if [ $? -ne 0 ]
      then
          refresh_fail
          return 1
      fi
    fi

    fresh_finish
}

function main()
{
    check_user || return 1
    # 加载公共方法
    . ${SCRIPT_PATH}/common.sh
    LOG "INFO" "Start to init ${LOG_FILE}."
    . /opt/oss/manager/bin/engr_profile.sh
    
    # 定义相关路径
    init_path
    
    # 校验入参，初始化参数
    check_input "$@" || return 1
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}"
    TASK_LOG_FILE="${TASK_ID_PATH}/task.log"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"
    
    NEW_TASK="TRUE"
    [ -d ${TASK_ID_PATH} ] && NEW_TASK="FALSE"

    # 检查是否已经全部执行完成
    check_finish && return 0
    
    init_taskmgr ${TASKID}

    # 检查数据库迁移工具是否存在
    precheck
    if [ $? -eq 1 ]
    then
        refresh_fail
        return 1
    fi

    update_mgr_db
    return $?
}

main "$@"
exit $?