#!/bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 1998-2020. All rights reserved. 
# File name     : migrate_databse.sh
# Description   : 数据库数据迁移框架
###########################################################

# 初始化脚本工作路径和脚本名
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

# 当前任务时间戳
TAIL=$(date +%Y%m%d%H%M%S)
    
# 初始化日志路径和日志文件名
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}_${TAIL}.log"

##################################################
# 校验执行用户
# 脚本要求使用oss用户执行
##################################################
function check_user()
{
    local user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script" | tee -a "${LOG_FILE}"
        return 1
    fi
}

##################################################
# 记录Easysuite前台展示日志
# @param
#   $1:日志级别 ERROR/WARN/INFO
#   $2:日志内容
##################################################
function TASK_LOG()
{
    BASH_PID=$$
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | $1 $2" | tee -a "${TASK_LOG_FILE}"
}

##################################################
# 记录Easysuite前台展示日志及后台日志
# @param
#   $1:日志级别 ERROR/WARN/INFO
#   $2:日志内容
##################################################
function DOUBLE_LOG()
{
    TASK_LOG "$1" "$2"
    LOG "$1" "$2"
}

##################################################
# 定义相关全局变量
##################################################
function init_variable()
{
    # Easysuite前台进度值
    PROGRESS=0

    # 升级框架工作目录
    WORK_PATH="/opt/upgrade"

    # Easysuite任务相关日志/进度/状态文件
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}"
    TASK_LOG_FILE="${TASK_ID_PATH}/task.log"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"

    # 是否是新任务标志位
    NEW_TASK="TRUE"
    [ -d "${TASK_ID_PATH}" ] && NEW_TASK="FALSE"

    # 欧方包路径
    OS_MEDIATATION="${UPLOAD}/${PKGNAME}"

    # 欧方包解压路径
    UNZIP_PKG_PATH=${OS_MEDIATATION//.zip/}

    # 欧方包版本
    VERSION=$(echo "${PKGNAME}" | awk -F_ '{print $2}')
    
    # 备份路径
    BACKUP_PATH="/opt/oss/NCEEngr/O2_path/upgrade_config_${VERSION}"
}

##################################################
# 入参格式化，将相关参数赋值给对应变量
##################################################
function init_params()
{
    ARGS=$(getopt -a -o h -l pkgname:,action:,upload:,taskid:,help -- "$@")
    if [ $? != 0 ]
    then
        LOG "ERROR" "Failed to parse the parameter."
        return 1
    fi
    eval set -- "${ARGS}"
    while :
    do
        case $1 in
            --pkgname) PKGNAME=$2;shift ;;
            --action) ACTION=$2;shift ;;
            --upload) UPLOAD=$2;shift ;;
            --taskid) TASKID=$2;shift ;;
            -h|--help) exit 0 ;;
            --) shift;break ;;
            *) LOG "ERROR" "Failed to parse the parameter.";return 1 ;;
        esac
    shift
    done
}


##################################################
# 刷新Easysuite任务标志文件
# @param
#   $1:进度 0-100
#   $2:状态 fail/running/finish
##################################################
function fresh_result()
{
    echo "Progress=$1" >"${TASK_PROGRESS_FILE}" || DOUBLE_LOG "ERROR" "Failed to echo 'Progress=$1' to ${TASK_PROGRESS_FILE}"
    echo "Status=$2" >"${TASK_STATUS_FILE}" || DOUBLE_LOG "ERROR" "Failed to echo 'Status=$2' to ${TASK_STATUS_FILE}"
}

##################################################
# 刷新Easysuite任务标志文件为“失败”状态
##################################################
function fresh_fail()
{
    fresh_result "100" "fail"
    DOUBLE_LOG "ERROR" "Failed to excute task:${TASKID}."
}

##################################################
# 刷新Easysuite任务标志文件为“完成”状态
##################################################
function fresh_finish()
{
    fresh_result "100" "finish"
    DOUBLE_LOG "INFO" "Finished to excute task:${TASKID}."
}

##################################################
# 刷新Easysuite任务进度文件
# @param
#   $1:进度增加值，达到100后不再增加
##################################################
function fresh_progress()
{
    local addnum=$1
    local base=$(expr 100 - ${addnum})
    if [ ${PROGRESS} -le ${base} ]
    then
        PROGRESS=$(expr ${PROGRESS} + ${addnum})
        echo "Progress=${PROGRESS}">"${TASK_PROGRESS_FILE}"
    fi
}

##################################################
# 检查欧方包是否存在
##################################################
function pre_check()
{
    if [ ! -f "${OS_MEDIATATION}" ]
    then
        DOUBLE_LOG "ERROR" "The ${OS_MEDIATATION} package does not exist."
        return 1
    fi
    fresh_progress 10
}


##################################################
# 检查上次任务是否完成，完成直接退出
##################################################
function check_finish()
{
    if [ "${NEW_TASK}" == "FALSE" ]
    then
        grep -q "Finished to excute task:${TASKID}." "${TASK_LOG_FILE}"
        if [ $? -eq 0 ]
        then
            fresh_finish
            return 0
        fi
    fi
    return 1
}


##################################################
# 初始化相关变量，并进行前置检查
##################################################
function init()
{
    # 校验调用用户
    check_user || return 1

    # 加载公共方法
    . "${SCRIPT_PATH}"/common.sh
    LOG "INFO" "Start to init ${LOG_FILE}."
    
    # 校验入参，初始化参数
    init_params "$@" || return 1
    
    # 导入平台环境变量
    . /opt/oss/manager/bin/engr_profile.sh
    
    # 定义任务相关全局变量
    init_variable
    
    # 初始化任务文件夹
    init_taskmgr "${TASKID}"
    
    return 0

}

##################################################
# 欧方执行前置准备
##################################################
function pre_operate()
{
    DOUBLE_LOG "INFO" "Start to init osmediation files."
    [ -d "${UNZIP_PKG_PATH}" ] && rm -rf "${UNZIP_PKG_PATH}"
    unzip -o "${OS_MEDIATATION}" -d "${UNZIP_PKG_PATH}" >/dev/null 2>&1
    if [ $? -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to unzip osmediation package."
        return 1
    fi
    fresh_progress 10
    DOUBLE_LOG "INFO" "Finished to init osmediation files."
}

##################################################
# 升级欧方
##################################################
function upgrade_osmediation()
{
    DOUBLE_LOG "INFO" "Start to upgrade osmediation."
    DOUBLE_LOG "INFO" "bash ${UNZIP_PKG_PATH}/execute_upgrade.sh upgrade ${VERSION} ${OS_MEDIATATION}"
    timeout 1800 bash ${UNZIP_PKG_PATH}/execute_upgrade.sh "upgrade" "${VERSION}" "${OS_MEDIATATION}" &>> ${TASK_LOG_FILE}
    local res=$?
    if [ "${res}" -eq 124 ]
    then
        DOUBLE_LOG "ERROR" "Timeout, failed to upgrade the osmediation."
        return 1
    fi
    if [ "${res}" -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to upgrade the osmediation."
        return 1
    fi
    fresh_progress 30
    DOUBLE_LOG "INFO" "Finished to upgrade osmediation."
}

##################################################
# 回滚欧方
##################################################
function rollback_osmediation()
{
    DOUBLE_LOG "INFO" "Start to rollback osmediation."
    DOUBLE_LOG "INFO" "${BACKUP_PATH}/execute_upgrade.sh rollback ${VERSION} ${OS_MEDIATATION}"
    timeout 1800 bash ${BACKUP_PATH}/execute_upgrade.sh "rollback" "${VERSION}" "${OS_MEDIATATION}" &>> ${TASK_LOG_FILE}
    local res=$?
    if [ "${res}" -eq 124 ]
    then
        DOUBLE_LOG "ERROR" "Timeout, failed to rollback the osmediation."
        return 1
    fi
    if [ "${res}" -ne 0 ]
    then
        DOUBLE_LOG "ERROR" "Failed to rollback the osmediation."
        return 1
    fi
    fresh_progress 30
    DOUBLE_LOG "INFO" "Finished to rollback osmediation."
}


##################################################
# 执行升级或回滚欧方
##################################################
function operate_osmediation()
{
    case ${ACTION} in
      "upgrade")
        pre_check || return 1
        pre_operate || return 1
        upgrade_osmediation || return 1
        ;;
      "rollback")
        rollback_osmediation || return 1
        ;;
      *)
        DOUBLE_LOG "ERROR" "The operation parameter is incorrect." || return 1
        ;;
    esac
}

##################################################
# 欧方升级后置操作
##################################################
function post_operate()
{
    DOUBLE_LOG "INFO" "Start to perform the osmediation post-operations."
    rm -rf "${UNZIP_PKG_PATH}"
    DOUBLE_LOG "INFO" "Finished to perform the osmediation post-operations."
}

##################################################
# 执行升级或回滚欧方
##################################################
function upgrade_rollback_osmediation()
{
    operate_osmediation || { fresh_fail;return 1; }

    post_operate || { fresh_fail;return 1; }

    fresh_finish
}

##################################################
# 入口函数
##################################################
function main()
{
    # 初始化及前置检查
    init "$@" || return 1

    # 重入结果检查
    check_finish && return 0
    
    # 启动数据库数据迁移
    upgrade_rollback_osmediation
    return $?
}

main "$@"
exit $?