#!/bin/bash -

###############################################################################################
# logging level
readonly SYS_LOGGING_DEBUG=0
readonly SYS_LOGGING_INFO=1
readonly SYS_LOGGING_WARN=2
readonly SYS_LOGGING_ERROR=3

###############################################################################################
# Deprecated
# For back-campatability reason, we do not remove it but invoke other function internally instead
function fn_sys_set_logging_level()
{
    fn_sys_set_logging_verbose "$1"

    return 0
}

###############################################################################################
# Specify system wide logging verbose. Logging messages whose level is greater than or equal to this verbose is supposed to be logged
# $1: Optional. Logging verbose
function fn_sys_set_logging_verbose()
{
    g_runtime_logging_verbose="$1"
    [ -z "$g_runtime_logging_verbose" ] && g_runtime_logging_verbose="$SYS_LOGGING_INFO"

    echo "$g_runtime_logging_verbose" | egrep '^[0-3]$' &> /dev/null
    [ "$?" -ne 0 ] && g_runtime_logging_verbose="$SYS_LOGGING_INFO"

    return 0
}

###############################################################################################
function fn_sys_set_logging_trace_id()
{
    g_runtime_logging_trace_id="$1"
    return 0
}

###############################################################################################
function fn_sys_set_suppress_display_flag()
{
    g_suppress_display="$1"

    return 0
}

###############################################################################################
function _fn_sys_remove_outdated_backup()
{
    local l_log_file="$1"
    local l_max_backup_count="$2"

    local l_log_root="`dirname $l_log_file`"

    local l_backup_count=$(ls -tr ${l_log_file}.*.zip | wc -l)
    [ "$l_backup_count" -le "$l_max_backup_count" ] && return 0

    ls -tr ${l_log_file}.[0-9]*.zip | head -n $((l_backup_count-l_max_backup_count)) | xargs -i rm -rf {}

    return 0
}

###############################################################################################
function _fn_sys_rotate_log_file()
{
    local l_log_file="$1"

    local readonly c_log_file_size_threshold=$((20*1024*1024))
    local readonly c_max_backup_count=20

    local l_log_root="$(dirname ${l_log_file})"
    local l_log_file_bytes="$(du -sb ${l_log_file} | awk '{ print $1 }')"
    [ "$l_log_file_bytes" -lt "$c_log_file_size_threshold" ] && return 0

    local l_backup_timestamp=$(date +"%Y%m%d%H%M%S")

    cp -p "$l_log_file" "${l_log_file}.${l_backup_timestamp}"
    echo "[$(date -d today +"%Y-%m-%d %H:%M:%S")] Rollup to this log file. Last backup piece: ${l_log_file}.${l_backup_timestamp}.zip" > "$l_log_file"
    cd "$l_log_root"
    zip -r -m "${l_log_file}.${l_backup_timestamp}.zip" "$(basename ${l_log_file}).${l_backup_timestamp}" > /dev/null
    cd - &> /dev/null

    _fn_sys_remove_outdated_backup "$l_log_file" "$c_max_backup_count"

    return 0
}

###############################################################################################
function _fn_sys_create_log_file_action()
{
    local l_log_file="$1"

    if [ ! -f "$l_log_file" ]; then
        touch "$l_log_file"
        return "$?"
    else
        _fn_sys_rotate_log_file "$l_log_file"
        return "$?"
    fi

    return 0
}

###############################################################################################
function fn_sys_create_log_file()
{
    local l_log_file="$1"

    local readonly c_module="create_log_$(basename $l_log_file)"

    if [ "$?" -eq "0" ]; then
        _fn_sys_create_log_file_action "$l_log_file"
        local l_result="$?"
        return "$l_result"
    fi

    return 0
}

###############################################################################################
# Specify log file
# $1: Required. Absolute path for log file
function fn_sys_set_log_file()
{
    g_log_file="$1"

    return 0
}

###############################################################################################
function _fn_sys_decode_logging_level()
{
    local l_level_number="$1"

    l_code=""
    [ -z "$l_level_number" ] && echo "FUNC" && return 0
    [ "$l_level_number" -eq "$SYS_LOGGING_DEBUG" ] && echo "DEBUG" && return 0
    [ "$l_level_number" -eq "$SYS_LOGGING_INFO" ] && echo "INFO" && return 0
    [ "$l_level_number" -eq "$SYS_LOGGING_WARN" ] && echo "WARNING" && return 0
    [ "$l_level_number" -eq "$SYS_LOGGING_ERROR" ] && echo "ERROR" && return 0

    return 0
}

###############################################################################################
# For internal use only
# $1: Logging level
# $*: Detail message
function _fn_sys_log()
{
    local l_logging_level="$1"
    l_logging_level=${l_logging_level:-$SYS_LOGGING_DEBUG}
    shift

    # Only logging message with logging level greater than or equal to logging verbose can be logged
    [ "$l_logging_level" -lt "${g_runtime_logging_verbose:-$SYS_LOGGING_INFO}" ] && return 0

    local l_logging_time=$(date -d today +"%Y-%m-%d %H:%M:%S.%N")

    local l_level_code=$(_fn_sys_decode_logging_level "$l_logging_level")

    # [0] -> this utility script of _fn_sys_log; [1] -> the utility script of fn_sys_log_xxx; [2] -> the invoking script
    local l_script_name="${BASH_SOURCE[2]}"
    l_script_name="$(basename ${l_script_name})"

    # [0] -> public logging function; [1] -> _fn_log; [2] -> the invoking function
    local l_function_name="${FUNCNAME[2]}"

    # [0] -> the invoking public logging function line; [1] -> outer invoking function
    local l_line_number="${BASH_LINENO[1]}"

    local l_current_user="$(whoami 2> /dev/null)"
    [ -z "$l_current_user" ] && l_current_user="Unkown"

    # format: [timestmap][verbose][script:func:lineno][user][pid] message
    if [ -z "$g_runtime_logging_trace_id" ]; then
        if [ -z "$g_log_file" ] || [ ! -f "$g_log_file" ]; then
            printf "[%s][%s][%s->(%s:%s)][%s][%s] %s\n" "$l_logging_time" "$l_level_code" "$l_script_name" "$l_function_name" "$l_line_number" "$l_current_user" "$$" "$*"
        else
            printf "[%s][%s][%s->(%s:%s)][%s][%s] %s\n" "$l_logging_time" "$l_level_code" "$l_script_name" "$l_function_name" "$l_line_number" "$l_current_user" "$$" "$*" >> "$g_log_file"
        fi
    else
       if [ -z "$g_log_file" ] || [ ! -f "$g_log_file" ]; then
            printf "[%s][%s][%s->(%s:%s)][%s][%s][%s] %s\n" "$l_logging_time" "$l_level_code" "$l_script_name" "$l_function_name" "$l_line_number" "$l_current_user" "$$" "$g_runtime_logging_trace_id" "$*"
        else
            printf "[%s][%s][%s->(%s:%s)][%s][%s][%s] %s\n" "$l_logging_time" "$l_level_code" "$l_script_name" "$l_function_name" "$l_line_number" "$l_current_user" "$$" "$g_runtime_logging_trace_id" "$*" >> "$g_log_file"

        fi
    fi

    return 0
}

###############################################################################################
function fn_sys_log_debug()
{
    _fn_sys_log "$SYS_LOGGING_DEBUG" "$@"
    return 0
}

###############################################################################################
function fn_sys_log_info()
{
    _fn_sys_log "$SYS_LOGGING_INFO" "$@"
    return 0
}

###############################################################################################
function fn_sys_log_warn()
{
    _fn_sys_log "$SYS_LOGGING_WARN" "$@"
    return 0
}

##############################################################################################
function fn_sys_log_error()
{
    _fn_sys_log "$SYS_LOGGING_ERROR" "$@"
    return 0
}

##############################################################################################
function fn_sys_log_param()
{
    local l_param_name="$1"

    local l_param_value=${!l_param_name}
    _fn_sys_log "$SYS_LOGGING_DEBUG" "Param [${l_param_name}] = [${l_param_value}]"

    return 0
}

##############################################################################################
function fn_sys_log_all_param()
{
    local l_param_count="${#@}"
    [ "$l_param_count" -eq 0 ] && _fn_sys_log "$SYS_LOGGING_DEBUG" "Param: no parameters at all" && return 0

    for((i=1; i<=l_param_count; i++)); do
        _fn_sys_log "$SYS_LOGGING_DEBUG" "Param [\$${i}] = [${!i}]"
    done

    return 0
}

##############################################################################################
function fn_sys_log_var()
{
    local l_var_name="$1"

    local l_var_value=${!l_var_name}
    _fn_sys_log "$SYS_LOGGING_DEBUG" "Variable [${l_var_name}] = [${l_var_value}]"

    return 0
}

###############################################################################################
function fn_sys_log_enter()
{
    local l_message="$1"

    local l_script_name="${BASH_SOURCE[2]}"
    l_script_name="$(basename ${l_script_name} | awk -F'.' '{ print $1 }')"

    if [ -z "$l_message" ]; then
        fn_sys_log_info "Enter ${l_script_name}"
    else
        fn_sys_log_info "Enter ${l_script_name}. ${l_message}"
    fi

    return 0
}

###############################################################################################
function fn_sys_log_leave()
{
    local l_message="$1"

    local l_script_name="${BASH_SOURCE[2]}"
    l_script_name="$(basename ${l_script_name} | awk -F'.' '{ print $1 }')"

    if [ -z "$l_message" ]; then
        fn_sys_log_info "Leave ${l_script_name}"
    else
        fn_sys_log_info "Leave ${l_script_name}. ${l_message}"
    fi

    return 0
}

###############################################################################################
function fn_sys_audit_log()
{
    local l_status_code="$1"
    local l_event="$2"
    local l_message="$3"

    which logger &> /dev/null
    [ "$?" -ne "0" ] && return 1

    local l_result="successful"
    [ "$l_status_code" != "0" ] && l_result="failed"

    local l_operate_ip=$(who -m | sed 's#\(^.*(\)\(.*\)\().*$\)#\2#g;s#[[:blank:]]##g')
    [ -z "$l_operate_ip" ] && l_operate_ip="127.0.0.1"

    logger -t ${USER} -p local0.info "${l_event}; ${BASH_SOURCE[1]}; ${l_result}; ${l_operate_ip}; ANA; ${l_message}"

    return 0
}


###############################################################################################
# 1 Ļ־Ϣ
###############################################################################################
function _fn_echo_screen()
{
    info_lever=$1
    info_msg=$2
    local l_script_name="${BASH_SOURCE[2]}"
    l_script_name="$(basename ${l_script_name})"
    local l_logging_time=$(date -d today +"%Y-%m-%d %H:%M:%S.%N")
    echo "[${l_logging_time}][${info_lever}][${l_script_name}] -> [${info_msg}]"
}

###############################################################################################
# 1 Ļinfo־Ϣ
###############################################################################################
function fn_info_screen()
{
    info_msg=$1
    _fn_echo_screen "INFO" "${info_msg}"
    return 0
}

###############################################################################################
# 1 ĻERROR־Ϣ
###############################################################################################
function fn_error_screen()
{
    info_msg=$1
    _fn_echo_screen "ERROR" "${info_msg}"
    return 0
}
