#!/bin/bash
# 初始全局变量
PKG_PATH="/opt/upgrade/ospatch"
ES_TASH_MGR="/opt/upgrade/easysuite_upgrade/taskmgr"
LATEST_OS_VERSION="${PKG_PATH}/LatestOSversion.flag"

# 获取脚本工作路径和脚本名
SELF_FILE=$(basename "${0}")
SCRIPT_PATH=$(cd $(dirname "${0}");pwd)

# 初始化日志路径和日志文件名
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
[ ! -d "${LOG_PATH}" ] && LOG_PATH="/opt/upgrade/os_log"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"

# ACTION参数匹配
PRE_UPGRADE="pre_upgrade_ospatch"
PRE_ROLLBACK="pre_rollback_ospatch"
EXEC_UPGRADE="upgrade"
EXEC_ROLLBACK="rollback"
EXEC_ARB_UPGRADE="upgrade_arb_ospatch"
EXEC_ARB_ROLLBACK="rollback_arb_ospatch"
POST_UPGRADE="upgrade_refresh_patch_status"
POST_ROLLBACK="rollback_refresh_patch_status"

PRE_TASK="|${PRE_UPGRADE}|${PRE_ROLLBACK}|"
EXEC_TASK="|${EXEC_UPGRADE}|${EXEC_ROLLBACK}|${EXEC_ARB_UPGRADE}|${EXEC_ARB_ROLLBACK}|"
POST_TASK="|${POST_UPGRADE}|${POST_ROLLBACK}|"

# 代表不存在备份操作系统任务，1代表存在备份操作系统任务，2代表接口连不通
NO_EXIST_OSBACK_TASK="0"
EXIST_OSBACK_TASK="1"
LINK_INTERFACE_FAILED="2"

##########################################
# 校验入参
##########################################
function check_input()
{
    LOG "INFO" "check_input."
    for i in {1..6}
    do
        [[ ${1#-} == "tool_pkg" ]] && {
            TOOL_PKG=$2
            TOOL_PKG_DIR=$(echo "${TOOL_PKG//.zip/}")
            LOG "INFO" "Tool Package: ${TOOL_PKG}"
            shift 2
            continue
        }
        [[ ${1#-} == "patch_pkg" ]] && {
            PATCH_PKG=$2
            LOG "INFO" "Patch Package: ${PATCH_PKG}"
            shift 2
            continue
        }
        [[ ${1#-} == "copy_path" ]] && {
            COPY_PATH=$2
            LOG "INFO" "Copy Path: ${COPY_PATH}"
            shift 2
            continue
        }
        [[ ${1#-} == "task_id" ]] && {
            TASK_ID=$2
            LOG "INFO" "Task ID: ${TASK_ID}"
            shift 2
            continue
        }
        [[ ${1#-} == "action" ]] && {
            ACTION=$2
            LOG "INFO" "Action: ${ACTION}"
            shift 2
            continue
        }
        [[ ${1#-} == "timeout" ]] && {
            TIMEOUT=$2
            LOG "INFO" "Timeout: ${TIMEOUT}"
            shift 2
            continue
        }
    done
    
    # 检查超时时间,单位:秒
    echo "${TIMEOUT}" | egrep '^[0-9]+$' &>/dev/null || TIMEOUT=$(expr 60 \* 60)
    # 入参不能为空
    if [ -z "${TOOL_PKG}" -o -z "${PATCH_PKG}" -o -z "${COPY_PATH}" -o -z "${TASK_ID}" ]
    then
        LOG "ERROR" "The input param is invalid."
        return 1
    fi
    # 运行模式白名单
    echo "|${PRE_TASK}|${EXEC_TASK}|${POST_TASK}|" | grep "|${ACTION}|" &>/dev/null 
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "The input param is invalid."
        return 1
    fi

    IS_SUSE=1
    if echo "${PATCH_PKG}" | grep -qi "suse"
    then
        LOG "INFO" "Upgrade type: SuSE"
        IS_SUSE=0
    fi

    TASK_PATH="${ES_TASH_MGR}/${TASK_ID}"
    LOG "INFO" "Check input param done."
    return 0
}

##########################################
# 检查压缩包是否存在
# 同时拷贝包至/opt/upgrade/ospatch/pkg
##########################################
function copy_pkg_workpath()
{
    LOG "INFO" "copy package from path."
    mkdir -p "${PKG_PATH}/pkg"
    
    cp -r "${COPY_PATH}/${TOOL_PKG}" "${PKG_PATH}/pkg" &>>${LOG_FILE}
    if [ ! -f "${COPY_PATH}/${TOOL_PKG}" ] && [ ! -f "${PKG_PATH}/pkg/${TOOL_PKG}" ]
    then
        LOG "ERROR" "Check ${COPY_PATH}/${TOOL_PKG} is not exist."
        return 1
    fi
    if [ "${ACTION}" == "${PRE_UPGRADE}" ]
    then
        cp -r "${COPY_PATH}/${PATCH_PKG}" "${PKG_PATH}/pkg" &>>${LOG_FILE}
        if [ ! -f "${COPY_PATH}/${PATCH_PKG}" ] && [ ! -f "${PKG_PATH}/pkg/${PATCH_PKG}" ]
        then
            LOG "ERROR" "Check ${COPY_PATH}/${PATCH_PKG} is not exist."
            return 1
        fi
        cp -r "${COPY_PATH}/${PATCH_PKG}.cms" "${PKG_PATH}/pkg" &>>${LOG_FILE}
        if [ ! -f "${COPY_PATH}/${PATCH_PKG}.cms" ] && [ ! -f "${PKG_PATH}/pkg/${PATCH_PKG}.cms" ]
        then
            LOG "ERROR" "Check ${COPY_PATH}/${PATCH_PKG}.cms is not exist."
        fi
        cp -r "${COPY_PATH}/${PATCH_PKG}.crl" "${PKG_PATH}/pkg" &>>${LOG_FILE}
        if [ ! -f "${COPY_PATH}/${PATCH_PKG}.crl" ] && [ ! -f "${PKG_PATH}/pkg/${PATCH_PKG}.crl" ]
        then
            LOG "ERROR" "Check ${COPY_PATH}/${PATCH_PKG}.crl is not exist."
        fi
    fi
    LOG "INFO" "Copy package from path done."
    return 0
}

##########################################
# 解压OSPatch Tool
##########################################
function unzip_init_install()
{
    LOG "INFO" "Unzip ospatch tool package."
    # 初始OSTool目录
    if [ -d "${PKG_PATH}/${TOOL_PKG_DIR}" ]
    then
        rm -rf "${PKG_PATH}/${TOOL_PKG_DIR}"
    fi
    # 解压OSTool包
    unzip "${PKG_PATH}/pkg/${TOOL_PKG}" -d "${PKG_PATH}/${TOOL_PKG_DIR}" &>>${LOG_FILE}
    if [ $? -eq 0 ]
    then
        find "${PKG_PATH}/${TOOL_PKG_DIR}" -name "*.sh" | xargs chmod 550 &>>${LOG_FILE} || { LOG "ERROR" "Failed to unzip the ospatch tool package.";return 1; }
        find "${PKG_PATH}/${TOOL_PKG_DIR}" -name "*.pyc" | xargs chmod 550 &>>${LOG_FILE} || { LOG "ERROR" "Failed to unzip the ospatch tool package.";return 1; }
        find "${PKG_PATH}/${TOOL_PKG_DIR}" -name "*.tar" | xargs chmod 640 &>>${LOG_FILE} || { LOG "ERROR" "Failed to unzip the ospatch tool package.";return 1; }
        find "${PKG_PATH}/${TOOL_PKG_DIR}" -name "*.json" | xargs chmod 640 &>>${LOG_FILE} || { LOG "ERROR" "Failed to unzip the ospatch tool package.";return 1; }
        LOG "INFO" "Unzip the ospatch tool package is successfully."
        return 0
    fi
    LOG "ERROR" "Failed to unzip the ospatch tool package."
    return 1
}

##########################################
# 检查是否支持升级OS补丁
##########################################
function pre_check_is_suppor_upgrade()
{
    local not_suppor_upgrade="The operating system is of the latest version, No operation is required."
    if [ -f ${LATEST_OS_VERSION} ]
    then
        echo "${POST_TASK}" | grep "|${ACTION}|" &>/dev/null
        if [ $? -eq 0 ]
        then
            # 清理临时[文件/目录]
            rm -rf ${PKG_PATH} &>>"${LOG_FILE}"
            rm -rf ${LATEST_OS_VERSION} &>>"${LOG_FILE}"
        fi
        LOG "WARN" "${not_suppor_upgrade}"
        fresh_es_progress "${TASK_ID}" "90"
        TASK_LOG "WARN" "${not_suppor_upgrade}"
        return 0
    fi
    return 1
}

##########################################
# 后置步骤: 检查服务启动完成
##########################################
function post_check_start_service()
{
    echo "${POST_TASK}" | grep "|${ACTION}|" &>/dev/null
    if [ $? -eq 0 ]
    then
        ps -ef | grep -v "grep" | grep "post_check_node_status.sh" &>/dev/null
        if [ $? -ne 0 ]
        then
            bash ${SCRIPT_PATH}/post_check_node_status.sh start_service_task_id &>/dev/null &
        fi
        local count=0
        local max_progress=60
        local start_time=$(date "+%s")
        local timeout=$(expr ${start_time} + ${TIMEOUT})
        while true
        do
            # 0.刷新进度方案
            sleep 5
            local count=$(expr ${count} + 1)
            if [ ${count} -ge 6 ]
            then
                # 初始count
                local count=0
                # 更新进度
                task_update_progress ${max_progress}
            fi
            # 1.检查是否超时
            local current_time=$(date "+%s")
            if [ ${current_time} -ge ${timeout} ]
            then
                cat ${ES_TASH_MGR}/start_service_task_id/task.log >> ${TASK_PATH}/task.log
                LOG "ERROR" "Failed to start the service due to timeout."
                update_progress "${max_progress}"
                TASK_LOG "ERROR" "Failed to start the service due to timeout."
                return 1
            fi
            # 2.检查标识文件是否已生成
            ls ${ES_TASH_MGR}/start_service_task_id/task.status &>/dev/null
            [ $? -ne 0 ] && continue
            ls ${ES_TASH_MGR}/start_service_task_id/task.progress &>/dev/null
            [ $? -ne 0 ] && continue
        
            # 3.检查启动服务任务是否失败
            grep -i "Status=FAIL" "${ES_TASH_MGR}/start_service_task_id/task.status" &>/dev/null
            if [ $? -eq 0 ]
            then
                cat ${ES_TASH_MGR}/start_service_task_id/task.log >> ${TASK_PATH}/task.log
                LOG "ERROR" "Failed to start the product service."
                fresh_es_progress "${TASK_ID}" "${max_progress}"
                TASK_LOG "ERROR" "Failed to start the product service."
                return 1
            fi
            
            # 4.检查启动服务进度
            grep -i "Status=SUCCESS" "${ES_TASH_MGR}/start_service_task_id/task.status" &>/dev/null
            local status_res=$?
            grep -i "Progress=100" "${ES_TASH_MGR}/start_service_task_id/task.progress" &>/dev/null
            local progress_res=$?
            if [ ${status_res} -ne 0 -o ${progress_res} -ne 0 ]
            then
                TASK_LOG "INFO" "Starting the service, Please wait..."
                continue
            fi
            rm -rf ${ES_TASH_MGR}/start_service_task_id/task.*
            LOG "INFO" "All Services Was Started, Startup service done."
            fresh_es_progress "${TASK_ID}" "${max_progress}"
            TASK_LOG "INFO" "All Services Was Started, Startup service done."
            return 0
        done
    fi
    return 0
}

##########################################
# 更新任务进度
##########################################
function task_update_progress()
{
    # 获取最新进度
    local max_progress=${1}
    local current_progress=$(grep -i "Progress=" ${TASK_PATH}/task.progress | awk -F'=' '{print $2}')
    if [ ${current_progress} -le ${max_progress} ]
    then
        # 计算下阶段进度
        local new_progress=$(expr ${current_progress} + 2)
        if [ ${new_progress} -ge ${max_progress} ]
        then
            local new_progress=${max_progress}
        fi
        # 打印进度
        LOG "INFO" "Operating system patch task progress........${new_progress}%"
    fi
}

##########################################
# 其他任务: 启动任务监控, 粗略更新进度
##########################################
function other_task_record_progress()
{
    local exec_script=${1}
    local count=0
    local max_progress=80
    local start_time=$(date "+%s")
    local timeout=$(expr ${start_time} + ${TIMEOUT})
    while true
    do
        # 1.刷新进度方案
        sleep 5
        local count=$(expr ${count} + 1)
        if [ ${count} -ge 6 ]
        then
            # 初始count
            local count=0
            # 更新进度
            task_update_progress ${max_progress}
        fi
        # 2.检查是否超时
        local current_time=$(date "+%s")
        if [ ${current_time} -ge ${timeout} ]
        then
            ps -ef | grep "${exec_script}" | grep -v "grep" | awk '{print $2}' | xargs kill -9 &>/dev/null
            LOG "ERROR" "Operating system patch upgrade time out, Action is [${ACTION}]."
            break
        fi
        
        # 3.检查任务进程
        ps -ef | grep "${exec_script}" | grep -v "grep" &>/dev/null || break
    done
}

##########################################
# 其他任务: 结束后处理
##########################################
function other_task_post_handling()
{
    local exec_result_code=${1}
    # 前置步骤: code为9的特殊情况
    echo "${PRE_TASK}" | grep "|${ACTION}|" &>/dev/null
    if [ $? -eq 0 ] && [ ${exec_result_code} -eq 9 ]
    then
        touch ${LATEST_OS_VERSION}
        pre_check_is_suppor_upgrade
        return 0
    fi
    # 任务执行结束
    if [ ${exec_result_code} -eq 0 ]
    then
        echo "${POST_TASK}" | grep "|${ACTION}|" &>/dev/null
        if [ $? -eq 0 ]
        then
            # 清理临时[文件/目录]
            rm -rf ${PKG_PATH} &>>"${LOG_FILE}"
            rm -rf ${LATEST_OS_VERSION} &>>"${LOG_FILE}"
        fi
        LOG "INFO" "The task is successfully executed."
        return 0
    fi
    LOG "ERROR" "Failed to patch the operating system, error code:${exec_result_code}."
    return 1
}

##########################################
# 配置开机启动项脚本内容
##########################################
function upgrade_task_config_reboot()
{
    local check_script="${1}"
    LOG "INFO" "Config reboot script."
    local tool_reboot_ospatch_sh="${SCRIPT_PATH}/tool_reboot_ospatch_tmp.sh"
    # 生成可运行脚本
    tool_reboot_ospatch_inc="${SCRIPT_PATH}/tool_reboot_ospatch.sh"
    if [ ! -f ${tool_reboot_ospatch_inc} ]
    then
        LOG "ERROR" "The tool_reboot_ospatch.sh file is not found."
        return 1
    fi
    cp -f ${tool_reboot_ospatch_inc} ${tool_reboot_ospatch_sh}
    # 设置常量
    sed -i "s/^ACTION={{0}}/ACTION=${ACTION}/g" ${tool_reboot_ospatch_sh}
    sed -i "s/^TIMEOUT={{1}}/TIMEOUT=${TIMEOUT}/g" ${tool_reboot_ospatch_sh}
    sed -i "s/^TASK_ID={{2}}/TASK_ID=${TASK_ID}/g" ${tool_reboot_ospatch_sh}
    sed -i "s/^TOOL_PKG_DIR={{3}}/TOOL_PKG_DIR=${TOOL_PKG_DIR}/g" ${tool_reboot_ospatch_sh}
    sed -i "s/^CHECK_SCRIPT={{4}}/CHECK_SCRIPT=${check_script}/g" ${tool_reboot_ospatch_sh}
    chmod 550 ${tool_reboot_ospatch_sh}
    chown ossadm:ossgroup ${tool_reboot_ospatch_sh}
    return 0
}

##########################################
# 升级任务: 启动后处理
##########################################
function upgrade_task_post_handling()
{
    # 检查启动任务结果
    local start_result_code=${1}
    if [ ${start_result_code} -ne 0 ]
    then
        LOG "ERROR" "Failed to start the OS patch upgrade task, error code:${check_result}."
        return 1
    fi
    # 检查执行任务结果
    local check_result_script=${2}
    local count=0
    local wait_code=2
    local max_progress=80
    local start_time=$(date "+%s")
    local timeout=$(expr ${start_time} + ${TIMEOUT})
    local query_install_result="${PKG_PATH}/${TOOL_PKG_DIR}/${check_result_script}"
    while true
    do
        # 0.刷新进度方案
        sleep 15
        local count=$(expr ${count} + 1)
        if [ ${count} -ge 2 ]
        then
            # 初始count
            local count=0
            # 更新进度
            task_update_progress ${max_progress}
        fi
        # 1.检查是否超时
        local current_time=$(date "+%s")
        if [ ${current_time} -ge ${timeout} ]
        then
            LOG "ERROR" "Operating system patch upgrade time out, end code:${wait_code}"
            return 1
        fi
        # 2.检查升级结果
        bash ${query_install_result} >>${TASK_PATH}/task.log
        local check_result=$?
        local flag_log="/opt/patch_manager/NCE_OSPatch/"
        # code 0:升级完成
        if [ ${check_result} -eq 0 -a -f "${flag_log}/ospatch_result_success" ]
        then
            rm -f "${flag_log}/ospatch_result_success" &>>${LOG_FILE}
            LOG "INFO" "The OS patch is successfully upgraded."
            return 0
        else
            # 执行非0,等待5秒
            sleep 5
        fi
        # code 1:升级失败
        if [ ${check_result} -eq 1 -a -f "${flag_log}/ospatch_result_failure" ]
        then
            rm -f "${flag_log}/ospatch_result_failure" &>>${LOG_FILE}
            LOG "ERROR" "Failed to upgrade the OS patch, error code:${check_result}."
            return 1
        fi
        # code 2:等待状态
        local wait_code=${check_result}
        if [ ${check_result} -eq 2 ]
        then
            LOG "INFO" "The OS upgrade task is running. Please wait..."
            continue
        fi
        LOG "WARN" "The check result is abnormal, Warning code:${check_result}."
    done
}


##########################################
# 创建定时备份操作系统任务
##########################################
function upgrade_task_post_backupos()
{
    #传入参数为upgrade_refresh_patch_status,install，代表着是升级流程
    [ "${1}" != "upgrade_refresh_patch_status" -o "${2}" != "install" ] && return 0
    #检查是否存在备份任务
    upgrade_task_post_check_osback
    FLAG_OSBACK=$?
    # FLAG_OSBACK=0,代表不存在备份操作系统任务，1代表存在备份操作系统任务，2代表接口连不通
    if [ ${FLAG_OSBACK} == "${EXIST_OSBACK_TASK}" ]; then
        TASK_LOG "INFO" "The scheduled task for backing up the operating system already exists. Do not create the scheduled task for backing up the operating system."
        return 0
    elif [ ${FLAG_OSBACK} == "${LINK_INTERFACE_FAILED}" ]; then
        TASK_LOG "ERROR" "Failed to query whether a scheduled task for backing up the operating system exists.The possible cause is that the interface is disconnected."
        return "${FLAG_OSBACK}"
    fi
    LOG "INFO" "Begin to creating a scheduled task for backing up the operating system."
    #获取产品名称
    PRODUCTNAMELIST=$(cat /opt/oss/manager/etc/sysconf/nodelists.json | grep assignedToTenancy | awk -F':' '{print $2}' | sed 's/"//g' | sed 's/ //g' | sed 's/,//g' | sort |uniq)
    BACKUP_OS_PARMAETER="/opt/patch_manager/NCE_OSPatch/backup.json"
    for PRODUCTNAME in ${PRODUCTNAMELIST}
    do
        CURRENTTIME=$(date --date='1 days' "+%Y-%m-%d %H:%M:%S")
        #获取明天日期
        STARTDAY=$(echo "${CURRENTTIME}" | awk -F' ' '{print $1}')
        #获取当前时分秒
        STARTTIME=$(echo "${CURRENTTIME}" | awk -F' ' '{print $2}')
        [ -f ${BACKUP_OS_PARMAETER} ] && rm -f ${BACKUP_OS_PARMAETER}
        cat > ${BACKUP_OS_PARMAETER} << EOF
        {
            "type": "os",
            "nodelist": [],
            "productName": "${PRODUCTNAME}",
            "scheduleMode": "one-time",
            "enable": "yes",
            "startDate": "${STARTDAY}",
            "startTime": "${STARTTIME}",
            "endDate": "",
            "endTime": "",
            "interval": "",
            "backupReason": {
                "zh_CN": "升级操作系统导致系统版本变更，系统自动创建",
                "en_US": "remark"
            }
        }
EOF
        bash /opt/oss/manager/tools/backuprestore/createscheduletask.sh -input "${BACKUP_OS_PARMAETER}" &>>${LOG_FILE}
        if [ $? -ne 0 ];then
            TASK_LOG "ERROR" "Failed to create a scheduled task for backing up the operating system."
            return 1
        fi
    done
    TASK_LOG "INFO" "Succeeded in creating a scheduled task for backing up the operating system."
    [ -f ${BACKUP_OS_PARMAETER} ] && rm -f ${BACKUP_OS_PARMAETER}
    return 0
}

##########################################
# 检查是否已经存在定时备份操作系统任务
##########################################
function upgrade_task_post_check_osback(){
    FLAG_OSBACK=$(. /opt/oss/manager/bin/engr_profile.sh;python -c "from util import httpclient
import sys
import json
# flag_osback=0,代表不存在备份操作系统任务，1代表存在备份操作系统任务，2代表接口连不通
flag_osback=${NO_EXIST_OSBACK_TASK}
# 最多尝试30秒
retry_count=10
for _ in range(retry_count):
    status, response=httpclient.IRHttpClient().get('/rest/plat/brmgr/v1/ui/schedule/serviceschedule')
    if status == 200:
        break
    time.sleep(3)
if status != 200:
    flag_osback=${LINK_INTERFACE_FAILED}
    print(flag_osback)
    sys.exit(0)
if response != '':
    response_json=json.loads(response)
    backup_date_list=response_json.get('entity').get('data')
    for backup_date in backup_date_list:
        if backup_date.get('taskType') == 'OS':
            flag_osback=${EXIST_OSBACK_TASK}
            break
print(flag_osback)
sys.exit(0)")
    return ${FLAG_OSBACK}
}

##########################################
# 启动执行OS任务
##########################################
function exec_ospatch_task()
{
    LOG "INFO" "Start the task of executing the OS patch."
    local exec_script="${1}.sh"
    if [ -d "${PKG_PATH}/${TOOL_PKG_DIR}" ]
    then
        # 检查是否支持OS补丁
        echo "${PRE_TASK}" | grep "|${ACTION}|" &>/dev/null
        if [ $? -eq 0 -a -f ${LATEST_OS_VERSION} ]
        then
            rm -rf ${LATEST_OS_VERSION} &>>${LOG_FILE} || return 1
        fi
        pre_check_is_suppor_upgrade && return 0

        # 切换工作目录,开始执行任务
        cd "${PKG_PATH}/${TOOL_PKG_DIR}" &>>${LOG_FILE} || return 1
        ls -l ${exec_script} &>>${LOG_FILE} || return 1
        
        # 非升级OS补丁步骤
        echo "${EXEC_TASK}" | grep "|${ACTION}|" &>/dev/null
        if [ $? -ne 0 ]
        then
            # 后置步骤:检查服务是否已启动
            post_check_start_service || return 1

            # 升级后置步骤：创建定时备份操作系统任务
            upgrade_task_post_backupos "${ACTION}" "${2}" || return 1
            
            # 线程: 启动更新进度任务
            other_task_record_progress ${exec_script} &
            
            # 开始执行
            LOG "INFO" "Exec cmd: bash ${exec_script} ${2}"
            bash ${exec_script} "${2}" &>>${TASK_PATH}/task.log
            
            # 运行结束
            other_task_post_handling "$?"
            return $?
        fi
        # 升级OS补丁步骤
        echo "${EXEC_TASK}" | grep "|${ACTION}|" &>/dev/null
        if [ $? -eq 0 ]
        then
            local check_script="${2}.sh"
            # 配置开机启动项
            upgrade_task_config_reboot "${check_script}" || return 1

            # 执行补丁任务
            LOG "INFO" "Exec cmd: bash ${exec_script}"
            bash ${exec_script} &>>${TASK_PATH}/task.log
            
            # 检查执行结果
            upgrade_task_post_handling "$?" "${check_script}"
            local upgrade_task_code=$?
            return ${upgrade_task_code}
        fi
    fi
    # 处理后置步骤成功后重入问题
    echo "${POST_TASK}" | grep "|${ACTION}|" &>/dev/null && return 0
    LOG "ERROR" "No Executable Task"
    return 1
}

##########################################
# 输出任务结果
##########################################
function do_return()
{
    local result_code=$1
    if [ ${result_code} -eq 0 ]
    then
        LOG "INFO" "The OS task is executed successfully, Action is [${ACTION}]."
        TASK_LOG "INFO" "The OS task is executed successfully, Action is [${ACTION}]."
        fresh_es_progress "${TASK_ID}" "100"
        fresh_es_status "${TASK_ID}" "SUCCESS"
    else
        LOG "ERROR" "Failed to execute the OS task, Action is [${ACTION}]."
        TASK_LOG "ERROR" "Failed to execute the OS task, This log: ${LOG_FILE}"
        fresh_es_progress "${TASK_ID}" "100"
        fresh_es_status "${TASK_ID}" "FAIL"
    fi
    return ${result_code}
}

#################################################
# 主入口: 打补丁预处理失败, 直接退出程序, 不重启#
#################################################
function main()
{
    source $(dirname ${SCRIPT_PATH})/common.sh
    check_ossadm_user || return 1

    LOG "INFO" "Start to init ${LOG_FILE}."

    check_input $@ || return 1

    # 初始化任务文件夹
    init_taskmgr "${TASK_ID}"

    TASK_LOG "INFO" "Start to execute the OS task. Action is [${ACTION}]."
    
    # 根据入参执行各自方法
    case "${ACTION}" in
    "${PRE_UPGRADE}")
        # 升级OS补丁前置
        copy_pkg_workpath || do_return $? || return 1
        unzip_init_install || do_return $? || return 1
        # 0: 成功, 非0: 失败, 9: 不支持
        exec_ospatch_task "pre_install_ospatch"
        do_return $?
        return $?
    ;;
    "${PRE_ROLLBACK}")
        # 回滚OS补丁前置
        copy_pkg_workpath || do_return $? || return 1
        unzip_init_install || do_return $? || return 1
        # 0: 成功, 非0: 失败, 9: 不支持
        if [ ${IS_SUSE} -eq 0 ]
        then
            exec_ospatch_task "pre_rollback_suse"
        else
            exec_ospatch_task "pre_rollback_euler"
        fi
        do_return $?
        return $?
    ;;
    "${EXEC_UPGRADE}")
        # NCE节点升级OS补丁
        # 启动任务: install_ospatch; 0: 成功, 非0: 失败
        # 检查任务: query_install_result; 0: 成功, 2: 等待, Other: 失败
        exec_ospatch_task "install_ospatch" "query_install_result"
        do_return $?
        return $?
    ;;
    "${EXEC_ROLLBACK}")
        # NCE节点回滚OS补丁
        # 启动回滚任务: rollback_euler; 0: 成功, 非0: 失败
        # 检查回滚任务: query_rollback_result; 0: 成功, 2: 等待, Other: 失败
        if [ ${IS_SUSE} -eq 0 ]
        then
            exec_ospatch_task "rollback_suse" "query_rollback_result"
        else
            exec_ospatch_task "rollback_euler" "query_rollback_result"
        fi
        do_return $?
        return $?
    ;;
    "${EXEC_ARB_UPGRADE}")
        # 仲裁三方节点升级OS补丁
        # 启动任务: install_ospatch; 0: 成功, 非0: 失败
        # 检查任务: query_install_result; 0: 成功, 2: 等待, Other: 失败
        exec_ospatch_task "install_ospatch" "query_install_result"
        do_return $?
        return $?
    ;;
    "${EXEC_ARB_ROLLBACK}")
        # 仲裁三方节点回滚OS补丁
        # 启动回滚任务: rollback_euler; 0: 成功, 非0: 失败
        # 检查回滚任务: query_rollback_result; 0: 成功, 2: 等待, Other: 失败
        if [ ${IS_SUSE} -eq 0 ]
        then
            exec_ospatch_task "rollback_suse" "query_rollback_result"
        else
            exec_ospatch_task "rollback_euler" "query_rollback_result"
        fi
        do_return $?
        return $?
    ;;
    "${POST_UPGRADE}")
        # 升级NCE节点后置
        exec_ospatch_task "refresh_patch_status" "install"
        do_return $?
        return $?
    ;;
    "${POST_ROLLBACK}")
        # 回滚NCE节点后置
        exec_ospatch_task "refresh_patch_status" "rollback"
        do_return $?
        return $?
    ;;
    esac
}

main $@
exit $?
