#!/bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
# Description: 用以修复R21C10SPC200 GW节点ssh不停重启的问题
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
# INPUT文件路径
INPUT_FILE="${2}"
# 升级源版本
SRC_VERSION=""
# GW节点信息
GW_LIST=""
# ssh连接命令
SSH_CMD="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 "

###################################################################################
# Description： 查询升级源版本
# @Return: 0 查询失败 1 查询成功
###################################################################################
function query_src_version() {
    LOG "INFO" "Input file path: ${INPUT_FILE}"
    SRC_VERSION=$(cat "${INPUT_FILE}" 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['src_version'])" 2>/dev/null)
    if [ -z "${SRC_VERSION}" ]; then
        LOG "ERROR" "The Product source version is null."
        return 1
    fi
    return 0
}

###################################################################################
# Description： 校验当前升级源版本是否需要进行修复
# @Return: 0 不需要修复 1 需要修复
###################################################################################
function check_need_fix() {
    # 仅R21C10SPC200版本需要此处理
    echo "${SRC_VERSION}" | grep -q "^V100R021C10SPC200"
    if [ $? -eq 0 ]; then
        return 1
    fi
    LOG "INFO" "Skip the current upgrade path. src: ${SRC_VERSION}"
    return 0
}

###################################################################################
# Description： 查询是否存在待修复的GW节点，存在的情况下，返回需要清理的浮动IP监听列表
# @GW_LIST: manager_ip1=待清理IP1,待清理IP2;manager_ip2=待清理IP1,待清理IP2
# @Return: 0不需要修复 1 需要修复
###################################################################################
function query_fix_node() {
    LOG "INFO" "Start to query GW node information."
    local node_list="/opt/oss/manager/etc/sysconf/nodelists.json"
    GW_LIST=$(python 2>${LOG_FILE}<< PEND
import json
path = "${node_list}"
check_ip = {"LB-DR-L2-STANDBY-VIP", "LB-VIP", "LB-DR-L2-ACTIVE-VIP"}
def get_gw_info(gw_node):
    lvs = [net_info.get("IP") for net_info in gw_node.get("IPAddresses", []) if list(check_ip & set(net_info.get("usage", [])))]
    if not lvs:
        return ""
    manger_ip = [net_info.get("IP") for net_info in gw_node.get("IPAddresses", []) if "maintenance" in net_info.get("usage", [])]
    return f"{manger_ip[0]}={','.join(lvs)}"
try:
    with open(path, mode='r') as node_file:
        data = json.load(node_file)
    host_list = data.get('nodeList', {})
    gw_list = [one for one in host_list.values() if one.get("hostname", "").startswith("GW")]
    gw_lvs_info = []
    for one_gw in gw_list:
        tmp_gw_info = get_gw_info(one_gw)
        if tmp_gw_info:
            gw_lvs_info.append(tmp_gw_info)
    print(";".join(gw_lvs_info))
except Exception as _:
    print("")
PEND
)
    LOG "INFO" "GW_LIST:${GW_LIST}"
    if [ -n "${GW_LIST}" ]; then
       return 1
    fi
    LOG "INFO" "No GW node exists. No action is required."
    return 0
}

###################################################################################
# Description： 清理单个GW节点上的浮动IP，降低升级失败概率，如果报错，不阻塞流程
# @$1: manager_ip1=待清理IP1,待清理IP2
###################################################################################
function fix_one_gw() {
    local fix_config="$1"
    local node_ip=$(echo "${fix_config}" | awk -F= '{print $1}')
    local delete_ip=$(echo "${fix_config}" | awk -F= '{print $2}')
    LOG "INFO" "Delete the ${delete_ip} on the ${node_ip} node."
    for one_delete_ip in $(echo "${delete_ip}" | sed 's/,/ /g'); do
          ${SSH_CMD} "${node_ip}" "sed -i '/^${one_delete_ip}$/d' '/opt/oss/manager/adapter/common/resmgr/floatingip.list'" &>>${LOG_FILE}
    done
}

###################################################################################
# Description： 清理所有GW节点上的浮动IP列表信息，降低升级失败概率，如果报错，不阻塞流程
###################################################################################
function fix_gw_node() {
    for one_gw in $(echo "${GW_LIST}" | sed 's/;/ /g' ); do
        fix_one_gw "${one_gw}"
    done
    LOG "INFO" "The GW node is restored."
}


function main() {
    . "${SCRIPT_PATH}"/common.sh
    . /opt/oss/manager/bin/engr_profile.sh
    LOG "INFO" "Start to init ${LOG_FILE}."
    query_src_version || return 1
    check_need_fix && return 0
    query_fix_node && return 0
    fix_gw_node || return 1
}

main "$@"
exit $?