#!/bin/bash
##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
UPGRADE_BACKUP_PATH="/opt/upgrade/backup"
OSS_USER=$(id -nu 3001)

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "${OSS_USER}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 日志记录进/tmp/${SELF_FILE}.log
##################################################
function LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${BASH_PID}] | ${Level} $2" | tee -a ${LOG_FILE}
}

# 入参校验
function check_param()
{
    local check_tmp=$1
    [ -z "${check_tmp}" ] && return 1
    echo "${check_tmp}" | fgrep -q "../"
    if [ $? -eq 0 ]
    then
        LOG "ERROR" "The input param \"${check_tmp}\" is invalid."
        return 1
    fi
    if [ ! -d "${check_tmp}" ]
    then
        LOG "ERROR" "The input param \"${check_tmp}\" is invalid."
        return 1
    fi
}

# 获取输入参数
function init_params()
{
    local num=$#
    if [ $((${num}%2)) -eq 1 ]
    then
        num=$((${num}/2+1))
    else
        num=$((${num}/2))
    fi
    local count=1
    for((i=1;i<=${num};i++))
    do
        [[ ${1#-} == "input" ]] && { INPUTPATH=$2;shift 2;check_param "${INPUTPATH}" || return 1; continue; }
    done
}

function check_input()
{
    init_params "$@"
    if [ $? -ne 0 ]
    then
        echo "Example: bash ${SELF_FILE} -input /opt/upgrade/easysuite_upgrade/workpath/V100R019C00SPC602-V100R020C00/workpath-NCE/"
        return 1
    fi
}

# 获取产品数据
function get_product_data()
{
    PRODUCTNAME=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['productname'])" 2>/dev/null)
    if [ -z "${PRODUCTNAME}" ]
    then
        LOG "ERROR" "Productname is null."
        return 1
    fi
    SRC_VERSION=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['src_version'])" 2>/dev/null)
    if [ -z "${SRC_VERSION}" ]
    then
        LOG "ERROR" "The Product source version is null."
        return 1
    fi
    DES_VERSION=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['des_version'])" 2>/dev/null)
    if [ -z "${DES_VERSION}" ]
    then
        LOG "ERROR" "The Product destination version is null."
        return 1
    fi
}

# 保存连接文件到目录下
function copy_db_connect_file()
{
    # 升级前老版本配置文件备份路径
    local product_backup_path="${UPGRADE_BACKUP_PATH}"/pre_upgrade/"${SRC_VERSION}-${DES_VERSION}"/"${PRODUCTNAME}"/

    # 创建目录
    mkdir -p "${INPUTPATH}"/plandata "${product_backup_path}" &>/dev/null

    # 保存配置文件到数据库迁移目录
    cp -fp "${backup_path}"/containerlist_"${PRODUCTNAME}".json "${INPUTPATH}"/plandata/containerlist_old.json || return 1

    # 保存配置文件到备份目录子域获取升级前配置信息
    cp -fp "${backup_path}"/containerlist_"${PRODUCTNAME}".json "${product_backup_path}"/containerlist.json || return 1

    # 保存配置文件
    chmod 600 "${backup_path}"/containerlist_${PRODUCTNAME}.json "${INPUTPATH}"/plandata/containerlist_old.json "${product_backup_path}"/containerlist.json || return 1
}


# 导出数据库配置连接文件
function export_containerlist()
{
    LOG "INFO" "Start to export DB config."

    local backup_path="/opt/upgrade/backup/migrate/old"
    if [ ! -d ${backup_path} ]
    then
        mkdir -p ${backup_path} >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            LOG "ERROR" "Failed to mkdir ${backup_path}."
            return 1
        fi
    fi

    python >>${LOG_FILE} 2>&1 << PEND
import easyhttputil
exportParam = {}
userList = ['adminUser', 'dbUser', 'readUser', 'rplUser']
exportParam['userList'] = userList
exportParam['tenant'] = '${PRODUCTNAME}'
backup_file='/opt/upgrade/backup/migrate/old/containerlist_${PRODUCTNAME}.json'
if hasattr(easyhttputil, 'httppostWithRetry'):
    http_post_func = easyhttputil.httppostWithRetry
else:
    http_post_func = easyhttputil.http_post_with_retry
response = http_post_func('/rest/plat/deploy-proxy/v1/containerlist/action?action-id=export-containerlist',exportParam, retry = 3, sleep = 5)
f = open(backup_file,'wb')
f.write(response)
f.close()
PEND
    if [ $? -eq 0 ]
    then
        LOG "INFO" "Finished to export DB config."
    else
        LOG "ERROR" "Failed to export DB config."
        return 1
    fi

    # 保存数据库连接文件
    copy_db_connect_file &>>${LOG_FILE}
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "Failed to save DB config."
        return 1
    fi
    return 0
}


function main()
{
    check_user || return 1
    # 创建日志目录
    [ ! -d "${LOG_PATH}" ] && mkdir -p ${LOG_PATH}
    . /opt/oss/manager/bin/engr_profile.sh
    check_input "$@" || return 1
    get_product_data || return 1
    export_containerlist || return 1
}

main "$@"
exit $?
