#!/bin/bash
#==============================================================
# Copyright  Huawei Technologies Co., Ltd. 1998-2012. All rights reserved.
# File Name             : setLocalIP.sh
# Version:Data          : 
# Description           : modifty /etc/sysconfig/network/ifcfg-X
# Others                : 
# History               : 
#==============================================================


MODULE_NAME="`basename $0 | sed 's/.sh//g'`"
RELATIVE_PATH="`dirname $0`"

#get absolute path
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`
#import common variable..
. ../common/error_code.sh
. ../common/common.sh
cd - >/dev/null 2>&1


 # Name        : fn_clean_tmp_file
# Description : Clear temporary files and directories in the installation process
function fn_remove_tmp_file()
{
    [ -d "${TMP_CONFIG_PATH}" ] && rm -rf "${TMP_CONFIG_PATH}"/tmp*
    return 0
}


# Name        : fn_write_operation_log
# Description : write operation log
# Parameters  : $1 return value
function fn_write_operation_log()
{
    local RETCODE=$1
    who am i | egrep -w '([0-9]{1,3}\.){3}([0-9]{1,3})' &> /dev/null 
    if [ $? -eq 0 ]
    then
        OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`
        OPERIP="${OPERIP//:*/}"
    else
        OPERIP=''
    fi
    
    if [ "${RETCODE}" = 0 ]
    then
        echo "`date` `hostname` `whoami` setLocalIP.sh;Successful;Set IP Successfully." >> /var/log/localmessages
        echo "Set IP Successfully. "
    else
        echo "`date` `hostname` `whoami` setLocalIP.sh;Failed;Set IP Failed." >> /var/log/localmessages
        echo "Set IP Failed."
    fi    
    chmod 640 /var/log/localmessages

    return 0
}


# Name        : fn_check_or_modify_ifcfg
# Description : check ifcfg-eth-X
# Parameters  : $1 ifcfg-eth-X
function fn_check_or_modify_ifcfg()
{
    local ifcfg_file=$1

    grep -w 'IPADDR' "${ifcfg_file}" > /dev/null 2>&1 || echo "IPADDR=''" >> "${ifcfg_file}" 
    grep -w 'NETMASK' "${ifcfg_file}" > /dev/null 2>&1 || echo "NETMASK=''" >> "${ifcfg_file}" 
    grep -w 'NETWORK' "${ifcfg_file}" > /dev/null 2>&1 || echo "NETWORK=''" >> "${ifcfg_file}" 
    grep -w 'BOOTPROTO' "${ifcfg_file}" > /dev/null 2>&1 || echo "BOOTPROTO='static'" >> "${ifcfg_file}" 
    grep -w 'BROADCAST' "${ifcfg_file}" > /dev/null 2>&1 || echo "BROADCAST=''" >> "${ifcfg_file}" 

    return 0
}


# Name        : fn_get_config
# Description : get the config from osmu
# Parameters  : $1 properties file
function fn_get_config()
{
    local TMP_CONFIG_FILE=$1
    local BOND_NAME
    local NIC_DEVICE
    local NIC_DEVICE_BACKUP
    local IPADDR
    local NETMASK
    local IP_HOSTNAME
    cat $TMP_CONFIG_FILE
    BOND_NAME=`fn_get_value "${TMP_CONFIG_FILE}" "BOND_NIC_NAME"`
    NIC_DEVICE=`fn_get_value "${TMP_CONFIG_FILE}" "NIC_DEVICE"`
    NIC_DEVICE_BACKUP=`fn_get_value "${TMP_CONFIG_FILE}" "NIC_DEVICE_BACKUP"`
    IPADDR=`fn_get_value "${TMP_CONFIG_FILE}" "IPADDR"`
    NETMASK=`fn_get_value "${TMP_CONFIG_FILE}" "NETMASK"`
    IP_HOSTNAME=`fn_get_value "${TMP_CONFIG_FILE}" "IP_HOSTNAME"`

    
    if [ -z "${NIC_DEVICE}" ] ||  [ -z "${IPADDR}" ] || [ -z "${NETMASK}" ]
    then
        PRINT_LOG "FATAL" "The param NIC_DEVICE<${NIC_DEVICE}>,IPADDR<${IPADDR}>,NETMASK<${NETMASK}>could not found!"
        return "${ERROR_PARAM}"
    fi
    
    fn_validate_ip "${IPADDR}" || return $?
    fn_validate_ip "${NETMASK}" || return $?
    
    if [ ! -z $BOND_NAME ]
    then
        eval $2="${BOND_NAME}"
        if [ ! -z $NIC_DEVICE_BACKUP ]
        then
            eval $4="${NIC_DEVICE_BACKUP}"
        else
            PRINT_LOG "FATAL" "The param NIC_DEVICE_BACKUP could not found! There must be two nic device to create a bond nic"
            return "${ERROR_PARAM}"
        fi
    fi

    eval $3="${NIC_DEVICE}"
    eval $5="${IPADDR}"
    eval $6="${NETMASK}"
    eval $7="${IP_HOSTNAME}"

    return 0
}


# Name        : fn_get_broadcast_and_network
# Description : get the broadcast and network
# Parameters  : $1 ip address
# Parameter   : $2 netmask
function fn_get_broadcast_and_network()
{
    local ip=(${1//[![:digit:]]/ })
    local mask=(${2//[![:digit:]]/ })

    for i in ${mask[*]}
    do
        j=7
        tag=1
        while [ $j -ge 0 ]
        do
          k=$((2**$j))
          if [ $(( $i & $k )) -eq $k ]; then
                if [ $tag -eq 1 ]; then
                    (( n += 1 ))
                else
                    echo -e "\n$2 is a bad netamsk with holes\n"
                    break
                fi
           else
               tag=0
           fi
           (( j -= 1 ))
          done
    done

    local a=
    local b=
    for i in 0 1 2 3
    do
    a=$a${a:+.}$((${ip[i]} & ${mask[i]}))
    b=$b${b:+.}$((${ip[i]} | (${mask[i]} ^ 255)))
    done

    eval $3="${a}"
    eval $4="${b}"

    return 0;
}

function fn_mask_to_prefixlen()
{
    local mask=$1
    OLD_IFS="$IFS" 
    IFS="." 
    arr=($mask) 
    IFS="$OLD_IFS" 
    
    local result=""
    for s in ${arr[@]} 
    do 
        num=$s
        a=`echo "obase=2;${num}" | bc`
        aa=`printf "%08d\n" $a`
        result=$result$aa
    done
    
    for ((i=0;i<=31;i++))
    do
        bb=${result:i:1}
        if [ "$bb" != "1" ]
        then
            echo $i
            break
        fi
    done
}

# Name        : fn_modify_ifcfg_file
# Description : modify ifcfg-eth-X
# Parameters  : $1 ifcfg-eth-X
# Parameter   : $2 network card mac address
# Parameter   : $3 pci bus id
function fn_modify_ifcfg_file()
{
    local ifcfg_file=$1
    local ipaddr=$2
    local netmask=$3
    local broadcast=$4
    local network=$5
    local card_name=$6
    local bus_pci=$7
    local prefixlen=`fn_mask_to_prefixlen $netmask`
    # 修改ifcfg-eth-X文件。
    sed -i -e "/^BOOTPROTO=/c BOOTPROTO='static'" \
           -e "/^IPADDR=/c IPADDR='${ipaddr}'" \
           -e "/^NETMASK=/c NETMASK='${netmask}'" \
           -e "/^BROADCAST=/c BROADCAST='${broadcast}'" \
           -e "/^STARTMODE=/c STARTMODE='auto'" \
           -e "/^NETWORK=/c NETWORK='${network}'" \
           -e "/^NAME=/c NAME='${card_name}'" \
           -e "/^PREFIXLEN=/c PREFIXLEN='${prefixlen}'" \
           -e "/^_nm_name=/c _nm_name='bus-pci-${bus_pci}'" "${ifcfg_file}" > "${TMP_CONTENT_FILE}" 2>&1
    if [ $? -ne 0 ]
    then
        PRINT_LOG  "FATAL" "Set ${ifcfg_file} configuration file ...failed"
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        return ${ERR_EXECSYS}
    fi
    
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
    PRINT_LOG  "INFO" "Set ${ifcfg_file} configuration file ...OK"
    
    return 0 
}

function fn_release_bond()
{
    local bondingPath="/proc/net/bonding/"
    local nic_dev=$1
    local nic_dev_back_up=$2
    if [ -d $bondingPath ]
    then
        local tmpNicDev
        local existBondNic=(`ls $bondingPath`)
        local ifcfgPath="/etc/sysconfig/network/ifcfg-"
        for tmpBondNic in ${existBondNic[@]}
        do
            if [ `cat "${bondingPath}${tmpBondNic}" | grep "^Slave Interface:" | awk -F": " '{print $2}' | egrep "^$nic_dev\$|^$nic_dev_back_up\$" | wc -l ` -eq 2 ]
            then
                rm "${ifcfgPath}${tmpBondNic}"
                rcnetwork stop ${tmpBondNic} > "${TMP_CONTENT_FILE}" 2>&1
                if [ $? -ne 0 ]
                then
                    PRINT_LOG "FATAL" "Failed to down ${tmpBondNic} for rename ${tmpNicDev}"
                    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
                    return 1
                fi
                continue
            fi
            tmpNicDev=`cat "${bondingPath}${tmpBondNic}" | grep "^Slave Interface:" | awk -F": " '{print $2}' | egrep "^$nic_dev\$|^$nic_dev_back_up\$" | tr "\n" " "`
            if [ -n "${tmpNicDev}" ]
            then
                PRINT_LOG "INFO" "Nic device ${tmpNicDev} has already been bond by ${tmpBondNic}. Start to release..."
                sed -i "s/${nic_dev}//g" "${ifcfgPath}${tmpBondNic}"
                sed -i "s/${nic_dev_back_up}//g" "${ifcfgPath}${tmpBondNic}"
                rcnetwork stop ${tmpBondNic} > "${TMP_CONTENT_FILE}" 2>&1
                if [ $? -ne 0 ]
                then
                    PRINT_LOG "FATAL" "Failed to down ${tmpBondNic} for release ${tmpNicDev}"
                    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
                    return 1
                fi
                PRINT_LOG "INFO" "Down ${tmpBondNic} for release ${tmpNicDev} successfully"
                PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            fi
        done
    fi

    return 0
}

function fn_bond_nic()
{
    local ETH_DEVICE1=$1
    local ETH_DEVICE2=$2
    local BOND_NAME=$3
    local BOND_IP=$4
    local BOND_NETMASK=$5
    local NETWORK
    local BROADCAST
    local NIC_STATUS1
    local NIC_STATUS2
    local INDEX
    local TOTAL
    local ifcfg_file="/etc/sysconfig/network/ifcfg-${BOND_NAME}"
    
    PRINT_LOG "INFO" "Bonding <${BOND_NAME}>..."

    PRINT_LOG "INFO" "Check ${BOND_NAME}..."
    ifconfig -a | grep -w "^${BOND_NAME}" >/dev/null 2>&1
    if [ $? -eq 0 ]
    then
        PRINT_LOG "INFO" " Bond <${BOND_NAME}> is already in use"
        PRINT_LOG "INFO" "Start to modify Bond <${BOND_NAME}>"
    fi
    PRINT_LOG "INFO" "Check ${BOND_NAME}...done"
    
    PRINT_LOG "INFO" "Exec <ifconfig -a>"
    ifconfig -a > "${TMP_CONTENT_FILE}" 2>&1
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
    
    PRINT_LOG "INFO" "Exec <ip link show>"
    ip link show > "${TMP_CONTENT_FILE}" 2>&1
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"

    #获取MAC地址
    ETH_MAC1=`ifconfig ${ETH_DEVICE1} | grep -w "HWaddr" | awk '{print $NF}'`
    ETH_MAC2=`ifconfig ${ETH_DEVICE2} | grep -w "HWaddr" | awk '{print $NF}'`  
    if [ -z "${ETH_MAC1}" ] || [ -z "${ETH_MAC2}" ]
    then
        PRINT_LOG "WARN" " ETH_MAC1<${ETH_MAC1}> or ETH_MAC2<${ETH_MAC2}> is unavailable!"
        return ${ERR_GETMAC};
    fi
    
    #解除已有绑定
    fn_release_bond "${ETH_DEVICE1}" "${ETH_DEVICE2}" || return 1
    
    #计算NETWORK BROADCAST
    if [ -n "${BOND_IP}" ] && [ -n "${BOND_NETMASK}" ]
    then
        PRINT_LOG "INFO" "Get configuration from properties..."
        fn_get_broadcast_and_network "${BOND_IP}" "${BOND_NETMASK}" NETWORK BROADCAST
        if [ $? != 0 ]
        then
            PRINT_LOG "WARN" "Exec<fn_get_broadcast_and_network \"${BOND_IP}\" \"${BOND_NETMASK}\" NETWORK BROADCAST> failure!"
            return ${ERROR_PARAM};
        fi
    fi
    PRINT_LOG "INFO" "BOND_IP<${BOND_IP} NETWORK<${NETWORK}> BROADCAST<${BROADCAST}> BOND_NETMASK<${BOND_NETMASK}>"

    #配置bond
    if [ ! -f "${ifcfg_file}" ]
    then
        cat << EOF > "${ifcfg_file}"
STARTMODE='onboot'
BONDING_MASTER='yes'
BONDING_MODULE_OPTS='mode=1 miimon=200'
device='${BOND_NAME}'
BONDING_SLAVE0='${ETH_DEVICE1}'
BONDING_SLAVE1='${ETH_DEVICE2}'
EOF
    fi

    local slave0="${ETH_DEVICE1}"
    local slave1="${ETH_DEVICE2}"

    # 如果/etc/sysconfig/network/ifcfg-${BOND_NAME}里面的BONDING_SLAVE值与用户给定的不一致,则更新
    sed -i "s/BONDING_SLAVE0=.*/BONDING_SLAVE0='${ETH_DEVICE1}'/ ; s/BONDING_SLAVE1=.*/BONDING_SLAVE1='${ETH_DEVICE2}'/" "${ifcfg_file}"

    grep "IPADDR_1=" "${ifcfg_file}" >/dev/null 2>&1
    if [ $? -ne 0 ]
    then
        if grep "IPADDR="  "${ifcfg_file}" >/dev/null 2>&1
        then
            sed -i "s/IPADDR=.*/IPADDR='${BOND_IP}'/g" "${ifcfg_file}"
            sed -i "s/NETMASK=.*/NETMASK='${BOND_NETMASK}'/g" "${ifcfg_file}"
            sed -i "s/NETWORK=.*/NETWORK='${NETWORK}'/g" "${ifcfg_file}"
            sed -i "s/BROADCAS=.*/BROADCAST='${BROADCAST}'/g" "${ifcfg_file}"
        else
            sed -i "4a IPADDR='${BOND_IP}'\nNETWORK='${NETWORK}'\nBROADCAST='${BROADCAST}'\nNETMASK='${BOND_NETMASK}'" "${ifcfg_file}"
        fi
    else
        if grep "IPADDR_0="  "${ifcfg_file}" >/dev/null 2>&1
        then
            sed -i "s/IPADDR_0=.*/IPADDR_0='${BOND_IP}'/g" "${ifcfg_file}"
            sed -i "s/NETMASK_0=.*/NETMASK_0='${BOND_NETMASK}'/g" "${ifcfg_file}"
            sed -i "s/NETWORK_0=.*/NETWORK_0='${NETWORK}'/g" "${ifcfg_file}"
            sed -i "s/BROADCAS_0=.*/BROADCAST_0='${BROADCAST}'/g" "${ifcfg_file}"
        else
            sed -i "4a IPADDR_0='${BOND_IP}'\nNETWORK_0='${NETWORK}'\nBROADCAST_0='${BROADCAST}'\nNETMASK_0='${BOND_NETMASK}'" "${ifcfg_file}"
        fi
    fi

    chmod 640 "${ifcfg_file}"
    PRINT_FILE_TO_LOG "${ifcfg_file}"
    
    #if both the NIC have a same mac address .. we consider they are correct ...
    #avoid to config the same mac address on both NIC
    if [ "${ETH_MAC1}" = "${ETH_MAC2}" ]
    then
        PRINT_LOG "WARN" " BOND_NIC1<${ETH_DEVICE1}>:ETH_MAC1<${ETH_MAC1}> and BOND_NIC2<${ETH_DEVICE2}>:ETH_MAC2<${ETH_MAC2}> is the same!"
    fi
    
    for ETH_DEVICE in ${ETH_DEVICE1} ${ETH_DEVICE2}
    do
        cat <<-\EOF > /etc/sysconfig/network/ifcfg-${ETH_DEVICE}
BOOTPROTO='static'
STARTMODE='onboot'
EOF
        chmod 640 /etc/sysconfig/network/ifcfg-${ETH_DEVICE}
        PRINT_FILE_TO_LOG "/etc/sysconfig/network/ifcfg-${ETH_DEVICE}"

        local routesPath="/etc/sysconfig/network/routes"
        if [ ! -f "${routesPath}" ]
        then
            touch "${routesPath}"
            chmod 640 "${routesPath}"
        fi
        grep ${ETH_DEVICE} ${routesPath}
        if [ $? -eq 0 ]
        then
            sed -i "/${ETH_DEVICE}/d" ${routesPath}
        fi
    done
    
    return 0;
}

function fn_main()
{
    local REF_BOND_NAME
    local REF_NIC_DEVICE
    local REF_NIC_DEVICE_BACKUP
    local REF_IPADDR
    local REF_NETMASK
    local REF_NETWORK
    local REF_BROADCAST
    local REF_DB_INTERFACES_ADDR
    
    local REF_OS_TYPE
    local network_card_name
    local NETWORK_CARD_NAME
    local HW_ADDR
    local BUS_PCI
    local IFCFG_FILE
    
    PRINT_LOG "INFO" "bash $0 $1"
    if [ $# -lt 1 ]
    then
        PRINT_LOG "FATAL" "Usage: $0 <config file>"
        PRINT_LOG "FATAL" "e.g:      $0 /opt/osmu/script/properties/sample.properties"
        fn_check_status "${ERROR_PARAM}"
    fi
    
    CONFIG_FILE=`echo "$1" | sed 's/\\\/\//g'`
    
    sed -i "s/[[:space:]]*$//g" "${CONFIG_FILE}"
    fn_check_config_file "${CONFIG_FILE}" || return $?

    fn_get_config "${TMP_CONFIG_FILE}" REF_BOND_NAME REF_NIC_DEVICE REF_NIC_DEVICE_BACKUP REF_IPADDR REF_NETMASK REF_IP_HOSTNAME

    if [ $? -ne 0 ]
    then
        return "${ERROR_PARAM}"
    fi

    if [ -n "$REF_BOND_NAME" ]
    then
        local old_ip=`ifconfig "${REF_BOND_NAME}" | grep "inet addr:" | awk -F"inet addr:" '{print $2}' | awk '{print $1}' | xargs`
        fn_bond_nic "$REF_NIC_DEVICE" "$REF_NIC_DEVICE_BACKUP" "$REF_BOND_NAME" "${REF_IPADDR}" "${REF_NETMASK}" || return ${ERR_EXECSYS};

        rcnetwork restart "${REF_NIC_DEVICE}" > "${TMP_CONTENT_FILE}" 2>&1
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE}\"> ...failure"
            PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            return ${ERR_EXECSYS}
        fi
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE}\"> ...OK"
        
        rcnetwork restart "${REF_NIC_DEVICE_BACKUP}" > "${TMP_CONTENT_FILE}" 2>&1        
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE_BACKUP}\"> ...failure"
            PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            return ${ERR_EXECSYS}
        fi
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE_BACKUP}\"> ...OK"

        rcnetwork restart "${REF_BOND_NAME}" > "${TMP_CONTENT_FILE}" 2>&1
        if [ $? -ne 0 ]
        then
            PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_BOND_NAME}\"> ...failure"
            return ${ERR_EXECSYS}
        fi
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_BOND_NAME}\"> ...OK"

    else
        local old_ip=`ifconfig "${REF_NIC_DEVICE}" | grep "inet addr:" | awk -F"inet addr:" '{print $2}' | awk '{print $1}' | xargs`
        
        ## 取得网卡总线号
        BUS_PCI="`ethtool -i ${REF_NIC_DEVICE} 2> /dev/null |grep 'bus-info' | cut -d ' ' -f 2`"
        ## 取得总线号对应网卡的说明 
        [ -z "${BUS_PCI}" ] || network_card_name=`lspci -v 2> /dev/null | awk -v 'RS=' '/'"${BUS_PCI#*:}"'/ {print $0}' | egrep -i "\<Subsystem: "`
        NETWORK_CARD_NAME=${network_card_name#*Subsystem:}
        ## 取得MAC地址
        HW_ADDR="`ifconfig ${REF_NIC_DEVICE} |awk -F 'ddr ' '/\<HWaddr / {print $2}' | tr [A-Z] [a-z] | xargs`"
        CFG_DIR="/etc/sysconfig/network"
        CFG_FILE_NAME="ifcfg-${REF_NIC_DEVICE}"
        IFCFG_FILE="${CFG_DIR}/${CFG_FILE_NAME}"
        IFCFG_BAK_DIR="/etc/sysconfig/network/OLD_IFCFG_BAK"
        
        [ -f "${IFCFG_BAK_DIR}" ] || mkdir -p "${IFCFG_BAK_DIR}"

        ## 计算IP及netmask对应的broadcast和network
        fn_get_broadcast_and_network "${REF_IPADDR}" "${REF_NETMASK}" REF_NETWORK REF_BROADCAST
        
        ## 检查网卡对应的配置文件是否存在
        if [ ! -f "${IFCFG_FILE}" ]
        then
            PRINT_LOG "WARN" "${IFCFG_FILE} does not exist"
            echo "BOOTPROTO='static'" >> "${IFCFG_FILE}"
            echo "DEVICE='${REF_NIC_DEVICE}'" >> "${IFCFG_FILE}"
            echo "IPADDR='${REF_IPADDR}'" >> "${IFCFG_FILE}"
            echo "NETMASK='${REF_NETMASK}'" >> "${IFCFG_FILE}"
            echo "STARTMODE='auto'" >> "${IFCFG_FILE}"
            echo "NETWORK='${REF_NETWORK}'" >> "${IFCFG_FILE}"
            echo "BROADCAST='${REF_BROADCAST}'" >> "${IFCFG_FILE}"
            PRINT_LOG "INFO" "Generate ${IFCFG_FILE} file success"
        else
            #备份一份原始配置文件
            [ -f "${IFCFG_BAK_DIR}/${CFG_FILE_NAME}.bak" ] && cp -a "${IFCFG_FILE}" "${IFCFG_BAK_DIR}/${CFG_FILE_NAME}.$$.bak"
            cp -a "${IFCFG_FILE}" "${IFCFG_BAK_DIR}/${CFG_FILE_NAME}.bak"
            
            #查询网卡的配置文件是否存在对应项目
            fn_check_or_modify_ifcfg "${IFCFG_FILE}"
            
            #修改配置文件
            fn_modify_ifcfg_file "${IFCFG_FILE}" "${REF_IPADDR}" "${REF_NETMASK}" "${REF_BROADCAST}" "${REF_NETWORK}" "${NETWORK_CARD_NAME}" "${BUS_PCI}"
            if [ $? -ne 0 ]
            then
                mv -f "${IFCFG_BAK_DIR}/${CFG_FILE_NAME}.bak" "${IFCFG_FILE}"
                return ${ERR_EXECSYS}
            fi
        fi

        #修改完成后重新启动网卡使配置文件生效。
        rcnetwork restart "${REF_NIC_DEVICE}" > "${TMP_CONTENT_FILE}" 2>&1
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE}\"> ...failure"
            PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
            return ${ERR_EXECSYS}
        fi
        PRINT_LOG "WARN" "Excute<rcnetwork restart \"${REF_NIC_DEVICE}\"> ...OK"
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
    fi
    
    local hosts_file="/etc/hosts"
    local tmp_hosts_file="/etc/hosts-maintain-tools-bak"
    if [ -n "${REF_IP_HOSTNAME}" ]
    then
        if [ -f "${hosts_file}" ]
        then
            cp -a "${hosts_file}" "${tmp_hosts_file}"
            if [ -n "${old_ip}" ]
            then
                grep -w "^${old_ip}" "${hosts_file}" > /dev/null 2>&1
                if [ $? -eq 0 ]
                then
                    sed -i "s#^${old_ip}.*#${REF_IPADDR} ${REF_IP_HOSTNAME}#g" "${hosts_file}" > "${TMP_CONTENT_FILE}" 2>&1
                    if [ $? -ne 0 ]
                    then
                        PRINT_LOG "FATAL" "Excute cmd<sed -i \"s#^${old_ip}.*#${REF_IPADDR} ${REF_IP_HOSTNAME}#g\" \"${hosts_file}\"> to modifty ${hosts_file} failure"
                        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
                        return 1
                    fi
                    PRINT_LOG "INFO" "Excute sed cmd to modifty ${hosts_file} success"
                else
                    echo "${REF_IPADDR} ${REF_IP_HOSTNAME}" >> "${hosts_file}"
                    PRINT_LOG "INFO" "Excute cmd<echo \"${REF_IPADDR} ${REF_IP_HOSTNAME}\"> to modifty ${hosts_file} success"
                fi
             else
                 echo "${REF_IPADDR} ${REF_IP_HOSTNAME}" >> "${hosts_file}"
                 PRINT_LOG "INFO" "Excute cmd<echo \"${REF_IPADDR} ${REF_IP_HOSTNAME}\"> to modifty ${hosts_file} success"
             fi
        else
            PRINT_LOG "WARN" "file ${hosts_file} is not exist"
        fi
     else
        PRINT_LOG "INFO" "REF_IP_HOSTNAME<${REF_IP_HOSTNAME}> is empty, no need to change ${hosts_file} file"
     fi
    
    return ${SUCCESS_CODE};
}

#=======================================================================
fn_main "$@"
ret=$?
fn_write_operation_log ${ret}
FUNC_CLEAN_TMP_FILE
exit ${ret}
#=======================================================================
