#!/bin/bash
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)
TAIL=$(date +%Y%m%d%H%M%S)
LOG_FILE="/var/log/${SELF_FILE//.sh/}_${TAIL}.log"

# 工程临时文件目录
TMP_DIR="/opt/engr_tmp"
# 成功执行sek_root.sh脚本后生成的标志文件
TMP_SUCCESS="${TMP_DIR}/expansion_success"


function LOG()
{
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" 1>&2
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" >> "${LOG_FILE}"
        logger -p local0.error "[${SELF_FILE}] ${Level} | $2"
    else
        Level="INFO"
        echo "[$(date +%Y)-$(date +%m)-$(date +%d) $(date +%H):$(date +%M):$(date +%S)] [${SELF_FILE}] ${Level} | $2" | tee -a "${LOG_FILE}"
        logger -p local0.info "[${SELF_FILE}] ${Level} | $2"
    fi
}

#create action.json check file
createCheckFile()
{
    echo "#!/bin/bash" >/opt/expansion/check_sek_root_result.sh
    echo "echo  RESULT:1" >>/opt/expansion/check_sek_root_result.sh
    chmod 750 /opt/expansion/check_sek_root_result.sh
}

# 异步查询日志
getcheckResult()
{
   if [ $1 -eq 0 ]
   then
       echo "#!/bin/bash
echo  RESULT:100 && rm -rf /opt/expansion/*
   ">/opt/expansion/check_sek_root_result.sh
   else
       echo "#!/bin/bash
echo  RESULT:255 && rm -rf /opt/expansion/check_sek_root_result.sh
   ">/opt/expansion/check_sek_root_result.sh
   fi
   chmod 750 /opt/expansion/check_sek_root_result.sh > /dev/null 2>&1
   return $1
}

postdo()
{
    LOG "INFO" "Executing Function: postdo"
    #修改su配置，禁用su之前的用户的环境变量
    if [ -f "/etc/default/su" ];then
        sed -i "/ALWAYS_SET_PATH/ s/no/yes/g" /etc/default/su
    fi

    #控制文件权限
    find /opt/install -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod 550
    find /opt/tools -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod 550

    for targetDir in "/opt/install" "/opt/SEK" "/opt/NCEICMR"
    do
        if [ -d "$targetDir" ];then
            find $targetDir -type f -name "*.log" -o -name "*.sh.log*" | xargs -i echo '"{}"' | xargs chmod 640
            find $targetDir -type f -name "*.xml" -o -name "*.properties" -o -name "*.conf" -o -name "*.ini" | xargs -i echo '"{}"' | xargs chmod 400
            find "${targetDir}" -type f -name "*.json" -o -name "*.xml" -o -name "*.txt" -o -name "*.conf" -o -name "*.properties" -o -name "*.csv" | xargs -L 1 chmod 400 >/dev/null 2>&1
        fi
    done

}

# 清理OMP节点的标志文件
function clean_tmp_file()
{
    if [ ${CONFIG_FLAG} -eq 1 ]
    then
        [ -f "${TMP_SUCCESS}" ] && rm -f "${TMP_SUCCESS}"
        if [[ -f "${TMP_SUCCESS}" ]]
        then
            LOG "ERROR" "Failed to delete the flag file."
        else
            LOG "INFO" "Succeeded in deleting the flag file."
        fi
    fi
}

# 获取OMP_01的ip地址
function get_omp01_ip()
{
    local manager_ips=$(cat /opt/oss/manager/var/agent/managerip.conf | \
    grep -w 'managerip' | awk -F= '{print $2}' | sed 's/,/\n/g')
    local node_ips=$(su - ossadm -c ". /opt/oss/manager/bin/engr_profile.sh; \
    cat /opt/oss/manager/var/tenants/manager/nodelist.json | python -c 'import json; import sys; obj=json.load(sys.stdin); \
    print(obj[\"nodeList\"][\"0\"][\"IPAddresses\"])'" | sed 's/,/\n/g' | grep -w 'IP' | awk -F"'" '{print $(NF-1)}')
    SETIP=""
    for tmp_node0 in ${node_ips}
    do
        for tmp_manager in ${manager_ips}
        do
            if [ "${tmp_manager}" == "${tmp_node0}" ]
            then
                SETIP="${tmp_manager}"
                return 0
            fi
        done
    done
}

# 判断节点是否需要执行
function check_node_message()
{
    # 调用用户禁用函数标志位，0表示不执行，1表示执行
    CONFIG_FLAG=0
    localip=$(su - ossadm -c '. /opt/oss/manager/bin/engr_profile.sh;python -c "from util import common;print(common.get_local_ip())"')
    get_omp01_ip
    if [ "${localip}" == "${SETIP}" ]
    then
        CONFIG_FLAG=1
    fi
}


# 启动sshd加固
function config_sshd()
{
    LOG "INFO" "Start to enable user isolation."
    sshd_tool="/opt/oss/manager/agent/bin/sshd_ipsadm.sh"
    if [ -f "${sshd_tool}" ]
    then
        # 启动加固
        su - ossadm -c "bash ${sshd_tool} -cmd set >/dev/null 2>&1"
        if [ $? -ne 0 ]
        then
            su - ossadm -c "bash ${sshd_tool} -cmd restore >/dev/null 2>&1"
            LOG "ERROR" "Execute '/opt/oss/manager/agent/bin/sshd_ipsadm.sh -cmd set' Failed."
            return 1
        else
            LOG "INFO" "Succeeded in enabling user isolation."
            return 0
        fi
    else
        LOG "ERROR" "/opt/oss/manager/agent/bin/sshd_ipsadm.sh not found."
        return 1
    fi
}

# 判断单节点sek_root脚本是否执行完成
function check_node()
{
    local node_ip=$1
    local tmp_time=0
    while true
    do
        if [ -f "${SCRIPT_PATH}"/success_node.log ]
        then
            grep -qw "${node_ip}" "${SCRIPT_PATH}"/success_node.log
            [ $? -eq 0 ] && return 0
        fi
        check_result=$(su - ossadm -c "ssh -o stricthostkeychecking=no \"${node_ip}\" \"([ -n "${TMP_SUCCESS}" ] && [ -f "${TMP_SUCCESS}" ]) && (rm -f "${TMP_SUCCESS}" && echo "success") || echo "false"\"" 2>/dev/null)
        if [ "${check_result}" == "success" ]
        then
            LOG "INFO" "Checking ${node_ip} succeeded."
            echo "${node_ip}" >> "${SCRIPT_PATH}"/success_node.log
            break
        fi
        let tmp_time++
        if [ ${tmp_time} -eq 1200 ]
        then
            LOG "ERROR" "Failed to check ${node_ip}."
            return 1
        fi
        if [ -f "${fail_flag}" ]
        then
            LOG "ERROR" "Failed to check ${node_ip}."
            return 1
        fi
        sleep 3
    done
}

# 判断除OMP主节点的其他sek_root操作已完成
function check_finish()
{
    local count=0
    nodelist=($(echo ${newip} | sed 's/,/\n/g'))
    [ ${#nodelist[@]} -eq 0 ] && return 0
    fail_flag="${SCRIPT_PATH}/check_node_fail.flag"
    [ -n "${fail_flag}" ] && rm "${fail_flag}" >/dev/null 2>&1
    LOG "INFO" "Start to check nodes(${nodelist[*]})."
    for str_node in "${nodelist[@]}"
    do
        {
            check_node "${str_node}"
            [ $? -ne 0 ] && touch "${fail_flag}"
        }&
        let count++
        if [ ${count} -eq 30 ]
        then
            wait
            count=0
        fi
    done
    wait
    if [ -f "${fail_flag}" ]
    then
        [ -n "${fail_flag}" ] && rm "${fail_flag}" >/dev/null 2>&1
        return 1
    fi
    LOG "INFO" "All nodes are checked successfully."
}

# 执行用户禁用，仅OMP01节点执行
function deny_user_ssh()
{
    rm -f "${TMP_SUCCESS}"
    check_finish
    if [ $? -eq 0 ]
    then
        local count=0
        while true
        do
            config_sshd
            if [ $? -ne 0 ]
            then
                sleep 60
                let count++
                if [ ${count} -eq 2 ]
                then
                    LOG "ERROR" "Failed to enable user isolation."
                    return 1
                fi
            else
                break
            fi
        done
    else
        LOG "ERROR" "Failed to check other nodes."
        return 1
    fi
}


# 开启root登录权限
function open_root_ssh()
{
    if [ -f "/opt/SEK/cmd/SetEnv.sh" ]
    then
        bash /opt/SEK/cmd/SetEnv.sh
    else
        LOG "ERROR" "/opt/SEK/cmd/SetEnv.sh does not exist. Please manually enable the SSH permission for the root user."
    fi
}

# 设置特定用户密码过期配置
function set_expiration()
{
    LOG "INFO" "Execute script: set_password_expiration.sh"
    if [ -f "${SCRIPT_PATH}/set_password_expiration.sh" ];then
        bash "${SCRIPT_PATH}/set_password_expiration.sh" 2>${LOG_FILE}
        if [ $? -eq 0 ];then
            LOG "INFO" "The ${SCRIPT_PATH}/set_password_expiration.sh script is executed successfully."
        else
            LOG "ERROR" "Failed to execute the ${SCRIPT_PATH}/set_password_expiration.sh script."
            return 1
        fi
    else
        LOG "ERROR" "${SCRIPT_PATH}/set_password_expiration.sh script is not found."
        return 1
    fi
}

# 创建sek_root标志文件
function create_sek_tmp()
{
    [ -f "${TMP_SUCCESS}" ] && rm -f "${TMP_SUCCESS}"
    mkdir -p ${TMP_DIR} 2>/dev/null
    chown ossadm:ossgroup ${TMP_DIR} 2>/dev/null
    su - ossadm -c "touch ${TMP_SUCCESS}" &>>${LOG_FILE}
    if [ $? -eq 0 ]
    then
        LOG "INFO" "The flag file is created successfully."
    else
        LOG "ERROR" "Failed to create the flag file."
        return 1
    fi
}

# 用于恢复由于Easysuite修改的配置项
recover_sshd_for_tunnel()
{
    sed -i "/.*#Easysuite$/d" /etc/ssh/sshd_config
}

function open_port_listen() {
    if [ -f "/opt/oss/manager/adapter/common/resmgr/init_config.sh" ]
    then
        LOG "INFO" "Start to open port listening."
        su - ossadm -c "cd /opt/oss/manager/adapter/common/resmgr && bash init_config.sh"
        if [ $? -ne 0 ]
        then
            LOG "ERROR" "Failed to open port listening."
            return 1
        fi
        LOG "INFO" "Port listening enabled successfully."
    fi
}

# 返回值0: 运行正常
#       1: 和root禁用无关的步骤失败
#       2：和root禁用相关的步骤失败，退出脚本前开启当前节点root登录权限
function main()
{
    LOG "INFO" "Start to execute the sek_root.sh script."

    # 生成加固日志文件
    createCheckFile

    # 设置特定用户密码不过期
    set_expiration

    # 检查是否需要执行deny_user_ssh
    check_node_message

    #执行后置刷权限
    postdo >/dev/null 2>&1

    # 南向ip禁用脚本 && 22端口监听
    open_port_listen || return 1

    # 防止sshd服务在5s内重启次数过多
    sleep 10

    # 启动平台加固,如有新增节点,必须要传入新增节点的节点间通讯IP
    # 如有旧节点也解开了root加固，也必须要传入旧节点的节点间通讯IP
    if [ ${CONFIG_FLAG} -eq 1 ]
    then
        newip=$2
        if [ -n "${newip}" ]
        then
            deny_user_ssh || return 2
        else
            config_sshd
            if [ $? -ne 0 ]
            then
                LOG "ERROR" "Failed to enable user isolation."
                return 2
            fi
        fi
    else
        LOG "INFO" "This node(${localip}) not need to enable user isolation, SETIP is ${SETIP}."
    fi

    recover_sshd_for_tunnel

    if [ "$1" == "id_sek_select_yes" ]
    then
        #选择启用安全加固时，禁用root SSH登陆
        LOG "INFO" "This node(${localip}) execute RunSEK.sh."
        if [ -f "/opt/SEK/RunSEK.sh" ];then
            cd /opt/SEK;bash RunSEK.sh >/dev/null 2>&1
            local lockResult=$?
            if [[ ${lockResult} != 0 ]]
            then
                LOG "ERROR" "Failed to execute the RunSEK.sh script."
                # 加固不成功,解加固后再退出
                cd /opt/SEK/cmd;bash SetEnv.sh >/dev/null 2>&1
                return 1
            fi
        fi
    elif [ "$1" == "rollback_env" ]
    then
        LOG "INFO" "This node(${localip}) execute RollBackEnv.sh."
        if [ -f "/opt/SEK/cmd/RollBackEnv.sh" ];then
            cd /opt/SEK/cmd;bash RollBackEnv.sh >/dev/null 2>&1
            local lockResult=$?
            if [[ ${lockResult} != 0 ]]
            then
                LOG "ERROR" "Failed to execute the RollBackEnv.sh script."
                # 加固不成功,解加固后再退出
                cd /opt/SEK/cmd;bash SetEnv.sh >/dev/null 2>&1
                return 1
            fi
        fi
    fi

    # 创建sek_root标志文件
    create_sek_tmp || return 2
}

# 更新aide数据库
function update_aide_database()
{
    LOG "INFO" "Execute script: update_aide_db.sh"
    if [ -f "${SCRIPT_PATH}/update_aide_db.sh" ];then
        bash "${SCRIPT_PATH}/update_aide_db.sh" 2> ${LOG_FILE}
        if [ $? -eq 0 ];then
            LOG "INFO" "The ${SCRIPT_PATH}/update_aide_db.sh script is executed successfully."
        else
            LOG "ERROR" "Failed to execute the ${SCRIPT_PATH}/update_aide_db.sh script."
            return 1
        fi
    else
        LOG "ERROR" "${SCRIPT_PATH}/update_aide_db.sh script is not found."
        return 1
    fi
}

#加锁
function lock() {
    [[ -f /usr/lib/systemd/system/lock.service ]] || return 0
    systemctl enable lock &> /dev/null
    local lockResult=$?
    if [[ ${lockResult} != 0 ]]
    then
        #记录下失败日志
        LOG "ERROR" "Enable lock failed with ${lockResult}"
        return ${lockResult}
    fi
    systemctl start lock &> /dev/null
    lockResult=$?
    if [[ ${lockResult} != 0 ]]
    then
        #记录下失败日志
        LOG "ERROR" "Start lock failed with ${lockResult}"
    fi
    return ${lockResult}
}

main $*
result=$?
if [ ${result} -eq 0 ]
then
    LOG "INFO" "Succeeded in executing the sek_root.sh script."
    clean_tmp_file
elif [ ${result} -eq 1 ]
then
    LOG "ERROR" "Failed to execute the sek_root.sh script."
elif [ ${result} -eq 2 ]
then
    LOG "ERROR" "Failed to execute the sek_root.sh script."
    open_root_ssh
else
    LOG "ERROR" "Failed to execute the sek_root.sh script."
    open_root_ssh
fi

if [ ${result} -ne 0 ]
then
  getcheckResult ${result}
  exit 1
fi

update_aide_database
lock
result=$?
getcheckResult ${result}
exit ${result}