#!/bin/bash
###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
# File name: post_rollback_refresh_os_patch_status.sh
# Description: refresh OS patch status on all nodes.
###############################################################################

WORKPATH=$(dirname $(readlink -f "$0"))
PATCH_NAME=""

GAUSSDB_IP=""
SQL_RESULT=""
RECORD_FILE=""

SCRIPT_NAME=$(basename $0)
LOG_FILE="/opt/oss/log/manager/easysuite_upgrade/${SCRIPT_NAME}.log"
REFRESH_OSPATCH_STATUS_FOR_ROLLBACK="/opt/patch_manager/NCE_OSPatch/osbackup_must_not_delete/refresh_ospatch_status_for_rollback"

##################################################
# Description: 校验脚本执行用户
# Parameters: null
##################################################
function check_user() {
    if [ "$(whoami)" != "ossadm" ]; then
        print_log "INFO" "Only the ossadm user can run this script."
        return 1
    fi
}

##################################################
# Description: 日志函数
# @Param1: 日志级别
# @Param2: 日志信息
##################################################
function print_log() {
    echo -e "$(date +%Y-%m-%d) $(date +%H:%M:%S) | ${1} | ${2}" | tee -a ${LOG_FILE}
}


###################################################################################
# Description： 查询数据库实例，非公开接口
# @Param: null
##################################################################################
function get_python_scripts() {
    if [ -f "${WORKPATH}/get_node_id.py" ]; then
        get_node_id_python_script="${WORKPATH}/get_node_id.py"
    else
        get_node_id_python_script="${WORKPATH}/get_node_id.pyc"
    fi

    if [ -f "${WORKPATH}/sql.py" ]; then
        sql_python_script="${WORKPATH}/sql.py"
    else
        sql_python_script="${WORKPATH}/sql.pyc"
    fi
    return 0
}

##################################################################################
# Description： 查询数据库实例，非公开接口
# @Param: null
##################################################################################
function get_query_db_instance() {
    local manager_dbnum=$(echo "${RECORD_FILE}" | grep -cw "manager")
    if [ "${manager_dbnum}" -eq 1 ];then
        GAUSSDB_IP=$(echo "${RECORD_FILE}" | grep -w "manager" | awk '{print $6}')
    else
        GAUSSDB_IP=$(echo "${RECORD_FILE}" | grep -w "manager" | grep -w "Master" | awk '{print $6}')
    fi
}

###################################################################################
# Description： 前置步骤，获取执行SQL所需的信息
# @Param: null
##################################################################################
function query_db_info(){
    . /opt/oss/manager/bin/engr_profile.sh
    # 查询所有节点ID
    ID_LIST=$(python "${get_node_id_python_script}") || return $?
    # 非公开接口
    RECORD_FILE=$(/opt/oss/manager/apps/DBAgent/bin/dbsvc_adm -cmd query-db-instance -type zenith) || return $?
    # 查询数据库实例
    get_query_db_instance || return $?

    if [ -z "${ID_LIST}" ] || [ -z "${ID_LIST}" ] || [ -z "${ID_LIST}" ]; then
        print_log "WARNING" "Failed to query the database information."
        return 1
    fi
    return 0
}

###################################################################################
# Description： 执行SQL
# @Param: SQL语句
##################################################################################
function execute_sql() {
    local sql="$1"
    #这几个变量固定
    local db_list="dbList"
    local db_name="ospatchdb"
    local db_user_passwd="dbUserPasswd"
    local container_path="/opt/oss/manager/var/tenants/manager/containerlist.json"
    local read_port="32080"
    local inst_number="managedbsvr-0-999"
    SQL_RESULT=$(python "${sql_python_script}" "${container_path}" "${inst_number}" "${db_list}" "${db_name}" "${db_user_passwd}" "${GAUSSDB_IP}" "${read_port}" "${sql}")
    if [ $? -ne 0 ]; then
        print_log "WARNING" "Execute sql '${sql}' Failed."
        return 1
    fi
}

###################################################################################
# Description： 执行SQL检查所有节点的补丁安装状态，SQL返回结果赋值给SQL_QUERY_RESULT
# @Param: null
# @Return: null
##################################################################################
function sql_query() {
    # 查询所有节点状态，返回值为('节点ID','补丁包名','补丁状态','安装日期',...)
    local query_sql="select NODEID,PATCHPACKAGENAME,PATCHINSTALLSTATUS,INSTALLTIME,RESERVED0,RESERVED1,RESERVED2,RESERVED3,RESERVED4,RESERVED5 from TAL_PATCH_INFO"

    execute_sql "${query_sql}" || return $?
    SQL_QUERY_RESULT=$(echo "${SQL_RESULT}" | sed 's/ //g' | sed 's/),/\n/g')
}

###################################################################################
# Description： 根据SQL_QUERY_RESULT与node_id检查各节点的补丁安装状态
# @Param: 节点ID
# @Return: 0--已安装补丁 | 1--未安装补丁 | 255--未知状态
##################################################################################
function check_each_node_status() {
    local node_id="$1"
    local patch_status

    # 单节点补丁状态字段
    local single_node_result=$(echo "${SQL_QUERY_RESULT}" | sed 's/ //g' | sed 's/),/\n/g' | grep "'${node_id}','${PATCH_NAME}'")

    if [ -z "${single_node_result}" ]; then
        print_log "INFO" "No patch is installed on node ${node_id}."
        return 1
    else
        # 获取对应某个节点ID的安装状态2是已安装
        patch_status=$(echo "${single_node_result}" | awk -F',' '{print $3}' | tr -d "'")
        if [ "${patch_status}" -eq 2 ]; then
            print_log "INFO" "Patch is already installed in node ${node_id}."
            return 0
        else
            print_log "WARNING" "Unknown patch status on node ${node_id}."
            return 255
        fi
    fi
}

###################################################################################
# Description： 查询所有节点的补丁安装状态
# @Param: null
# @Return: 0--已安装补丁 | 1--未安装补丁 | 2--部分节点安装补丁
##################################################################################
function query_patch_status() {
    print_log "INFO" "Begin to query patch status~"
    local installed_node_count=0
    local all_node_count=$(echo "${ID_LIST}" | awk '{print NF}')

    sql_query || return $?
    print_log "INFO" "SQL Query Result: (${SQL_QUERY_RESULT})"

    for node_id in ${ID_LIST}; do
        check_each_node_status "${node_id}"
        if [ $? -eq 0 ]; then
            let installed_node_count++
        fi
    done

    if [ "${installed_node_count}" -eq 0 ];then
        print_log "INFO" "Patch status: Not Installed"
        return 1
    elif [ "${installed_node_count}" -eq "${all_node_count}" ]; then
        print_log "INFO" "Patch status: Installed"
        return 0
    elif [ "${installed_node_count}" -lt "${all_node_count}" ]; then
        print_log "INFO" "Patch status: Partially Installed"
        return 2
    fi
}

###################################################################################
# Description： 将所有节点的补丁安装状态刷新为已安装
# @Param: null
# @Return: 0--已安装补丁 | 1--未安装补丁 | 2--部分节点安装补丁
##################################################################################
function to_installed() {
    local localtime=$(date "+%Y%m%d%H%M%S")

    print_log "INFO" "Begin to update the patch status to installed~"
    sql_query || return $?
    for node_id in ${ID_LIST}; do
        check_each_node_status "${node_id}" > /dev/null
        # 若未查询到补丁安装状态，则插入一条数据，否则更新原有数据
        if [ $? -ne 0 ]; then
            local insert_sql="INSERT INTO TAL_PATCH_INFO (NODEID, PATCHPACKAGENAME, PATCHINSTALLSTATUS, INSTALLTIME) VALUES ('${node_id}','${PATCH_NAME}','2','${localtime}')"
            execute_sql "${insert_sql}" || return $?
        fi
    done
    # 刷新数据库成功之后，清除文件
    rm ${REFRESH_OSPATCH_STATUS_FOR_ROLLBACK}
    print_log "INFO" "Updating the patch status succeeded(install)."
}

###################################################################################
# Description： 将所有节点的补丁安装状态刷新为未安装
# @Param: null
# @Return: 0--已安装补丁 | 1--未安装补丁 | 2--部分节点安装补丁
##################################################################################
function to_uninstall() {
    local localtime=$(date "+%Y%m%d%H%M%S")

    print_log "INFO" "Begin to update the patch status to uninstall~"
    for node_id in ${ID_LIST}; do
        local modify_sql="DELETE FROM TAL_PATCH_INFO WHERE NODEID='${node_id}' and PATCHPACKAGENAME='${PATCH_NAME}'"
        execute_sql "${modify_sql}" || return $?
    done
    print_log "INFO" "Updating the patch status succeeded(uninstall)."
}


###################################################################################
# Description： 判断是否需要更新数据库中记录的OS补丁状态
# @Param: null
# @Return: 0--需要更新 | 1--不需要更新 
##################################################################################
function is_need_to_refresh_ospatch_status() {
    # $REFRESH_OSPATCH_STATUS_FOR_ROLLBACK 文件不存在时，则说明不需要更新数据库中记录的OS补丁状态
    if [ -f ${REFRESH_OSPATCH_STATUS_FOR_ROLLBACK} ]; then
        PATCH_NAME=$(cat ${REFRESH_OSPATCH_STATUS_FOR_ROLLBACK} |awk -F ' ' '{print $2}')
        print_log "INFO" "PATCH_NAME=${PATCH_NAME}"
        return 0
    fi
    print_log "INFO" "The OS patch information recorded in the database does not need to be updated."
    return 1
}


function main() {
    check_user || return $?
    is_need_to_refresh_ospatch_status || return 0
    get_python_scripts || return $?
    query_db_info || return $?
    
    to_installed || return $?
    query_patch_status

}
main
exit 0