# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from abc import ABC, abstractmethod

from data.alarm_receive_param import get_alarm_receive_param
from utils.common import exec_cmd, check_ping
from constant.constant import SSH_CMD, RetCode
from utils.health_check_result import HealthCheckResult
from utils.log_utils import get_logger
from utils.manager import Manager
from data.service_info import ServiceInfo

LOGGER = get_logger()


class Service(ABC):
    """ 服务健康检查 基类 """

    def __init__(self, service_info: ServiceInfo):
        self.service_info = service_info
        self.alarm_receive_param = get_alarm_receive_param(self.service_info.service_ip)

    @abstractmethod
    def check_connect_cmd(self):
        """ 获取检查用户名，密码认证命令 """
        pass

    @abstractmethod
    def get_service_alarm_forward_param(self):
        """ 获取告警转发参数 """
        pass

    def execute_cmd(self, cmd: str) -> tuple:
        """ 执行远程命令 """
        cmd_str = f"bash {SSH_CMD} {self.service_info.service_ip} {self.service_info.service_user} " \
                  f"{self.service_info.service_type} '{cmd}'"
        return exec_cmd(cmd=cmd_str, pwd=self.service_info.pwd)

    def check_local_alarm_ip_ping_to_service_ip(self) -> bool:
        """ 校验本地告警监听IP到服务器/交换机/FC配置IP的连通性 """
        LOGGER.info(f"[{self.service_info}] start to "
                    f"ping from {Manager().alarm_listening_ip} to {self.service_info.service_ip}")
        return check_ping(Manager().alarm_listening_ip, self.service_info.service_ip)

    def check_connect(self) -> bool:
        """ 检查用户名，密码是否正确，是否可以正常登录 """
        LOGGER.info(f"[{self.service_info}] start to check connect")
        ret, _, _ = self.execute_cmd(self.check_connect_cmd())
        return ret == 0

    def check_alarm_receive_param_exist(self) -> bool:
        """ 检查健康检查配置的服务是否在管理面告警接收参数中配置 """
        LOGGER.info(f"[{self.service_info}] start to check alarm receive param {self.alarm_receive_param} exist")
        if not self.alarm_receive_param:
            LOGGER.error(f"[{self.service_info}] can not get alarm receive param from manager")
            return False
        return True

    def check_alarm_receive_param(self) -> bool:
        """ 校验告警接收参数 """
        LOGGER.info(f"[{self.service_info}] start to check alarm receive param by cloudSOP rest interface")
        return Manager().verify_alarm_receive_param(self.alarm_receive_param.get_json())

    def check_alarm_forward_param(self) -> bool:
        """ 检查服务器的告警转发参数 """
        LOGGER.info(f"[{self.service_info}] start to check server alarm config")
        alarm_listening_ip_list = Manager().alarm_listening_ip_list
        LOGGER.info(f"[{self.service_info}] alarm listening ip list is {alarm_listening_ip_list}")
        LOGGER.info(f"[{self.service_info}] alarm receive param is {self.alarm_receive_param}")
        alarm_forward_param = self.get_service_alarm_forward_param()
        error_alarm_listening_ip_list = []
        for alarm_listening_ip in alarm_listening_ip_list:
            result = [snmp for snmp in alarm_forward_param if
                      snmp.match(alarm_listening_ip, self.alarm_receive_param.snmp_user,
                                 Manager().alarm_listening_port)]
            LOGGER.info(f"[{self.service_info}] alarm listening ip {alarm_listening_ip} match result is {result}")
            if not result:
                error_alarm_listening_ip_list.append(alarm_listening_ip)
        LOGGER.info(f"[{self.service_info}] get wrong alarm listening ip is {error_alarm_listening_ip_list}")
        return len(error_alarm_listening_ip_list) == 0

    def add_result(self, ret_code: RetCode) -> None:
        """ 添加检查结果 """
        HealthCheckResult().add_result(ret_code, self.service_info)

    @abstractmethod
    def build_check_function_list(self) -> list:
        """ 生成check list，列表内容为(func, RetCode)"""
        pass

    def check(self) -> None:
        """ 健康检查入口 """
        LOGGER.info(f"[{self.service_info}] start to health check")
        for check_func in self.build_check_function_list():
            if not check_func[0]():
                self.add_result(check_func[1])
                return
