# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import datetime
import os

from service.service_factory import get_service_list
from constant.constant import ServiceType, InstallType, RetCode
from utils.health_check_result import HealthCheckResult
from utils.log_utils import get_logger
from utils.manager import Manager
from utils.thread_pool import MultiThreadPool

LOGGER = get_logger()


class HardwareParamCheck(object):

    def health_check(self):
        """
        执行硬件参数健康检查
        """
        LOGGER.info("start enter health check")
        if not Manager().alarm_listening_ip_list:
            # 没有告警监听IP，默认是私有云场景，不进行健康检查
            LOGGER.info("alarm listening ip is empty, ignore to enter health check")
            return
        server_service_list = Manager().get_service_info_list(str(ServiceType.SERVER.value))
        fc_service_list = Manager().get_service_info_list(str(ServiceType.FC.value))
        switch_service_list = Manager().get_service_info_list(str(ServiceType.SWITCH.value))
        self.check_hardware_info_configuration(server_service_list, fc_service_list, switch_service_list)

        service_info_list = server_service_list + fc_service_list + switch_service_list
        if not service_info_list:
            # 如果查询到DFS的信息为空，则忽略检查
            LOGGER.info("service info is empty, ignore to enter health check")
            return
        service_list = get_service_list(service_info_list)
        thread_pool = MultiThreadPool()
        for service in service_list:
            thread_pool.add_thread(service.check)
        thread_pool.execute_with_result()

    @staticmethod
    def check_hardware_info_configuration(server_service_list, fc_service_list, switch_service_list) -> None:
        """
        检查是否进行了硬件信息配置
        :param server_service_list: 服务器信息
        :param fc_service_list: FC信息
        :param switch_service_list: 交换机信息
        :return: None
        """
        install_type = Manager().install_type
        LOGGER.info("install type is %s", install_type)
        if install_type == InstallType.SINGLE and not server_service_list:
            # 单管：没有服务器信息
            LOGGER.error("install type is %s, server info not exist",install_type)
            HealthCheckResult().add_result(RetCode.HARDWARE_INFO_NOT_EXIST_SINGLE)
        if install_type == InstallType.MINI and (not server_service_list or not fc_service_list):
            # 小型化：没有服务器信息或者FC信息
            LOGGER.error("install type is %s, server info or fc info not exist", install_type)
            HealthCheckResult().add_result(RetCode.HARDWARE_INFO_NOT_EXIST_MINI)
        if install_type == InstallType.DISTRIBUTED and (
                not server_service_list or not fc_service_list or not switch_service_list):
            # 分布式：没有服务器信息或者FC信息或者交换机信息
            LOGGER.error("install type is %s, server info or fc info or switch info not exist", install_type)
            HealthCheckResult().add_result(RetCode.HARDWARE_INFO_NOT_EXIST_DISTRIBUTED)

    @staticmethod
    def get_result_list() -> list:
        """ 健康检查后置处理 """
        Manager().clean_tmp_file()
        return HealthCheckResult().print_result()
