# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import json
import os

from constant.constant import WORK_DIR, RetCode
from data.service_info import ServiceInfo
from exception.exception import HealthCheckException
from utils.common import LazyProperty
from utils.log_utils import get_logger
from utils.manager import Manager
from utils.singleton import Singleton

LOGGER = get_logger()


class SingleResult(object):
    """ 单个Service检查结果 """

    def __init__(self, err_info: dict):
        self.error_level = err_info.get("error_level", "")
        self.error_code = err_info.get("error_code", "")
        self.error_info = err_info.get("error_info", "")
        self.suggest = err_info.get("suggest", "")

    def to_json(self):
        return {
            "ErrLevel": int(self.error_level),
            "ErrInfo": self.error_info,
            "ErrCode": self.error_code,
            "Suggest": self.suggest
        }


class HealthCheckResult(Singleton):
    """ 健康检查结果汇总类 单实例 """
    SERVICE_TYPE_FLAG = "{service_type}"
    SERVICE_IP_FLAG = "{service_ip}"
    ALARM_LISTENING_IP_FLAG = "{alarm_listening_ip}"
    HEALTH_CHECK_RESULT = []

    def __init__(self):
        super().__init__()
        self._lan = "en_US"

    @LazyProperty
    def _resource_content(self) -> dict:
        resource = os.path.join(WORK_DIR, "resource", "resource.json")
        if not os.path.exists(resource):
            raise HealthCheckException(f"resource file {resource} not exist")
        with open(resource, mode="r", encoding="utf-8") as file_obj:
            content = json.load(file_obj)
        LOGGER.info(f"resource content is {content}")
        return content

    def _replace_flag(self, msg: str, service: ServiceInfo) -> str:
        service_type = self._resource_content.get("service_type").get(self._lan).get(service.service_type)
        msg = msg.replace(self.SERVICE_TYPE_FLAG, service_type) if self.SERVICE_TYPE_FLAG in msg else msg
        msg = msg.replace(self.SERVICE_IP_FLAG, service.service_ip) if self.SERVICE_IP_FLAG in msg else msg
        msg = msg.replace(self.ALARM_LISTENING_IP_FLAG,
                          Manager().alarm_listening_ip) if self.ALARM_LISTENING_IP_FLAG in msg else msg
        return msg

    def _get_msg(self, err_code: RetCode) -> SingleResult:
        return SingleResult(self._resource_content.get("result").get(self._lan).get(err_code.value))

    def add_result(self, ret_code: RetCode, service: ServiceInfo = None) -> None:
        """ 新增检查结果 """
        LOGGER.info(f"add result, service: {service}, ret code: {ret_code}")
        msg = self._get_msg(ret_code)
        if service:
            msg.error_info = self._replace_flag(msg.error_info, service)
            msg.suggest = self._replace_flag(msg.suggest, service)
        LOGGER.info(f"add result: {msg.to_json()}")
        HealthCheckResult.HEALTH_CHECK_RESULT.append(msg.to_json())

    @staticmethod
    def print_result() -> list:
        """ 打印结果 """
        if not HealthCheckResult.HEALTH_CHECK_RESULT:
            # 无错误检查结果，默认置为成功。
            HealthCheckResult().add_result(RetCode.SUCCESS_CODE)
        result = json.dumps(HealthCheckResult.HEALTH_CHECK_RESULT)
        LOGGER.info(f"print result: {result}")
        return HealthCheckResult.HEALTH_CHECK_RESULT
