#!/bin/bash

##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

##################################################
# 初始化日志路径和日志文件名
##################################################
LOG_PATH="/opt/upgrade/easysuite_upgrade/logs"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
UPGRADE_PATH="/opt/upgrade"
SOP_GROUP_ID="2002"
OSS_USER="ossadm"
SOP_USER="sopuser"

##################################################
# 校验执行用户
# 脚本要求使用oss用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script"
        return 1
    fi
}

##################################################
# 定义相关全局变量
##################################################
function init_variable()
{
    # Easysuite任务相关日志/进度/状态文件
    TASK_ID_PATH="/opt/upgrade/easysuite_upgrade/taskmgr/${TASKID}"
    TASK_LOG_FILE="${TASK_ID_PATH}/task.log"
    TASK_PROGRESS_FILE="${TASK_ID_PATH}/task.progress"
    TASK_STATUS_FILE="${TASK_ID_PATH}/task.status"

    # OMP目录
    OMP_PATH="/opt/upgrade/${PKGNAME//.zip/}"
    ROLLBACK_SCRIPT="${OMP_PATH}/Rollback_manager.sh"

    # 是否是新任务标志位
    NEW_TASK="TRUE"
    [ -d "${TASK_ID_PATH}" ] && NEW_TASK="FALSE"
}

##################################################
# 刷Easysuite任务状态
##################################################
function fresh_status() {
    local status=$1
    echo Status=${status} >${TASK_STATUS_FILE}
}

##################################################
# 刷Easysuite任务进度
##################################################
function fresh_progress() {
    local progress=$1
    echo Progress=${progress} >${TASK_PROGRESS_FILE}
}

##################################################
# 刷Easysuite结果为成功
##################################################
function fresh_finish()
{
    fresh_progress "100"
    fresh_status "finish"
}

##################################################
# 刷Easysuite结果为失败
##################################################
function fresh_fail()
{
    fresh_progress "100"
    fresh_status "fail"
}

##################################################
# 校验入参
# @param
#   action --rollback/retry
#   pkgname   --NCEV100R019C00*_OMP_linux_x64.zip
#   taskid   --c6bfc6d0a769c2b319f9
##################################################
function check_input()
{
    input_vaild="TRUE"
    if [ "$#" -ne 6 ]
    then
        input_vaild="FALSE"
    fi
    if [ "${1}" != "-action" -o "${3}" != "-pkgname" -o "${5}" != "-taskid" ]
    then
        input_vaild="FALSE"
    fi
    case "${2}" in
        # step2、pre_rollback、default_rollback、app_rollback、kernel_rollback
        step1 | step2  | rollback | rollback_icmr | pre_rollback | default_rollback | app_rollback | kernel_rollback )
             input_vaild="TRUE"
        ;;
        *)
             input_vaild="FALSE"
        ;;
    esac
    if [ "${4}" == "" -o "${6}" == "" ]
    then
        input_vaild="FALSE"
    fi
    if [ "${input_vaild}" == "FALSE" ]
    then
        echo "bash ${SELF_FILE} -action rollback/rollback_icmr -pkgname NCEV100R019C00*_OMP_linux_x64.zip -taskid c6bfc6d0a769c2b319f9"
        LOG_PRINT "ERROR" "The input param is invalid."
        return 1
    fi
}

##################################################
# 校验omp包是否存在
# 同时拷贝omp包到/opt/upgrade工作路径
# 前台已做完整性校验，不再重复校验
##################################################
function copy_omp_workpath()
{
    if [ ${NEW_TASK} != "TRUE" -a -d "${OMP_PATH}" ]
    then
        return 0
    fi
    # 新建升级任务重新解压
    rm -rf ${OMP_PATH}

    unzip_omp_package || return $?
    return 0
}

##################################################
# 解压OMP软件包
##################################################
function unzip_omp_package()
{
    if [ -d "${OMP_PATH}" ]
    then
        return 0
    fi
    if [ -f "/opt/pub/software/mgr-installdisk/${PKGNAME}" ]
    then
        unzip /opt/pub/software/mgr-installdisk/${PKGNAME} -d ${OMP_PATH} >>${LOG_FILE} 2>&1
        if [ $? -ne 0 ]
        then
            TASK_LOG "ERROR" "Failed to unzip (${PKGNAME})."
            # 解压失败后清理目录(解决空间不足导致的文件残留)
            rm -rf ${OMP_PATH}
            fresh_fail
            return 1
        fi
    else
        TASK_LOG "ERROR" "The software package (${PKGNAME}) is not uploaded. "
        fresh_fail
        return 1
    fi
}

#####################################################
# 将OMP日志文件打包
#####################################################
function pack_omp_log()
{
    # 回滚完成删除OMP包、日志打包
    local omp_log_path="${OMP_PATH}/logs"
    local easysuite_log_path="/opt/oss/log/manager/easysuite_upgrade/"
    local task_id_log_path="${easysuite_log_path}/${TASKID}.zip"
    if [ -d "${omp_log_path}" -a ! -f "${task_id_log_path}" ]
    then
        zip -j ${task_id_log_path} ${omp_log_path}/*
        chmod 440 ${task_id_log_path}
    fi
    rm -rf ${OMP_PATH} >/dev/null 2>&1
}

##########################################################
# 等待OMP回滚完成后管理面系统监控状态正常(等待180秒退出)
##########################################################
function wait_product_and_exit()
{
    for percent in {0..100}
    do
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [$$] Wait product on node:[0] ----------- ${percent}%" >>${TASK_LOG_FILE}
        sleep 1.8
    done
    fresh_status "finish"
}

##################################################
# 防止文件句柄被大量占用，touch失败
##################################################
function create_result_flag()
{
    local flag_file=$1
    TASK_LOG "INFO" "Start to create the ${flag_file} file."
    for i in $(seq 10)
    do
        [ ! -f "${flag_file}" ] && touch "${flag_file}"
        if [ -f "${flag_file}" ]
        then
            TASK_LOG "INFO" "The ${flag_file} file is created successfully."
            return 0
        fi
        sleep 1
    done
    TASK_LOG "ERROR" "Failed to create the ${flag_file} file."
    return 1
}


##################################################
# 刷新第一阶段Easysuite进度信息
##################################################
function fresh_step1()
{
    local success_flag=$1
    local failed_flag=$2
    CHGRP_FLAG="TRUE"
    # 回滚顺序 UpgradePortConfig->UpgradeUserConfig->UpgradeUniEP
    while true
    do
        change_group
        # 1、失败或成功标志文件存在，进入判断逻辑
        if [ -f "${success_flag}" -o -f "${failed_flag}" ]
        then
            break
        fi
        # 考虑模拟进度
        sleep 1
    done
}

##################################################
# 如果升级前无sopgroup用户组，回滚完ICMR之后
# 切换升级工作目录属组为ossgroup
##################################################
function change_group()
{
    if [ ${CHGRP_FLAG} == 'FALSE' ]
    then
        return 0
    fi
    id ${SOP_USER} 2>/dev/null
    # 无sopuser用户
    if [ $? -ne 0 ]
    then
        find ${UPGRADE_PATH} -type d -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs chown ${OSS_USER}:
        find ${UPGRADE_PATH} -type f -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs chown ${OSS_USER}:
        CHGRP_FLAG="FALSE"
        return 0
    fi
    # 有sopuser用户
    path="${UPGRADE_PATH} ${UPGRADE_PATH}/easysuite_upgrade ${UPGRADE_PATH}/easysuite_upgrade/scripts ${UPGRADE_PATH}/easysuite_upgrade/databasemigrate ${UPGRADE_PATH}/easysuite_upgrade/upgrade_frame_scripts"
    for one_path in ${path}
    do
        path_id=$(stat -c %g "${one_path}")
        if [ ${SOP_GROUP_ID} == ${path_id} ]
        then
            setfacl -m u:${SOP_USER}:r-x "${one_path}"
            chown ${OSS_USER}: "${one_path}"
        fi
    done
    find ${UPGRADE_PATH}/easysuite_upgrade/taskmgr -type f -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs setfacl -m u:${SOP_USER}:r--
    find ${UPGRADE_PATH}/easysuite_upgrade/taskmgr -type d -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs setfacl -m u:${SOP_USER}:r-x

    find ${UPGRADE_PATH} -type f -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs chown ${OSS_USER}:
    find ${UPGRADE_PATH} -type d -user ${OSS_USER} -group ${SOP_GROUP_ID} | xargs chown ${OSS_USER}:
    CHGRP_FLAG="FALSE"
    return 0
}

##################################################
# 刷新第二阶段Easysuite进度信息
##################################################
function fresh_step2()
{
    local success_flag=$1
    local failed_flag=$2
    # 回滚顺序 configICMR->Osconfig->Sudoconfig
    while true
    do
        # 1、失败或成功标志文件存在，进入判断逻辑
        if [ -f "${success_flag}" -o -f "${failed_flag}" ]
        then
            if [ -f "${failed_flag}" ]
            then
                fresh_status "fail"
            elif [ -f "${success_flag}" ]
            then
                pack_omp_log
                wait_product_and_exit
                fresh_status "finish"
            fi
            break
        fi
        # 考虑模拟进度
        sleep 1
    done
}

##################################################
# 回滚portconfig-userconfig-uniep
##################################################
function rollback_step1()
{
    # Case:step1、pre_rollback、app_rollback、kernel_rollback
    local step_action="${1}"

    # 前置
    if [ "${step_action}" == "step1" -o "${step_action}" == "pre_rollback" ]
    then
        # 解压omp
        copy_omp_workpath || return $?
    fi
    unzip_omp_package || return $?
    UPGRADE_FLAG="$(bash ${ROLLBACK_SCRIPT} -action query_flag)"

    local TAIL=$(date +%Y%m%d%H%M%S)
    local success_flag="${OMP_PATH}/${step_action}_success_${TAIL}.flag"
    local fail_flag="${OMP_PATH}/${step_action}_failed_${TAIL}.flag"

    bash ${ROLLBACK_SCRIPT} -action "${step_action}" >>${TASK_LOG_FILE} 2>&1 && create_result_flag ${success_flag} || create_result_flag ${fail_flag}

    # 结束时
    step_status="fail"
    if [ -f "${success_flag}" ]
    then
        step_status="success"
    fi

    if [ "${step_status}" == "success" ]
    then
        # step1 | pre_rollback | default_rollback | app_rollback | kernel_rollback
        if [ "${step_action}" == "step1" -o "${step_action}" == "default_rollback" -o "${step_action}" == "kernel_rollback" ]
        then
            # 所有步骤的最后的后置操作OK,打包日志文件
            pack_omp_log
        fi
        fresh_status "finish"
    elif [ "${step_status}" == "fail" ]
    then
        fresh_status "fail"
    fi

    # 只刷新结束状态
    fresh_progress "100"
}

##################################################
# 回滚icmr-osconfig-sudoconfig
##################################################
function rollback_step2()
{
    copy_omp_workpath || return $?
    UPGRADE_FLAG="$(bash ${ROLLBACK_SCRIPT} -action query_flag)"

    if [ ! -f ${UPGRADE_FLAG} ]
    then
        TASK_LOG "INFO" "Rollback UniEP Successfully."
        fresh_finish
        return 0
    fi

    local TAIL=$(date +%Y%m%d%H%M%S)
    local success_flag="${OMP_PATH}/step2_success_${TAIL}.flag"
    local fail_flag="${OMP_PATH}/step2_failed_${TAIL}.flag"

    bash ${ROLLBACK_SCRIPT} -action step2 >>${TASK_LOG_FILE} 2>&1 && create_result_flag ${success_flag} || create_result_flag ${fail_flag} &
    # 再启一个线程进程进度刷新
    fresh_step2 "${success_flag}" "${fail_flag}" &
    wait
    # 只刷新结束状态
    fresh_progress "100"
}

##################################################
# 脚本主入口
##################################################
function main()
{
    # 加载公共方法
    . ${SCRIPT_PATH}/../common.sh
    LOG "INFO" "Start to init log."
    # 校验入参，初始化参数
    check_input "$@" || return 1
    ACTION="${2}"
    PKGNAME="${4}"
    TASKID="${6}"

    init_variable
    
    init_taskmgr ${TASKID}

    # 根据入参执行各自方法
    case ${ACTION} in
    "step2")
        rollback_step2
        return $?
    ;;
    *)
        rollback_step1 "${ACTION}"
        return $?
    ;;
    esac
}

main "$@"
exit $?