# encoding=utf-8
"""
功 能：备份管理面
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2021-02-03 14:39 创建
"""
import sys
import os
import json
import time
import re
from datetime import datetime
from commonlog import Logger
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr
from pre_backup_uniep import PreBackupUniep

# 日志级别枚举
INFO = "INFO"
ERROR = "ERROR"
LOGGER = Logger().getinstance(sys.argv[0])

# 备份管理面接口
BACKUP_UNIEP_URL = "/rest/plat/brmgr/v1/mgmtbackup/systembackuptask"

# 备份管理面状态值
FAILURE_STATE = ["FAILURE", "PARTIAL_SUCCESS", "TERMINATING", "TERMINATED"]
RUNNING_STATE = ["INIT", "RUNNING", "WAITING"]

# 查询进度状态枚举
FINISH = 0
RUNNING = 1
FAULT = 2

# 备份文件
BACKUP_FILE = '/opt/upgrade/backup/backup_timestamp.properties'


class BackupUniep:
    """
    升级框架备份管理面
    """

    def __init__(self, args):
        LOGGER.info("Init backup Uniep.")
        self.task_id = args.get("task_id", "")
        e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
        self.path = os.path.join(e_taskmgr_path, self.task_id)
        self.backup_uniep_collection = args.get("backup_uniep_collection", "")
        self.product_name = args.get("product_name", "")
        self.upgrade_path = args.get("upgrade_path", "")
        self.rest = Unieptaskmgr()
        self.taskmgr = Taskmgrutil()
        self.taskmgr.init_e_taskmgr(self.path)
        self.msg_list = []

    def record_log(self, msg, level=""):
        """
        更新任务process、status、和日志
        :param msg:
        :param log_type:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        str_msg = "[%s] [%s] %s | %s" % (now_time, os.getpid(), level, msg)
        self.msg_list.append(str_msg)
        self.taskmgr.set_e_taskmsg(self.path, "\n".join(self.msg_list))
        if level == ERROR:
            self.taskmgr.set_e_taskstatus(self.path, "error")
            self.taskmgr.set_e_taskprogress(self.path, "100")
        return True

    def record_timestap(self, url):
        """
        记录备份时间戳
        :param url:查询接口
        :return:
        """
        self.record_log("Start to write backup timestamp.", INFO)
        LOGGER.info("Start to write backup timestamp.")
        try:
            status, result = self.rest.send_get_request(url)
            if status:
                result = json.loads(result.decode())[0]
                details = result.get("details")
                timestamp = re.findall(r"\d+", details)[0]
                timestamp_str = "backup_single_uniep_%s_%s: %s" % (self.product_name, self.upgrade_path, timestamp)
                cmd = "echo %s >> %s" % (timestamp_str, BACKUP_FILE)
                ret_code = os.system(cmd)
                if ret_code == 0:
                    LOGGER.info("Finished to run %s" % cmd)
                    self.record_log("Wirte backup timestamp file success.",
                                    INFO)
                    self.taskmgr.set_e_taskstatus(self.path, "finish")
                    self.taskmgr.set_e_taskprogress(self.path, "100")
                    return True
                LOGGER.error("Failed to run %s" % cmd)
                self.record_log("Wirte backup timestamp file failed.",
                                ERROR)
                return False
            self.record_log("Query backup timestamp failed.", ERROR)
            LOGGER.error("Query backup timestamp failed.")
            return False
        except (TypeError, ValueError) as e_msg:
            self.record_log(
                "Wirte backup timestamp file failed. Exception:{}".format(
                    e_msg),
                ERROR)
            LOGGER.error(
                "Wirte backup timestamp file failed. Exception:{}".format(
                    e_msg))
            return False

    def check_progress(self, url):
        """
        检查备份进度
        :param url:查询接口
        :return:
        """
        try:
            status, result = self.rest.send_get_request(url)
            if status:
                result = json.loads(result.decode())[0]
                execute_result = result.get("currentState")
                progress = result.get("progress", "0")
                if not execute_result or execute_result in FAILURE_STATE:
                    return FAULT, progress
                if execute_result in RUNNING_STATE:
                    return RUNNING, progress
                return FINISH, progress
            return FAULT, "0"
        except (TypeError, ValueError) as e_msg:
            self.record_log(
                "Failed to query task msg. Exception:{}".format(e_msg),
                ERROR)
            LOGGER.error("Failed to query task msg. Exception:{}".format(e_msg))
            return FAULT, "0"

    def query_progress(self, url):
        """
        查询备份进度
        :param url:查询接口
        :return:
        """
        self.record_log("Start to query backup task progress.", INFO)
        LOGGER.info("Start to query backup task progress.")
        max_wait_seconds = 14400
        delay_seconds = 10
        end_time = int(time.time()) + max_wait_seconds
        status = RUNNING
        while status != FINISH:
            status, progress = self.check_progress(url)
            if status == FAULT:
                self.record_log("Failed to query backup task progress.", ERROR)
                LOGGER.error("Failed to query backup task progress.")
                return False
            self.record_log(
                "Check backup uniep process is running, execution progress is %s%%" % progress,
                INFO)
            if status == FINISH:
                if self.record_timestap(url):
                    self.record_log("Finish to execute backup uniep task.",
                                    INFO)
                    LOGGER.info(
                        "Finish to execute backup uniep task. Task_id:{}.".format(
                            self.task_id))
                    return True
                self.record_log("Failed to execute backup uniep task.",
                                ERROR)
                LOGGER.info("Failed to execute backup uniep task.")
                return False
            time.sleep(delay_seconds)
            if int(time.time()) >= end_time:
                self.record_log(
                    "Check backup uniep progress timeout, waitting for %d "
                    "seconds" % max_wait_seconds, ERROR)
                LOGGER.error("Check backup uniep progress timeout.")
                return False
        return True

    def parse_to_backup_uniep_collection(self):
        backup_uniep_list = str(self.backup_uniep_collection).split(",")
        backup_uniep_collection = []
        if not backup_uniep_list or self.backup_uniep_collection == "None":
            return ["MGMT_COMBINE_DB_APP"]
        if "MGMT_COMBINE_DB_APP" in backup_uniep_list:
            backup_uniep_collection.append("MGMT_COMBINE_DB_APP")
        if "MGMT_DB_DATA" in backup_uniep_list:
            backup_uniep_collection.append("MGMT_DB_DATA")
        return backup_uniep_collection

    def backup_uniep(self):
        """
        备份管理面
        :return:
        """
        self.record_log("Start to backup uniep.", INFO)
        LOGGER.info("Start to backup uniep. backup_uniep_collection {0}".format(self.backup_uniep_collection))
        backup_uniep_collection = self.parse_to_backup_uniep_collection()
        LOGGER.info("Start to backup uniep. backup_uniep_collection {0}".format(backup_uniep_collection))
        param = dict(
            isDeleteExceedingData="no",
            isCheckRepetitiveTasks="yes",
            isBackupPubSoftware="no",
            ignoreNormalize="yes",
            backupCollection=backup_uniep_collection
        )
        try:
            status, result = self.rest.send_post_request(BACKUP_UNIEP_URL,
                                                         param)
            if status:
                result = json.loads(result.decode())
                task_url = result.get("url", "")
                if task_url:
                    return self.query_progress(task_url)
                task_result = result.get("result", "")
                task_entity = result.get("entity", None)
                if task_result == "ERROR" and task_entity is not None:
                    self.record_log(
                        "Failed to create a backup task. Error msg:{}".format(
                            task_entity.get("details", "None")), ERROR)
                    LOGGER.error(
                        "Failed to create a backup task. Error msg:{}".format(
                            task_entity.get("details", "None")))
                else:
                    self.record_log("Failed to backup uniep.", ERROR)
                    LOGGER.error("Failed to backup uniep.")
                return False
            self.record_log("Failed to backup uniep.", ERROR)
            LOGGER.error("Failed to backup uniep.")
            return False
        except (TypeError, ValueError) as e_msg:
            self.record_log("Failed to backup uniep. Exception:{}".format(e_msg),
                            ERROR)
            LOGGER.error("Failed to backup uniep. Exception:{}".format(e_msg),
                         ERROR)
            return False

    def check_finish(self):
        """
        根据task_id检查上次任务是否成功
        :return:
        """
        cmd = "grep %s %s" % (self.task_id, LOGGER.logfile)
        ret_code = os.system(cmd)
        if ret_code == 0:
            self.record_log("Finish to execute backup uniep task.", INFO)
            self.taskmgr.set_e_taskstatus(self.path, "finish")
            self.taskmgr.set_e_taskprogress(self.path, "100")
            return True
        return False


def main(argv):
    """
    根据id、产品名称、升级路径启动备份管理面
    :return:
    """
    task_id = argv[1]
    backup_uniep_collection = argv[2]
    product_name = argv[3]
    upgrade_path = argv[4]
    args = {}
    args.update({"task_id": task_id})
    args.update({"backup_uniep_collection": backup_uniep_collection})
    args.update({"product_name": product_name})
    args.update({"upgrade_path": upgrade_path})

    backup_function = BackupUniep(args)
    if not backup_function.check_finish():
        PreBackupUniep.modify_backup_config(["/opt/oss/easysuite"])
        result = backup_function.backup_uniep()
        PreBackupUniep.restore_backup_config()
        return result
    return True


if __name__ == '__main__':
    main(sys.argv)
