# -*- coding:utf-8 -*-
"""
功 能：记录日志
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import copy
import os
import sys
import json
import time
import pathlib

from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr
from upgrade_error_code import ErrorCode

MODIFY_VER_ERROR = "OperateProductUtil.modify_product_version,result:False"

LOGGER = Logger().getinstance(sys.argv[0])
SEND_URL_TO_MANAGER = Unieptaskmgr()
config_dir = "/opt/oss/log/manager/easysuite_upgrade_config"
CLOUDSOP_ERROR_CODE_PATH = "/opt/upgrade/easysuite_upgrade/cloudsop_error_path"

# ipmc命令行错误码(0和47之外是前置检查失败)
DEPLOY_SUCCESS = 0
DEPLOY_FAIL = 47


class OperateProductUtil:
    """
    功能说明:产品运维面ipmc_tool方式升级
    """
    @staticmethod
    def build_upgrade_data(param_dict, target_version, packages):
        """
        功能说明:生成升级运维面参数信息
        参数:param_dict, target_version, packages
        返回:ret_data
        """
        # productname productversion featuregrouplist
        # product_*.json中为productname,ipmc_tool用productName识别。两者都为必填参数;真实有效,
        ret_data = {'productname': param_dict.get("productName")}
        ret_data.update({'productName': param_dict.get("productName")})
        ret_data.update({'productVersion': target_version})
        ret_data.update({'senario': param_dict.get("senario")})
        ret_data.update({'startupPolicy': param_dict.get("upgrade_start_policy")})
        ret_data.update({'parallelInstallation': param_dict.get("parallelInstallation", "False")})
        LOGGER.info("upgrade_start_policy: %s" % param_dict.get("upgrade_start_policy"))
        # 构造features
        __features = []
        for one_feature in param_dict.get("features"):
            __features.append({"name": one_feature})
        ret_data.update(
            {'featuregrouplist': [{"name": param_dict.get("senario", 'all'), "features": __features}]})

        # 构造packages
        __packages = []
        for one_pkg in packages:
            __packages.append({"name": one_pkg.get("name"),
                               "version": one_pkg.get("version")})
        ret_data.update({'packages': __packages})

        # 构造installparam
        __installparam = {}
        for one_param in param_dict.get("parameters"):
            __installparam.update({one_param.get("name"): one_param.get("value")})
        ret_data.update({'installparam': __installparam})
        ret_data.update({'deployType': 'UPGRADE'})

        # 指定运维面升级错误码输出路径
        product = param_dict.get("productName", "")
        error_code_file = os.path.join(CLOUDSOP_ERROR_CODE_PATH, f"{product}_error_code.json")
        OperateProductUtil.create_error_code_file(error_code_file)
        ret_data.update({'errmsgfile': error_code_file})

        return ret_data

    @staticmethod
    def create_error_code_file(error_code_file):
        """
        初始化平台错误码文件路径
        """
        try:
            if not os.path.isdir(CLOUDSOP_ERROR_CODE_PATH):
                os.makedirs(CLOUDSOP_ERROR_CODE_PATH)
            if os.path.exists(error_code_file):
                os.remove(error_code_file)
            pathlib.Path(error_code_file).touch()
        except Exception as e:
            LOGGER.error(f"Failed to create file: {error_code_file}, exception: {e}")

    @staticmethod
    def filer_senstive_character(input_dict: dict):
        """
        功能说明:过滤字典中的敏感数据
        参数: input_dict
        """
        for key, value in input_dict.items():
            if isinstance(value, dict):
                OperateProductUtil.filer_senstive_character(value)
            if isinstance(value, str) and value.startswith("0000000100000001"):
                input_dict.update({key: "******"})

    @staticmethod
    def operate_product_and_fresh_version(cmd, data, task_path):
        """
        功能说明:下发ipmc_tool命令并且更新产品版本号
        :param cmd:
        :param data:
        :param task_path:
        :return:
        """
        result = os.system(cmd)
        if result != 0:
            LOGGER.error("upgrade_cmd:%s,result:%s" % (cmd, result))
            # 生成错误码
            error_code_file = data.get("errmsgfile", "")
            ErrorCode.convert_cloudsop_error_codes(task_path, error_code_file)
            return False, result
        if data.get('installparam'):
            del data['installparam']
        LOGGER.info("upgrade_cmd:%s,result:%s" % (cmd, result))
        upgrade_data_for_log = copy.deepcopy(data)
        OperateProductUtil.filer_senstive_character(upgrade_data_for_log)
        LOGGER.info("upgrade_data:%s" % upgrade_data_for_log)
        product_name = data.get('productname')
        product_version = data.get('productVersion')
        if not OperateProductUtil.modify_product_version(product_name, product_version):
            LOGGER.error(MODIFY_VER_ERROR)
            return False, result
        return True, result

    @staticmethod
    def ipmc_tool_upgrade_product(upgrade_data, task_detail, work_path):
        """
        功能说明:升级运维面
        参数: upgrade_data
        返回: True/False
        """
        # 构造input.json
        # productname productversion featuregrouplist
        # packages installparam
        # 部署任务创建标识
        is_deploy_task_created = False
        result, __ipmc_tool = OperateProductUtil.get_ipmc_tool()
        if not result:
            return False, is_deploy_task_created

        input_json_file = os.path.join(work_path, 'input.json')
        with os.fdopen(os.open(input_json_file,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o600), "w", encoding="utf-8") as w_stream:
            w_stream.write(json.dumps(upgrade_data))
        upgrade_cmd = "unbuffer %s -cmd productmgr -o deployv2 -action UPGRADE -input " \
                      "%s >> %s 2>&1" % (__ipmc_tool, input_json_file, task_detail)
        task_path = os.path.dirname(task_detail)
        status, return_code = OperateProductUtil.operate_product_and_fresh_version(upgrade_cmd, upgrade_data, task_path)
        if return_code == DEPLOY_SUCCESS or (return_code >> 8) == DEPLOY_FAIL:
            is_deploy_task_created = True
        return status, is_deploy_task_created

    @staticmethod
    def ipmc_tool_upgrade_product_firstphase(upgrade_data, task_detail, work_path):
        """
        功能说明:升级运维面第一段
        参数: upgrade_data
        返回: True/False
        """
        # productname productversion featuregrouplist
        # packages installparam
        # -input upgrade_param.json所在path -mode deploy
        # 部署任务创建标识
        is_deploy_task_created = False
        result, __ipmc_tool = OperateProductUtil.get_ipmc_tool()
        if not result:
            return False, is_deploy_task_created

        upgrade_param_json_file = os.path.join(work_path, 'upgrade_param.json')
        with os.fdopen(os.open(upgrade_param_json_file,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o600), "w", encoding="utf-8") as w_stream:
            w_stream.write(json.dumps(upgrade_data))
        upgrade_cmd = "unbuffer %s -cmd productmgr -o deployv2 -action UPGRADE -input " \
                      "%s -mode deploy>> %s 2>&1" % (__ipmc_tool, upgrade_param_json_file,
                                                     task_detail)
        task_path = os.path.dirname(task_detail)
        status, return_code = OperateProductUtil.operate_product_and_fresh_version(upgrade_cmd, upgrade_data, task_path)
        if return_code == DEPLOY_SUCCESS or (return_code >> 8) == DEPLOY_FAIL:
            is_deploy_task_created = True
        return status, is_deploy_task_created

    @staticmethod
    def ipmc_tool_upgrade_product_secondphase(task_detail, work_path):
        """
        功能说明:升级运维面第二段
        参数: upgrade_data
        返回: True/False
        """
        # productname productversion featuregrouplist
        # packages installparam
        # 第二段：ipmc_tool -cmd productmgr -o deployv2 -action UPGRADE
        # -input upgrade_param.json所在path -mode active
        result, __ipmc_tool = OperateProductUtil.get_ipmc_tool()
        if not result:
            return False
        upgrade_param_json_file = os.path.join(work_path, 'upgrade_param.json')
        upgrade_cmd = "unbuffer %s -cmd productmgr -o deployv2 -action UPGRADE -input " \
                      "%s -mode active>> %s 2>&1" % (__ipmc_tool, upgrade_param_json_file,
                                                     task_detail)
        result = os.system(upgrade_cmd)
        if result != 0:
            LOGGER.error("upgrade_cmd:%s,result:%s" % (upgrade_cmd, result))
            # 生成错误码
            task_path = os.path.dirname(task_detail)
            with open(upgrade_param_json_file, "r") as f:
                upgrade_params = json.load(f)
            error_code_file = upgrade_params.get("errmsgfile", "")
            ErrorCode.convert_cloudsop_error_codes(task_path, error_code_file)
            return False
        LOGGER.info("upgrade_cmd:%s,result:%s" % (upgrade_cmd, result))
        return True

    @staticmethod
    def get_ipmc_tool():
        """
        功能说明:获取ipmc_tool工具所在位置
        返回: True/False,ipmc_tool_cmd
        """
        default_ipmc_path = "/opt/oss/manager/agent/DeployAgent"
        env_ipmc_path = os.getenv("IPMC_AGENT_ROOT")
        if not env_ipmc_path:
            LOGGER.info("Use default_ipmc_path:%s" % default_ipmc_path)
            env_ipmc_path = default_ipmc_path
        if not os.path.isdir(env_ipmc_path):
            LOGGER.error("env_ipmc_path (%s) is not exist." % env_ipmc_path)
            return False, ""
        ipmc_tool_cmd = os.path.join(default_ipmc_path, 'bin', 'ipmc_tool')
        return True, ipmc_tool_cmd

    @staticmethod
    def ipmc_tool_rollback_product(deploy_id, product_name, task_detail, work_path,
                                   target_version, **kwargs):
        """
        功能说明:回滚运维面
        参数:deploy_id 、 product_name 、task_detail 、 work_path、target_version
        返回:True/False
        """
        result, __ipmc_tool = OperateProductUtil.get_ipmc_tool()
        if not result:
            return False

        input_json_file = os.path.join(work_path, 'input.json')
        # 是否清理数据库
        clean_db = kwargs.get("is_clean_db_while_rollback", "true")
        # 是否启动产品服务
        rollback_start_policy = kwargs.get("rollback_start_policy")
        # 是否开启极速回滚
        parallel_installation = kwargs.get("parallel_installation", "False")

        rollback_data = {"deployId": deploy_id, "productName": product_name,
                         "startupPolicy": rollback_start_policy, "clearRedundantDB": clean_db,
                         "parallelInstallation": parallel_installation}
        with os.fdopen(os.open(input_json_file,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o600), "w", encoding="utf-8") as w_stream:
            w_stream.write(json.dumps(rollback_data))
        rollback_cmd = "unbuffer %s -cmd productmgr -o deployv2 -action ROLLBACK -input " \
                       "%s >> %s 2>&1" % (__ipmc_tool, input_json_file, task_detail)
        result = os.system(rollback_cmd)
        if result != 0:
            LOGGER.error("rollback_cmd:%s,result:%s" % (rollback_cmd, result))
            return False
        LOGGER.info("rollback_cmd:%s,result:%s" % (rollback_cmd, result))
        if not OperateProductUtil.modify_product_version(product_name, target_version):
            LOGGER.error(MODIFY_VER_ERROR)
            return False
        return True

    @staticmethod
    def compare_packages(src_list, des_list):
        """
        功能说明:比较软件列表
        参数:src_list, des_list
        """
        if not isinstance(src_list, list) or not isinstance(des_list, list):
            return False
        if len(src_list) != len(des_list):
            return False
        for one_obj in src_list:
            if one_obj not in des_list:
                return False
        return True

    @staticmethod
    def save_pre_upgrade_config(deploy_data, plan_data_path):
        """功能说明:
        功能说明:保存升级前部署数据
        参数:deploy_data、plan_data_path
        """
        upgrade_deploy_task = os.path.join(plan_data_path, "pre_upgrade_config.json")
        if not os.path.isdir(plan_data_path):
            os.system("mkdir -p %s" % plan_data_path)
        deploy_save_data = {}
        for key in ("productVersion", "packages"):
            if deploy_data.get(key):
                deploy_save_data.update({key: deploy_data.get(key)})
        with os.fdopen(os.open(upgrade_deploy_task,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o600), 'w', encoding='utf-8') as file:
            file.write(json.dumps(deploy_save_data))
        return True

    @staticmethod
    def get_product_plan_version(product_name):
        """
        功能说明:查询对应版本规划数据中的产品版本号
        参数:product_name
        """
        product_info_url = "/rest/productconfiguration/v1/products/services?productname=%s" % \
                           product_name
        product_version = ''
        status = False
        response = [{}]
        for _ in range(0, 30):
            status, response = SEND_URL_TO_MANAGER.send_get_request(product_info_url)
            if status:
                LOGGER.info("Finish to send rest(%s) request to managaer" % product_info_url)
                break
            LOGGER.error("Fail to send rest(%s) request to managaer" % product_info_url)
        if status:
            product_version = json.loads(response)[0].get("productVersion")
        LOGGER.info("get_product_plan_version.product_version (%s)" % product_version)
        return product_version

    @staticmethod
    def is_already_upgrade(current_deploy_data, upgrade_deploy_data):
        """
        功能说明:检查是否已经升级完成
        参数:current_deploy_data、upgrade_deploy_data
        """
        # 检查当前软件包列表是否和升级后软件包列表一致
        condition = True
        product_name = current_deploy_data.get("productName")
        current_version = current_deploy_data.get("productVersion")
        target_version = upgrade_deploy_data.get("target_version")
        product_version = OperateProductUtil.get_product_plan_version(product_name)
        LOGGER.info("product_name:%s current_version:%s target_version:%s" % (product_name,
                                                                              current_version,
                                                                              target_version))
        LOGGER.info("product_version:%s" % product_version)
        current_deploy_pkgs = current_deploy_data.get("packages")
        upgrade_deploy_pkgs = upgrade_deploy_data.get("packages")
        LOGGER.info("current_deploy_pkgs:%s" % current_deploy_pkgs)
        LOGGER.info("upgrade_deploy_pkgs:%s" % upgrade_deploy_pkgs)
        if not OperateProductUtil.compare_packages(current_deploy_pkgs, upgrade_deploy_pkgs):
            LOGGER.info("compare_packages:%s" % condition)
            condition = False

        # 检查部署状态是否为成功
        current_deploy_status = current_deploy_data.get('deploy_status')
        LOGGER.info("current_deploy_status:%s" % current_deploy_status)
        if current_deploy_status != 'DEPLOY_SUCCESS':
            condition = False
        if condition and product_version == target_version \
                and not OperateProductUtil.modify_product_version(product_name, target_version):
            LOGGER.error("OperateProductUtil.modify_product_version,result:False")
        return condition

    @staticmethod
    def is_already_rollback(current_deploy_data, plan_data_path, target_version):
        """
        功能说明:检查是否已经回滚
        参数:current_deploy_data、plan_data_path
        """
        upgrade_deploy_task = os.path.join(plan_data_path, "pre_upgrade_config.json")
        if not os.path.isfile(upgrade_deploy_task):
            # 文件不存在直接校验版本号
            if current_deploy_data.get('productVersion') == target_version:
                return True
            return False

        with open(upgrade_deploy_task, 'r', encoding='utf-8') as r_stream:
            pre_upgrade_deploy_data = json.load(r_stream)

        if not pre_upgrade_deploy_data:
            return False

        condition = OperateProductUtil.check_rollback_finish(current_deploy_data,
                                                             pre_upgrade_deploy_data,
                                                             target_version)
        return condition

    @staticmethod
    def check_rollback_finish(current_deploy_data, pre_upgrade_deploy_data, target_version):
        """
        功能说明:是否已回滚完成，检查软件包是否一致、版本是否一致
        :param current_deploy_data:
        :param pre_upgrade_deploy_data:
        :param target_version:
        :return:
        """
        condition = True
        # 检查当前软件包列表和升级前软件包列表是否相同
        current_deploy_pkgs = current_deploy_data.get("packages")
        pre_upgrade_deploy_pkgs = pre_upgrade_deploy_data.get("packages")
        LOGGER.info("current_deploy_pkgs:%s pre_upgrade_deploy_pkgs:%s" %
                    (current_deploy_pkgs, pre_upgrade_deploy_pkgs))
        if not OperateProductUtil.compare_packages(current_deploy_pkgs, pre_upgrade_deploy_pkgs):
            condition = False
        # 检查回滚后状态是否为成功状态
        current_deploy_status = current_deploy_data.get('deploy_status')
        LOGGER.info("current_deploy_status:%s" % current_deploy_status)
        if current_deploy_status != 'DEPLOY_SUCCESS':
            condition = False
        # 检查回滚后版本和升级前版本是否一致
        product_name = current_deploy_data.get('productName')
        current_deploy_version = current_deploy_data.get('productVersion')
        pre_upgrade_deploy_version = pre_upgrade_deploy_data.get('productVersion')
        LOGGER.info("current_deploy_version:%s" % current_deploy_version)
        LOGGER.info("pre_upgrade_deploy_version:%s" % pre_upgrade_deploy_version)
        if condition and current_deploy_version != pre_upgrade_deploy_version \
                and not OperateProductUtil.modify_product_version(product_name, target_version):
            LOGGER.error("OperateProductUtil.modify_product_version,result:False")
        return condition

    @staticmethod
    def modify_extend_product_version(extend_over_view, version):
        """
        功能说明:刷新拓展信息字段信息
        """
        return_extend_over_views = {}
        try:
            all_over_view_items = []
            __extend_over_view = json.loads(extend_over_view.replace("'", '"'))
            __items = __extend_over_view.get('items')
            for one_items in __items:
                if one_items.get('key').get('en_US') == 'Product version':
                    one_items.get('value').update({'zh_CN': version, 'en_US': version})
                all_over_view_items.append(one_items)
            return_extend_over_views.update({'title': __extend_over_view.get('title'),
                                             'items': all_over_view_items})
        except (IndexError, ValueError, TypeError):
            return extend_over_view
        return json.dumps(return_extend_over_views)

    @staticmethod
    def modify_product_version(product_name, product_version):
        """
        功能说明:刷新产品版本号
        参数:product_name, product_version
        返回:True、False
        """
        # ${CloudSOP-UniEP安装路径}/manager/tools/resmgr/queryproduct.sh
        # ${CloudSOP-UniEP安装路径}/manager/tools/resmgr/modifyproductinfo.sh
        oss_root = os.getenv('OSS_ROOT')
        if not os.path.isdir(oss_root):
            LOGGER.error("OSS_ROOT(%s) is not exit." % oss_root)
            return False
        time_stamp = int(time.time())
        if not os.path.isdir(config_dir):
            os.makedirs(config_dir)
        out_path = "%s/%s_modify_%s_version" % (config_dir, time_stamp, product_name)
        if os.path.isdir(out_path):
            os.system("rm -rf %s" % out_path)
        os.mkdir(out_path)
        os.chmod(out_path, 0o700)
        product_json = os.path.join(out_path, "product_%s.json" % product_name)
        query_product_cmd = "bash %s/tools/resmgr/queryproduct.sh -pn %s -output %s" \
                            % (oss_root, product_name, out_path)
        modify_product_cmd = "bash %s/tools/resmgr/modifyproductinfo.sh -input %s" \
                             % (oss_root, product_json)
        result = os.system(query_product_cmd)
        if result != 0:
            LOGGER.error("query_product_cmd:%s,result:%s" % (query_product_cmd, result))
            return False
        if not os.path.isfile(product_json):
            LOGGER.error("product_json:%s is not exit" % product_json)
            return False
        with open(product_json, 'r', encoding='utf-8') as r_stream:
            product_data = json.load(r_stream)
            product_data.get('productext').update({'product_version': product_version})
            extend_over_view = product_data.get('productext').get('extendOverView')
            if extend_over_view:
                new_over_view = OperateProductUtil.modify_extend_product_version(extend_over_view,
                                                                                 product_version)
                product_data.get('productext').update({'extendOverView': new_over_view})
        with os.fdopen(os.open(product_json,
                               os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                               mode=0o600), 'w', encoding='utf-8') as w_stream:
            w_stream.write(json.dumps(product_data))
        result = os.system(modify_product_cmd)
        if result != 0:
            LOGGER.error("modify_product_cmd:%s,result:%s" % (modify_product_cmd, result))
            return False
        if os.path.isdir(out_path):
            LOGGER.info("clear out_path:%s" % out_path)
            os.system("rm -rf %s" % out_path)
        return True

    @staticmethod
    def is_path_valid(path_str):
        if path_str == "" or (not isinstance(path_str, str)):
            return False
        illegal_str_list = ["\\", "/", " ", ".."]
        for illegal_str in illegal_str_list:
            if illegal_str in path_str:
                return False
        return True

    @staticmethod
    def format_params(input_params: list):
        """
        功能说明:获取参数
        :param input_params:
        :return:
        """
        params = dict()
        # 第一个元素为脚本文件名，过滤掉
        input_params = input_params[1:]
        params_len = len(input_params)
        # key,value匹配长度必须为偶数
        if (params_len % 2) != 0:
            return {}
        for count in range(0, params_len, 2):
            key = input_params[count][1:]
            value = input_params[count + 1]
            if input_params[count + 1] == "None":
                value = ""
            params.update({key: value})
        return params
