#!/bin/bash
# /opt/目录下执行
##################################################
# 获取脚本工作路径和脚本名
##################################################
SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)

##################################################
# 初始化日志路径和日志文件名
##################################################
OPT_UPGRADE="/opt/upgrade"
UPGRADE_UPLOAD_PATH="/opt/upgrade_upload"
EASYSUITE_UPGRADE="${OPT_UPGRADE}/easysuite_upgrade"

RUN_USER=$(whoami)
OSS_USER=$(id -nu 3001)
OTHER_USER="sopuser"

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    if [ "${RUN_USER}" != "${OSS_USER}" -a "${RUN_USER}" != "${OTHER_USER}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script" 1>&2
        return 1
    fi
}

##################################################
# 清空目录下所有文件
##################################################
function del_files()
{
    local path="${1}"
    if [ -d "${path}" ]
    then
        rm -rf ${path}/* 2>/dev/null
    elif [ -f "${path}" ]
    then
        rm -f ${path} 2>/dev/null
    fi
}

function initial_upgrade_path()
{
    # ROOT_CMD
    # /opt/upgrade刷成ossadm和sopuser可读
    # 升级根目录,给750权限
    # sopuser用户存在,给sopuser权限
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${OPT_UPGRADE}
    chown ${OSS_USER}: ${EASYSUITE_UPGRADE}/taskmgr ${EASYSUITE_UPGRADE}/workpath
    mkdir -p ${EASYSUITE_UPGRADE}/taskmgr  ${EASYSUITE_UPGRADE}/workpath
    chmod 750 ${EASYSUITE_UPGRADE}/taskmgr ${EASYSUITE_UPGRADE}/workpath

    # 特定目录给sopuser读权限
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}/taskmgr

    initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/scripts"

    initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/databasemigrate"

    initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/upgrade_frame_scripts"
}

function initial_scripts_path()
{
    local path_permission="${1}"
    local work_path="${2}"
    # 删除临时文件
    del_files "${work_path}"

    # 创建子目录
    mkdir -p ${work_path}
    chown ${OSS_USER}: ${work_path}
    chmod 750 ${work_path}

    # 目录sopuser要有读权限,子目录不需要
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:${path_permission} ${work_path}
    return 0
}

function unzip_scripts()
{
    if [[ ! -f ${UPGRADE_UPLOAD_PATH}/upload_pkg.zip ]]
    then
        return 0
    fi

    # 控制权限避免无法覆盖
    chmod -R u+w ${EASYSUITE_UPGRADE}/scripts 2>/dev/null
    unzip -qo ${UPGRADE_UPLOAD_PATH}/upload_pkg.zip -d ${EASYSUITE_UPGRADE}/scripts

    # 删除临时文件
    del_files "${UPGRADE_UPLOAD_PATH}/upload_pkg.zip"

    # 目录sopuser要有读权限,子目录不需要
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}/scripts

    # 控制脚本权限
    find ${EASYSUITE_UPGRADE}/scripts -type f -name "*.sh" | xargs chmod 550
}

function migrate_unzip_scrips()
{
    if [[ ! -f ${UPGRADE_UPLOAD_PATH}/migrate_pkg.zip ]]
    then
        return 0
    fi
    # 控制权限避免无法覆盖
    chmod -R u+w ${EASYSUITE_UPGRADE}/databasemigrate 2>/dev/null
    unzip -qo ${UPGRADE_UPLOAD_PATH}/migrate_pkg.zip -d ${EASYSUITE_UPGRADE}/databasemigrate

    # 删除临时文件
    del_files "${UPGRADE_UPLOAD_PATH}/migrate_pkg.zip"

    # 目录sopuser要有读权限,子目录不需要
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}/databasemigrate

    # 控制脚本权限
    find ${EASYSUITE_UPGRADE}/databasemigrate -type f -name "*.sh" | xargs chmod 550
}

function unzip_frame_scripts()
{
    if [[ ! -d ${EASYSUITE_UPGRADE}/scripts/common/NCE-Common/upgrade_frame_scripts ]]
    then
        return 0
    fi

    # 删除临时文件
    del_files "${EASYSUITE_UPGRADE}/upgrade_frame_scripts"

    # 控制权限避免无法覆盖
    chmod -R u+w ${EASYSUITE_UPGRADE}/upgrade_frame_scripts 2>/dev/null
    cp -fp ${EASYSUITE_UPGRADE}/scripts/common/NCE-Common/upgrade_frame_scripts/* ${EASYSUITE_UPGRADE}/upgrade_frame_scripts/

    # 目录sopuser要有读权限,子目录不需要
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}/upgrade_frame_scripts
    id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE}/upgrade_frame_scripts/check_script_pid.sh

    # 控制脚本权限
    find ${EASYSUITE_UPGRADE}/upgrade_frame_scripts -type f -name "*.sh" | xargs chmod 550
    return 0
}

function set_file_permit()
{
    local sensitive_key=""000""000""010""000""000" "AAA""AHg""AAA""AAA""AA""
    file_path="${OPT_UPGRADE}/backup/migrate ${OPT_UPGRADE}/easysuite_upgrade/scripts/ ${OPT_UPGRADE}/easysuite_upgrade/workpath/"
    for one_key in ${sensitive_key}
    do
        for find_path in ${file_path}
        do
            # 满足 rw*r**** 权限文件, *为统配含义
            find ${find_path} -type f -name "*.json" -perm -640 2>/dev/null | xargs grep -l ${one_key} | sort -u | xargs chmod 600 &>/dev/null
        done
    done
    return 0
}

function main()
{
    check_user || return 1
    if [ "${RUN_USER}" == "${OTHER_USER}" ]
    then
        # 历史版本扫描上传的残留sopuser脚本文件
        find ${EASYSUITE_UPGRADE}/scripts -type d -user ${OTHER_USER} 2>/dev/null | xargs rm -rf
        find ${EASYSUITE_UPGRADE}/scripts -type f -user ${OTHER_USER} 2>/dev/null | xargs rm -f
        return 0
    fi
    action=${1}
    case "${action}" in
    "clear_scripts")
        initial_upgrade_path
        return $?
    ;;
    "clear_upgrade_scripts")
        initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/scripts"
        initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/taskmgr"
        initial_scripts_path "r-x" "${EASYSUITE_UPGRADE}/upgrade_frame_scripts"
        id ${OTHER_USER} 2>/dev/null && setfacl -m u:${OTHER_USER}:r-x ${EASYSUITE_UPGRADE} || pwd
        return $?
    ;;
    "unzip_scripts")
        unzip_scripts
        migrate_unzip_scrips
        unzip_frame_scripts
        set_file_permit
        return $?
    ;;
    esac
}

main "$@"
exit $?