#!/bin/bash
#
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
#

###################################################################################
# 执行节点： 各FI节点
# 执行依赖： 无
# 参数：
#     无
# 执行返回值：
#     0 ：执行检查成功
#     非0 ：执行检查失败
###################################################################################

function fn_log_screen() {
    local log_level="$1"
    local l_message="$2"
    echo "[$(date -d today +"%Y-%m-%d %H:%M:%S")][${log_level}] ${l_message}"
}

# 检查ZK目录以及/opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS文件内容，取值范围1-255的整数
function check_fi_zookeeper() {
	#检查有没安装zookeeper
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/ >/dev/null 2>&1; then
		fn_log_screen "INFO" "Zookeeper is not deployed on the node. Skip."
		return 0
	fi
	#检查zookeeper相关目录
	local result=0
	[[ ! -d /srv/BigData/zookeeper ]] && fn_log_screen "ERROR" "Zookeeper:'/srv/BigData/zookeeper' does not exist. Perform the upgrade after the directory exists." && result=1
	#检查zookeeper文件存不存在
    if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS >/dev/null 2>&1; then
		fn_log_screen "ERROR" "Zookeeper:'/opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS' is missing. Copy the file from another node to the current node."
		return 1
    fi
	#检查zookeeper文件内容
	ZOO_MYID=$(cat /opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS | grep "^ZOO_MYID=" | cut -d= -f2)
	if [[ $ZOO_MYID =~ ^[0-9]+$ && $ZOO_MYID -le 255 ]]; then
		fn_log_screen "INFO" "Zookeeper:/opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS: 'ZOO_MYID=$ZOO_MYID'.Is an integer and the range is 1 - 255."
	else
		fn_log_screen "ERROR" "Zookeeper:/opt/huawei/Bigdata/FusionInsight_Current/*quorumpeer/etc/ENV_VARS: 'ZOO_MYID=$ZOO_MYID'.The value of ZOO_MYID is not an integer or ranges from 1 to 255."
		result=1
	fi
	return $result
}

# 检查/opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties文件内容，取值范围1-65535的整数
function check_fi_kafka() {
	#检查有没安装kafka
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*Broker >/dev/null 2>&1;then
		fn_log_screen "INFO" "Kafka is not deployed on the node. Skip."
		return 0
	fi
	#检查kafka文件存不存在
    if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties >/dev/null 2>&1;then
		fn_log_screen "ERROR" "Kafka:'/opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties' does not exist. Copy the file from another node to the current node."
		return 1
	fi
	#检查kafka文件内容
	broker_id=$(cat /opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties | grep -oP 'broker\.id = \K\d+' | cut -d= -f2)
	if [[ $broker_id =~ ^[0-9]+$ && $broker_id -le 65535 ]]; then
		fn_log_screen "INFO" "Kafka:/opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties: 'broker.id=$broker_id'. Is an integer and the range is 1 - 65535."
		return 0
	else
		fn_log_screen "ERROR" "Kafka:/opt/huawei/Bigdata/FusionInsight_Current/*Broker/etc/server.properties: 'broker.id=$broker_id'. Is not an integer. The value is not an integer ranging from 1-65535. Check whether the parameter is valid."
		return 1
	fi
}

# 检查NameNode目录以及/opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml文件内容，自定义约束为1-255的整数
function check_fi_namenode(){
	#检查有没安装namenode
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*NameNode >/dev/null 2>&1;then
		fn_log_screen "INFO" "NameNode is not deployed on the node. Skip."
		return 0
	fi
	#检查namenode相关目录
	local result=0
	[[ ! -d "/srv/BigData/namenode" ]] && fn_log_screen "ERROR" "NameNode:'/srv/BigData/namenode' does not exist. Perform the upgrade after the directory exists." && result=1
	[[ ! -d "/srv/BigData/namenode/current" ]] && fn_log_screen "ERROR" "NameNode:'/srv/BigData/namenode/current' does not exist. Perform the upgrade after the directory exists." && result=1
	#检查namenode文件存不存在
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml >/dev/null 2>&1;then
		fn_log_screen "ERROR" "NameNode:'/opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml' does not exist. Copy the file from another node to the current node."
		return 1
	fi
	#检查namenode文件内容
	namenode_id=$(cat /opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml| grep "<name>dfs.ha.namenode.id</name>" -A 1 | grep "<value>" | sed 's/.*<value>\(.*\)<\/value>.*/\1/')
	if [[ $namenode_id =~ ^[0-9]+$ && $namenode_id -le 255 ]]; then
		fn_log_screen "INFO" "NameNode:/opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml: 'dfs.ha.namenode.id=$namenode_id'. Is an integer and the range is 1 - 255."
	else
		fn_log_screen "ERROR" "NameNode:/opt/huawei/Bigdata/FusionInsight_Current/*NameNode/etc/hdfs-site.xml: 'dfs.ha.namenode.id=$namenode_id'. The value is not an integer ranging from 1-255. Check whether the parameter is valid."
		result=1
	fi
	return $result
}

# 检查/opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml文件内容，取值范围1-65535的整数
function check_fi_resourcemanager() {
	#检查有没安装ResourceManager
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager >/dev/null 2>&1;then
		fn_log_screen "INFO" "ResourceManager is not deployed on the node. Skip."
		return 0
	fi
	#检查ResourceManager文件存不存在
    if ! ls /opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml >/dev/null 2>&1;then
		fn_log_screen "ERROR" "ResourceManager:'/opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml' does not exist. Copy the file from another node to the current node."
		return 1
	fi
	#检查resourcemanager文件内容
	resourcemanager_id=$(cat /opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml | grep "<name>yarn.resourcemanager.ha.id</name>" -A 1 | grep "<value>" | sed 's/.*<value>\(.*\)<\/value>.*/\1/')
	if [[ $resourcemanager_id =~ ^[0-9]+$ && $resourcemanager -le 65535 ]]; then
		fn_log_screen "INFO" "ResourceManager:/opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml: 'yarn.resourcemanager.ha.id=$resourcemanager_id'. Is an integer and the range is 1 - 65535."
		return 0
	else
		fn_log_screen "ERROR" "ResourceManager:/opt/huawei/Bigdata/FusionInsight_Current/*ResourceManager/etc/yarn-site.xml: 'yarn.resourcemanager.ha.id=$resourcemanager_id'. The value is not an integer ranging from 1-65535. Check whether the parameter is valid."
		return 1
	fi
}

# 检查ldap相关目录
function check_fi_ldap(){
	#检查环境有没安装ldap
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*SlapdServer >/dev/null 2>&1;then
		fn_log_screen "INFO" "LDAP is not deployed on the node. Skip."
		return 0
	fi
	ldap_path=/srv/BigData/ldapData
	if ! ls -d $ldap_path >/dev/null 2>&1;then
		fn_log_screen "ERROR" "LDAP:'$ldapData' does not exist. Perform the upgrade after the directory exists."
		return 1
	fi
	return 0
}

#检查journalnode相关目录
function check_fi_journalnode(){
	#检查环境有没安装journalnode
	if ! ls -d /opt/huawei/Bigdata/FusionInsight_Current/*JournalNode >/dev/null 2>&1;then
		fn_log_screen "INFO" "JournalNode is not deployed on the node. Skip."
		return 0
	fi
	#检查journalnode相关目录
	local journalnode_path=/srv/BigData/journalnode
	if ! ls -d $journalnode_path >/dev/null 2>&1;then
		fn_log_screen "ERROR" "Journalnode:'$journalnode_path' does not exist. Perform the upgrade after the directory exists."
		return 1
	fi
	return 0
}

#检查/opt/huawei/Bigdata和/srv/BigData/tmp
function check_fi_sys(){
	local result=0
	[[ ! -d "/opt/huawei/Bigdata" ]] && fn_log_screen "ERROR" "'/opt/huawei/Bigdata' does not exist. Perform the upgrade after the directory exists." && result=1
	[[ ! -d "/srv/BigData/tmp" ]] && fn_log_screen "ERROR" "'/srv/BigData/tmp' does not exist. Perform the upgrade after the directory exists." && result=1
	return $result
}

#检查om目录（管理节点）
function check_fi_om(){
	#判断是否为管理节点
	if ! ls -d /opt/huawei/Bigdata/om-server/  >/dev/null 2>&1;then
		fn_log_screen "INFO" "The node is not a FusionInsight management node. Skip"
		return 0
	fi
	#检查证书文件和配置文件是否存在
	local result=0
	local ca_path="/opt/huawei/Bigdata/om-server/om/security/cert/root/ca.crt"
	[[ ! -f $ca_path ]] && fn_log_screen "ERROR" "'$ca_path' does not exist. Restore the certificate file of the om-server service in the directory and then perform the upgrade." && result=1
	local config_path="/opt/huawei/Bigdata/om-server/OMS/workspace/conf/oms-config.ini"
	[[ ! -f $config_path ]] && fn_log_screen "ERROR" "'${config_path}' does not exists. Restore the configuration file of the om-server service in the directory and then perform the upgrade." && result=1
	return $result
}

# 检查/etc/pam.d/common-password中pam_unix.so是否有多余条目,若不是suse系统则不会打印信息
function check_pam() {
    local osname=$(cat /etc/os-release | grep "^ID=" | awk -F= '{print $2}' | sed 's/"//g')
    [[ ${osname} != "sles" ]] && return 0
    #检查/etc/pam.d/common-password存不存在
    [[ ! -f "/etc/pam.d/common-password" ]] && fn_log_screen "ERROR" "'/etc/pam.d/common-password' does not exist" && return 1
    #检查pam_unix.so是否有多余条目
    local pam_line=$(grep -c "pam_unix.so" /etc/pam.d/common-password 2>/dev/null)
    if [[ ${pam_line} != 1 ]]; then
        fn_log_screen "ERROR" "The 'pam_unix=$pam_line' more than '1' line or does not exist in '/etc/pam.d/common-password'."
        return 1
    fi
    fn_log_screen "INFO" "SUSE:'/etc/pam.d/common-password':'pam_unix.so=$pam_line' is unique."
    return 0
}

function check_system_conf() {
    local result=0
    local hadoop_sys_dir="/sys/fs/cgroup/cpu,cpuacct/hadoop-yarn"
    [[ ! -d $hadoop_sys_dir ]] && fn_log_screen "ERROR" "'$hadoop_sys_dir' does not exist. Reinstall or update the cgroups software package, or re-create the directory, restore the directory, and then perform the upgrade." && result=1
    local sssd_dir="/var/log/sssd"
	[[ ! -d $sssd_dir ]] && fn_log_screen "ERROR" "'$sssd_dir' does not exist.After the sssd directory is manually created, create the sssd.log file in the sssd directory. Restore the sssd.log file, and then perform the upgrade." && result=1

    local system_conf=("/etc/sssd/sssd.conf" "/etc/openldap/ldap.conf" "/etc/hosts" "/etc/cron.allow")
    for conf in "${system_conf[@]}"
    do
        [[ ! -f $conf ]] && fn_log_screen "ERROR" "'$conf' does not exist. Restore the file and then perform the upgrade." && result=1
    done

    # 只在suse环境检测
    local osname=$(cat /etc/os-release | grep "^ID=" | awk -F= '{print $2}' | sed 's/"//g')
    if [[ ${osname} == "sles" ]];then
        local nsswitch_conf_path="/etc/nsswitch.conf"
        local suse_crontab_file="/usr/bin/crontab"
        [[ ! -f $nsswitch_conf_path ]] && fn_log_screen "ERROR" "'$nsswitch_conf_path' does not exist. Restore the file and then perform the upgrade." && result=1

        # check suse crontab
        [[ ! -f $suse_crontab_file ]] && fn_log_screen "ERROR" "'$suse_crontab_file' does not exist. The file is necessary." && result=1
        [[ "$(stat -c '%a' $suse_crontab_file)" != "4755" ]]  && fn_log_screen "ERROR" "'$suse_crontab_file' permission is not 4755.Please run command [ chmod 4755 $suse_crontab_file ] to makesure permission check OK" && result=1
    fi
    return $result
}

function check_fi_files() {
    fn_log_screen "INFO" "Start to check the FI file content."
    local check_result=0
    check_fi_zookeeper || check_result=1
    check_fi_namenode || check_result=1
    check_fi_kafka || check_result=1
    check_fi_resourcemanager || check_result=1
    check_fi_ldap || check_result=1
    check_fi_journalnode || check_result=1
    check_fi_sys || check_result=1
    check_fi_om || check_result=1
    check_pam || check_result=1
    check_system_conf || check_result=1
    return $check_result
}

check_fi_files
exit $?
