#!/bin/bash
###########################################
# 用于判断是否可以在当前OS环境下回退NCE
# 如果当前OS版本是V2R9，且目标版本低于R20C10
# 则脚本返回1，不可回滚NCE。否则可回滚NCE
###########################################

##################################################
# Description:  全局公共变量定义
# Parameters:   无
##################################################
function define_public() {
    # 目录
    LOGPATH="/opt/oss/log/manager/easysuite_upgrade"
    # 文件
    LOGFILE="${LOGPATH}/rollback_check_nce.log"

    # 变量
    EULER_VR=""
}

##################################################
# Description:  创建目录文件
# Parameters:   无
##################################################
function create_public() {
    # 创建目录
    mkdir -p "${LOGPATH}"

    # 创建文件
    touch "${LOGFILE}"
}

##################################################
# Description:  日志打印
#               使用方法: log 日志级别 日志信息 [-nostd]
# Parameters:   日志级别: DEBUG、INFO、WARN、ERROR
#               日志信息: message
#               -nostd: 不显示到标准输出流，仅打印到日志文件
#Example:       log ERROR "message"
#               log INFO "message" -nostd
##################################################
function log() {
    # 接收参数
    local logtype=$1;
    local logmsg=$2;
    local nostd=$3;

    # 当前时间
    local datetime=$(date +'%F %T')

    # 函数调用
    local fncall=""

    # 代码行位置
    local line=""

    # 有错显示函数调用及代码行位置
    if [[ "${logtype}" =~ "ERROR" ]] || [[ "${logtype}" =~ "WARN" ]];then
        fncall="${FUNCNAME[@]/log/}"
        line="[line:$(caller 0 | awk '{print$1}')]"
    fi

    # 日志格式
    logformat="[${datetime}] ${logtype} ${fncall} ${line} ${logmsg}"

    # 日志输出
    [ "${nostd}" = "-nostd" ] && echo -e "${logformat}" >> "${LOGFILE}" || echo -e "${logformat}" | tee -a "${LOGFILE}"
}

##################################################
# Description:  Euler版本判断(V200R007/V200R008)
# Parameters:   无
##################################################
function check_euler_version() {
    log INFO "\n********** Start to execute the check_euler_version() **********\n" -nostd

    EULER_VR=$(uname -r | awk -F 'eulerosv2r' '{print $2}' | awk -F '.' '{print $1}')
    if [ ! -n "${EULER_VR}" ];then
        log ERROR "FATAL:Euler Version does not match"
        return 1
    fi
    log INFO "Euler Version is v2r${EULER_VR}" -nostd

    [ "${EULER_VR}" ] && return 0
}

##################################################
# Description:  检查版本是否需要回滚
# Parameters:   无
##################################################
function check_version_older_than() {
    log INFO "\n********** Start to execute the check_version_older_than() **********\n" -nostd

    local currentEulerVersion="${EULER_VR}"
    local targetNCEVersion=$(echo "${SRC_VERSION}" |grep -P "R[0-9]{1,3}C[0-9]{1,3}" -o)

    log INFO "current Euler Version is v2r${currentEulerVersion}, target NCE Version is ${targetNCEVersion}" -nostd

    if [[ "${targetNCEVersion}" < "R019C00" || "${currentEulerVersion}" -lt "7" ]]; then
        log ERROR "Error, The target version does not support rollback."
        return 1
    fi

    if [[ "${targetNCEVersion}" < "R020C10" && "${currentEulerVersion}" -gt "8" ]]; then
        log ERROR "Error, The current OS version does not support NCE rollback, please roll back the OS first."
        return 1
    fi

    if [[ "${targetNCEVersion}" < "R021C10" && "${currentEulerVersion}" -gt "9" ]]; then
        log ERROR "Error, The current OS version does not support NCE rollback, please roll back the OS first."
        return 1
    fi

    if [[ "${targetNCEVersion}" < "R023C00" && "${currentEulerVersion}" -gt "10" ]]; then
        log ERROR "Error, The current OS version does not support NCE rollback, please roll back the OS first."
        return 1
    fi

    log INFO "Version matching, supporting rollback."
    return 0
}

##################################################
# Description:  解析出所需要的参数
# Parameters:   INPUTPATH 入参保存的json文件
##################################################
function get_upgrade_data()
{
    log INFO "\n********** Start to execute the get_upgrade_data() **********\n" -nostd
    INPUTPATH="${2}"
    if [ -z "${INPUTPATH}" -o ! -d "${INPUTPATH}" ];then
        log INFO "${INPUTPATH} is not exits."
        return 1
    fi

    # 源版本
    SRC_VERSION=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['src_version'])" 2>/dev/null)
    if [ -z "${SRC_VERSION}" ];then
        log INFO "Source Version:${SRC_VERSION} is not exits."
        return 1
    fi

    # 是否升级OS
    IS_UPGRADE_OSPATCH=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['os_patch'])" 2>/dev/null)
    if [ -z "${IS_UPGRADE_OSPATCH}" ];then
        log INFO "Is select upgrade ospatch:${IS_UPGRADE_OSPATCH} is not exits."
        return 1
    fi
    return 0
}

##################################################
# Description:  判断是否需要检查，勾选升级，非欧拉操作系统跳过检查
# Parameters:   INPUTPATH 入参保存的json文件
##################################################
function is_skip_check()
{
    log INFO "\n********** Start to execute the is_skip_check() **********\n" -nostd
    # IS_UPGRADE_OSPATCH 为true 说明升级时候勾选了OS，跳过检查
    if [ "${IS_UPGRADE_OSPATCH}" = "true" ];then
        log INFO "Is select upgrade ospatch, skip checking whether the operating system supports rollback."
        return 0
    fi

    # 判断是否为欧拉
    uname -r | grep euleros >/dev/null
    if [ "$?" -ne 0 ];then
        log INFO "If the operating system is not EulerOS, skip checking whether the operating system supports rollback."
        return 0
    fi
    return 1
}

##################################################
# Description:  主函数
# Parameters:   无
##################################################
function main_can_rollback() {
    define_public || return 1
    create_public || return 1
    # 检查入参
    get_upgrade_data "$@" || return 1
    # 勾选升级，非欧拉操作系统直接返回成功
    is_skip_check
    if [ $? -eq 0 ];then
        return 0
    fi

    check_euler_version || return 1
    check_version_older_than "$@" || return 1
}
main_can_rollback "$@"
