#!/bin/bash
###############################################################################
# Copyright © Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
# File name: vrpatch_precheck.sh
# Description: Pre-installation Check for VR Patches.
# Usage： bash vrpatch_precheck.sh upgrade/rollback package_name
###############################################################################

VR_PATH="/opt/oss/secsoter"
VRPATCH_PATH="/opt/oss/vrpatch"

##################################################
# Description: 日志函数
# @Param: null
##################################################
function print_log() {
    echo -e "$(date +%Y-%m-%d) $(date +%H:%M:%S) | ${1} | ${2}"
}

##################################################
# Description: 脚本入参校验
# @Param1: upgrade--升级场景检查 | rollback--回滚场景检查
# @Param2: VRPatch包名
##################################################
function check_param() {
    local action="$1"
    local package_name="$2"

    # 校验参数用途
    local usage_list=("upgrade" "rollback")
    echo "${usage_list[@]}" | grep -w "${action}" >/dev/null
    if [ $? -ne 0 ] || [ -z "${package_name}" ] ; then
        print_log "ERROR" "Parameter error, the correct usage is : vrpatch_precheck.sh upgrade/rollback package_name"
        return 1
    fi
}

##################################################
# Description: 前置检查
# @Param1: upgrade--升级场景检查 | rollback--回滚场景检查
# @Param2: VRPatch包名
##################################################
function pre_check() {
    local package_version=$(echo $2 | awk -F'-' '{print $1}')
    local package_name="${package_version}.zip"
    if [ "$1" = "upgrade" ]; then
        return 0
    elif [ "$1" = "rollback" ]; then
        # 若回滚版本不是上次升级的版本则报错
        local last_upgrade_version=$(tail -1 "${VRPATCH_PATH}"/upgrade_result 2>/dev/null | awk -F'=' '{print $1}')
        local last_upgrade_result=$(tail -1 "${VRPATCH_PATH}"/upgrade_result 2>/dev/null | awk -F'=' '{print $2}')
        if [ "${last_upgrade_version}" != "${package_version}" ]; then
            print_log "ERROR" "Current VRPatch version ${last_upgrade_version} cannot rollback to ${package_version}."
            return 1
        fi

        local count=$(ls "${VR_PATH}" | grep "VRPatch$" | wc -l)
        if [ "${count}" -eq 1 ] && [ "${last_upgrade_result}" == "success" ] && [ -f /opt/oss/vrpatch/Secondary_upgrade.flag ]; then
            print_log "ERROR" "The VRPatch does not support two consecutive rollbacks."
            return 1
        fi
    else
        print_log "ERROR" "Unsupported usage: $@"
        return 1
    fi
}

function main() {
    check_param "$@" || return $?
    pre_check "$@" || return $?
    print_log "INFO" "Pre-check finished."
}
main "$@"