# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
# -*- coding: utf-8 -*-
"""
检查项配置类与任务类
"""
import os
from utils.log_util import EasysuiteLogger
from .check_exception import CheckException
from .check_util import CheckOperationUtils, CheckOSInfoUtils


class OSCheckConfig:

    def __init__(self, config_dict):
        # 检查项名称
        self.__task_name = config_dict.get('task_name', 'task_unknown')
        # sudo包在检查项目录中的相对路径
        self.__sudo_pkg_loc = config_dict.get('sudo_pkg_loc', '')
        # sudo包名
        self.__sudo_pkg_file = os.path.basename(self.sudo_pkg_loc)
        # sudo包临时解压目录
        self.__sudo_tmp_path = config_dict.get('sudo_tmp_path', '')
        # 上传源目录
        self.__upload_src_path = config_dict.get('upload_src_path', '')
        # 上传目标目录
        self.__upload_trg_path = config_dict.get('upload_trg_path', '')

    @property
    def task_name(self):
        return self.__task_name

    @property
    def sudo_pkg_loc(self):
        return self.__sudo_pkg_loc

    @property
    def sudo_pkg_file(self):
        return self.__sudo_pkg_file

    @property
    def sudo_tmp_path(self):
        return self.__sudo_tmp_path

    @property
    def upload_trg_path(self):
        return self.__upload_trg_path

    @property
    def upload_src_path(self):
        return self.__upload_src_path


class OSCheckTask:
    """
    检查项任务类，主要定义检查项的执行结构。
    """

    def __init__(self, config, kvs):
        """
        任务初始化
        :param config: 任务配置OSCheckConfig对象
        :param kvs: 全局字典
        """
        self.kvs = kvs
        self.config = config
        self.nodes_info = CheckOSInfoUtils.get_node_connect_info(kvs)
        self.nodes_info_backup = self.nodes_info.copy()
        self.logger = EasysuiteLogger.get_logger(kvs, self.name)
        self.logger.easysuite_init(f"Task {self.name} init success")

    @property
    def name(self):
        """
        获取检查项任务名
        :return: str 检查项任务名
        """
        return self.config.task_name

    def check_kvs_contain(self, target, keyword):
        """
        检查kvs指定字段的值中是否存在目标内容
        :param target: 目标内容
        :param keyword: kvs字典指定字段
        :return: bool 执行是否成功
        """
        return target in self.kvs.get(keyword, '')

    def mask_sites(self, sites):
        """
        屏蔽某些站点的后续检查操作
        :param sites: list，屏蔽站点列表
        :return: None
        """
        for site in sites:
            if site in self.nodes_info:
                self.nodes_info[site] = None

    def unmask_sites(self, sites):
        """
        解除某些站点的检查屏蔽
        :param sites: list，解除屏蔽站点列表
        :return: None
        """
        for site in sites:
            if site in self.nodes_info:
                self.nodes_info[site] = self.nodes_info_backup[site]

    def upload_sudo_pkg(self):
        """
        上传提权包
        :return: bool 执行是否成功
        """
        src_file = self.config.sudo_pkg_loc
        trg_file = os.path.basename(src_file)
        return self.upload_file(src_file, trg_file)

    def upload_file(self, src_file, trg_file=""):
        """
        上传脚本到远程指定目录
        :param src_file: 上传源文件名
        :param trg_file: 上传目标文件名，如无传入则与源文件名一致
        :return: bool 执行是否成功
        """
        try:
            src_path = self.config.upload_src_path
            trg_path = self.config.upload_trg_path
            if not trg_file:
                trg_file = src_file
            self.logger.easysuite_info(f"Begin to upload file {src_path}/{src_file} to "
                                       f"remote:{trg_path}/{trg_file}.")
            CheckOperationUtils.upload_file_to_all_nodes(self.nodes_info, src_file, src_path, trg_file, trg_path)
            self.logger.easysuite_info("Success to upload file.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to upload file: {e}")
            return False

    def execute_sudo_script(self, script_name, exec_params=''):
        """
        执行提权脚本进行检查
        :param script_name: 提权执行脚本名称
        :param exec_params: 执行脚本时传入的参数
        :return: bool 执行是否成功
        """
        try:
            upload_pkg = self.config.sudo_pkg_file
            sudo_pkg_full_path = os.path.join(self.config.upload_trg_path, upload_pkg)
            tmp_path = self.config.sudo_tmp_path
            script_with_params = f"{script_name} {exec_params}" if exec_params else script_name
            self.logger.easysuite_info(f"Begin to execute sudo check script: {script_name}.")
            CheckOperationUtils.execute_sudo_script(self.nodes_info, script_with_params, sudo_pkg_full_path, tmp_path)
            self.logger.easysuite_info(f"Success to execute sudo check script: {script_name}.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to execute sudo check script: {e}.")
            return False

    def execute_script(self, script_name, exec_params=''):
        """
        正常执行脚本进行检查
        :param script_name: 执行脚本名称
        :param exec_params: 执行脚本时传入的参数
        :return: bool 执行是否成功
        """
        try:
            script_full_path = os.path.join(self.config.upload_trg_path, script_name)
            script_with_params = f"{script_full_path} {exec_params}" if exec_params else script_full_path
            self.logger.easysuite_info(f"Begin to execute check script: {script_full_path}.")
            CheckOperationUtils.execute_script(self.nodes_info, script_with_params)
            self.logger.easysuite_info(f"Success to execute check script: {script_full_path}.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to execute check script: {e}.")
            return False

    def check_results(self, result_full_path, check_keyword='check'):
        """
        检查执行脚本后的结果
        :param result_full_path: 执行结果文件绝对路径
        :param check_keyword: 执行结果中判断是否通过的关键字
        :return: bool 执行是否成功
        """
        try:
            task_name = self.config.task_name
            self.logger.easysuite_info(f"Begin to check results: {result_full_path}.")
            CheckOperationUtils.check_script_results(self.nodes_info, result_full_path, task_name, check_keyword)
            self.logger.easysuite_info(f"Success to check results: {result_full_path}.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check results: {e}.")
            return False

    def rm_sudo_pkg(self):
        """
        删除上传sudo包
        :return: bool 执行是否成功
        """
        return self.rm_remote_file(self.config.sudo_pkg_file)

    def rm_remote_file(self, file_name):
        """
        删除远程文件
        :param file_name: 待删除远程文件名
        :return: bool 执行是否成功
        """
        try:
            file_full_path = os.path.join(self.config.upload_trg_path, file_name)
            self.logger.easysuite_info(f"Begin to clear uploaded script: {file_full_path}")
            CheckOperationUtils.remove_nodes_file(self.nodes_info, file_full_path)
            self.logger.easysuite_info(f"Success to clear uploaded script: {file_full_path}")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to clear upload script: {e}.")
            return False

    def rm_remote_files(self, file_list):
        """
        删除远程复数文件
        :param file_list: 待删除远程文件名列表
        :return: bool 执行是否成功
        """
        for file in file_list:
            if not self.rm_remote_file(file):
                return False
        return True

    def check_nodes_os_type_consistence(self):
        """
        检查节点操作系统版本类型一致性
        :return: bool 执行是否成功
        """
        try:
            self.logger.easysuite_info("Begin to check nodes consistence")
            result = CheckOSInfoUtils.check_nodes_os_type_consistence(self.nodes_info)
            for ip, os_type in result:
                self.logger.easysuite_info(f"The OS type of the node {ip} is {os_type}.")
            self.logger.easysuite_info("Success to check nodes consistence")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check nodes consistence: {e}")
            return False

    def check_suse_kernel(self):
        """
        检查Suse系统内核
        :return: bool 是否为合法Suse版本内核
        """
        try:
            self.logger.easysuite_info(f"Begin to check suse kernel version")
            results = CheckOSInfoUtils.get_suse_kernel(self.nodes_info)
            for node_ip, kernel_version in results.items():
                self.logger.easysuite_info(f"{node_ip} suse kernel is {kernel_version}.")
            self.logger.easysuite_info("Success to check suse kernel version. "
                                       "The current kernel version is a non-problem version.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check suse kernel version: {e}")
            return False

    def get_sites_os_consistence(self):
        """
        检查站点中节点操作系统的一致性
        :return: list 操作系统一致的站点
        """
        try:
            self.logger.easysuite_info(f"Begin to check nodes OS version.")
            sites = CheckOSInfoUtils.get_sites_os_consistence(self.nodes_info)
            self.logger.easysuite_info("Success to check nodes OS version.")
            return sites
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check nodes OS version: {e}")
            return []

    def get_customer_suse_os_type(self):
        """
        获取客户自备Suse版本.
        :return: str Suse版本
        """
        try:
            self.logger.easysuite_info(f"Begin to get suse os type.")
            os_type = CheckOSInfoUtils.get_suse_os_type(self.kvs)
            self.logger.easysuite_info(f"Success to get suse os type: {os_type}")
            return os_type
        except CheckException as e:
            self.logger.easysuite_info(f"Failed to get suse os type: {e}.")
            return ""

    def check_customer_suse_os_type(self):
        """
        检查客户自备Suse版本是否为支持版本
        :return: bool 是否为合法Suse版本类型
        """
        try:
            self.logger.easysuite_info(f"Begin to check suse os type.")
            os_type = CheckOSInfoUtils.check_suse_os_type(self.kvs)
            self.logger.easysuite_info(f"Success to check suse os type. "
                                       f"The os type is {os_type}, which is supported os type")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check suse os type: {e}.")
            return False

    def check_env_rpm(self):
        """
        检查操作系统组件
        :return: bool 执行是否成功
        """
        try:
            self.logger.easysuite_info(f"Begin to check OS components.")
            CheckOSInfoUtils.check_nodes_rpm(self.kvs, self.nodes_info, self.config.upload_src_path)
            self.logger.easysuite_info("Success to check OS components.")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check OS components: \n{e}")
            return False

    def check_nscd_version(self):
        """
        检查NSCD组件版本
        :return: bool 执行是否成功
        """
        try:
            self.logger.easysuite_info(f"Begin to check nscd version.")
            message = CheckOSInfoUtils.check_nodes_nscd_version(self.nodes_info)
            self.logger.easysuite_info(f"Success to check nscd version: \n{message}")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check nscd version: \n{e}.")
            return False

    def check_os_patch_consistence(self):
        """
        检查操作系统补丁一致性
        :return: bool 执行是否成功
        """
        try:
            self.logger.easysuite_info(f"Begin to check os patch consistence.")
            CheckOSInfoUtils.check_os_patch_consistence(self.kvs)
            self.logger.easysuite_info("Success to check os patch consistence")
            return True
        except CheckException as e:
            self.logger.easysuite_error(f"Failed to check os patch consistence: {e}")
            return False

    def upload_and_execute_script(self, script_name, exec_params=""):
        """
        上传脚本并执行脚本
        :param script_name: 待执行脚本名
        :param exec_params: 执行脚本时传入的参数
        :return: bool 执行是否成功
        """
        return self.upload_file(script_name) and self.execute_script(script_name, exec_params)

    def finish(self):
        """
        任务结束时的收尾操作
        :return: None
        """
        # 完成后，最后在前端对应检查项显示打勾
        self.logger.easysuite_finish(f"Task {self.config.task_name} execute success.")
