import json

from datetime import datetime

from exception.exception import HealthCheckException, UnclearedAlarmCheckException
from data.alarm_receive_param import get_alarm_receive_param
from steps.hardware_alarm_param_check import HardwareParamCheck
from steps.uncleared_hardware_alarm_check import UnclearedHardwareAlarmCheck
from utils.log_utils import get_logger
from utils.manager import Manager

LOGGER = get_logger()
PREVIOUS_CHECK_RESULT = "/opt/oss/tmp/manager/HardwareAlarmCheck/previous_check.json"
RET_MSG = []


def process_param_check(result: list) -> bool:
    """
    将硬件参数检查结果处理成标准硬件告警检查结果格式
    :param result: 硬件参数检查结果
    """
    check_flag = False
    for obj in result:
        if obj.get('ErrLevel', -1) != 0:
            msg = f"Error code is :{obj.get('ErrCode', '')}, Error information: {obj.get('ErrInfo', '')}."
            RET_MSG.append(msg)
            check_flag = True
    return check_flag


def get_result(flag: bool) -> dict:
    """
    输出硬件告警检查结果字典
    :param flag:检查结果
    :return: 检查结果字典
    """
    return {'CheckStatus': flag, 'ResultMessage': RET_MSG}


def process_exception(error_info: str) -> dict:
    """
    处理异常结果并打印对应消息
    :param error_info:异常内容
    :return: 检查结果字典
    """
    LOGGER.error(error_info, exc_info=True)
    RET_MSG.append(error_info)
    return get_result(False)


def hardware_alarm_check():
    """
    硬件告警检查主入口
    """

    LOGGER.info(f"Hardware alarm check start at {str(datetime.now())}.")

    """ 检查当前环境是否为E9000,决定是否跳过硬件告警参数检查 """
    e9000_param_check = False
    node_list = Manager().alarm_receive_info
    if not node_list:
        LOGGER.error("Alarm receiving parameters are empty.")
        msg = f"Error code is: 13, Error information: Alarm receiving parameters are empty."
        RET_MSG.append(msg)
        return get_result(False)
    for node in node_list:
        if node.get("nodeType", "") == "E9000":
            LOGGER.info("Current server type is E9000, skip hardware health check.")
            e9000_param_check = True
            break
    return hardware_alarm_params_check(e9000_param_check, node_list)


def hardware_alarm_params_check(e9000_param_check: bool, node_list: list):
    """ 检查硬件告警参数 """
    if not e9000_param_check:
        # 正常检查硬件参数配置
        LOGGER.info("Begin hardware param check.")
        hardware_param_check = HardwareParamCheck()
        try:
            hardware_param_check.health_check()
            check_flag = process_param_check(hardware_param_check.get_result_list())
            if check_flag:
                return get_result(False)
        except HealthCheckException as e:
            error_info = f"health check catch an health check exception: {e}"
            return process_exception(error_info)
        except Exception as e:
            error_info = f"health check catch an exception: {e}"
            return process_exception(error_info)
    else:
        # E9000环境特殊检查
        LOGGER.info("Begin E9000 hardware param check.")
        param_check_result = True
        for node in node_list:
            ip = node.get("nodeIP", "")
            node_type = node.get("nodeType", "")
            LOGGER.info(f"Current node IP is: {ip}, and node type is: {node_type}")
            # 跳过FC端口检查
            if node_type == "FusionCompute":
                continue
            alarm_receive = get_alarm_receive_param(ip)
            if not Manager().verify_alarm_receive_param(alarm_receive.get_json()):
                param_check_result = False
                LOGGER.error(f"{ip} alarm receiving parameters are not configured corretly.")
                msg = f"Error code is: 14, Error information: Alarm receiving parameters in node {ip} are not " \
                      f"configured correctly."
                RET_MSG.append(msg)
        if not param_check_result:
            LOGGER.error("Check E9000 hardware parameters failed.")
            return get_result(False)
    return uncleared_hardware_alarm_check()


def uncleared_hardware_alarm_check():
    """ 检查未清除告警 """
    LOGGER.info("Begin hardware uncleared alarm check.")
    try:
        uncleared_alarm = UnclearedHardwareAlarmCheck()
        uncleared_result_list = uncleared_alarm.check_uncleared_alarm()
        if uncleared_result_list:
            RET_MSG.extend(uncleared_result_list)
            return get_result(False)
    except UnclearedAlarmCheckException as e:
        error_info = f"uncleared alarm check catch an uncleared alarm check exception: {e}"
        return process_exception(error_info)
    except Exception as e:
        error_info = f"uncleared alarm check catch an exception: {e}"
        return process_exception(error_info)

    LOGGER.info("Finish hardware alarm check.")
    RET_MSG.append("Finish hardware alarm check.")
    return get_result(True)


def main():
    result = hardware_alarm_check()
    if result.get("CheckStatus", False):
        previous_check = {"check_result": "Successes"}
    else:
        previous_check = {"check_result": "Failed"}
    write_buffer = json.dumps(previous_check)
    with open(PREVIOUS_CHECK_RESULT, 'w') as previous_result_file:
        previous_result_file.write(write_buffer)
    print(json.dumps(result))


if __name__ == '__main__':
    main()

