# -*- coding: utf-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from constant.constant import RetCode
from utils.common import LazyProperty
from data.alarm_forward_param import AlarmForwardParam
from service.service import Service
from utils.log_utils import get_logger

LOGGER = get_logger()


class ServerService(Service):
    """ 服务器健康检查 """

    def __init__(self, service_info):
        super().__init__(service_info)

    def check_connect_cmd(self):
        return "ipmcget -t smbios -d systemname"

    @LazyProperty
    def service_alarm_forward_user(self) -> str:
        """
        获取服务器配置的告警转发用户
        :return: str
        """
        LOGGER.info(f"[%s] start to get alarm forward user", self.service_info)
        ret, stdout, stderr = self.execute_cmd("ipmcget -t trap -d user")
        if ret != 0:
            LOGGER.error(f"[{self.service_info}] get alarm forward user for exec cmd failed, stderr {stderr}")
            return ""
        user_list = [line for line in stdout.split("\n") if "Trap user" in line]
        if not user_list:
            LOGGER.error(f"[{self.service_info}] get snmp user failed for parse cmd failed, stdout {stdout}")
            return ""
        return user_list[0].split(":")[-1].strip()

    def get_service_alarm_forward_param(self) -> list:
        """ 获取告警转发参数 """
        server_snmp_user = self.service_alarm_forward_user
        LOGGER.info(f"[{self.service_info}] start to get service alarm forward param")
        ret, stdout, stderr = self.execute_cmd("ipmcget -t trap -d trapiteminfo")
        if ret != 0:
            LOGGER.error(f"[{self.service_info}] get service alarm forward param failed, stderr {stderr}")
            return []
        snmp_info_list = []
        check_start = False
        for line in stdout.split("\n"):
            if "|" in stdout and "alert address" in line:
                check_start = True
                continue
            if check_start and "|" in line:
                snmp_info_list.append(line)
        result = []
        for snmp_info in snmp_info_list:
            snmp_info_detail = [info.strip() for info in snmp_info.split("|")]
            if len(snmp_info_detail) < 4:
                LOGGER.error(f"[{self.service_info}] alarm forward param is wrong: detail {snmp_info_detail}")
                continue
            state = snmp_info_detail[1]
            port = snmp_info_detail[2]
            ip = snmp_info_detail[3]
            LOGGER.info(f"[{self.service_info}] alarm forward param {ip} state is {state}")
            if state != "enabled":
                continue
            alarm_info = AlarmForwardParam(ip=ip, user=server_snmp_user, port=port)
            LOGGER.info(f"[{self.service_info}] add alarm param {alarm_info}")
            result.append(alarm_info)
        LOGGER.info(f"[{self.service_info}] get alarm forward param is {result}")
        return result

    def build_check_function_list(self) -> list:
        """ 生成check list，列表内容为(func, RetCode)"""
        return [
            (self.check_local_alarm_ip_ping_to_service_ip, RetCode.PING_ERROR),
            (self.check_connect, RetCode.CONNECT_ERROR),
            (self.check_alarm_receive_param_exist, RetCode.ALARM_RECEIVE_PARAM_NOT_EXIST_ERROR),
            (self.check_alarm_receive_param, RetCode.ALARM_RECEIVE_PARAM_ERROR),
            (self.check_alarm_forward_param, RetCode.ALARM_FORWARD_PARAM_ERROR)
        ]
