#!/bin/bash
###########################################
# 用于判断是否可以在当前OS环境下回退NCE
# 如果当前OS版本是V2R9，且目标版本低于R20C10
# 则脚本返回1，不可回滚NCE。否则可回滚NCE
###########################################

##################################################
# Description:  全局公共变量定义
# Parameters:   无
##################################################
function define_public() {
    # 目录
    LOGPATH="/opt/oss/log/manager/easysuite_upgrade"
    # 文件
    LOGFILE="${LOGPATH}/can_rollback_nce.log"

    # 变量
    EULER_VR=""
}

##################################################
# Description:  创建目录文件
# Parameters:   无
##################################################
function create_public() {
    # 创建目录
    mkdir -p "${LOGPATH}"

    # 创建文件
    touch "${LOGFILE}"
}

##################################################
# Description:  日志打印
#               使用方法: log 日志级别 日志信息 [-nostd]
# Parameters:   日志级别: DEBUG、INFO、WARN、ERROR
#               日志信息: message
#               -nostd: 不显示到标准输出流，仅打印到日志文件
#Example:       log ERROR "message"
#               log INFO "message" -nostd
##################################################
function log() {
    # 接收参数
    local logtype=$1;
    local logmsg=$2;
    local nostd=$3;

    # 当前时间
    local datetime=$(date +'%F %T')

    # 函数调用
    local fncall=""

    # 代码行位置
    local line=""

    # 级别颜色
    case "${logtype}" in
        DEBUG)
            logtype="\033[35m${logtype}\033[0m" ;;
        INFO)
            logtype="\033[36m${logtype}\033[0m" ;;
        WARN)
            logtype="\033[33m${logtype}\033[0m" ;;
        ERROR)
            logtype="\033[31m${logtype}\033[0m" ;;
    esac

    # 有错显示函数调用及代码行位置
    if [[ "${logtype}" =~ "ERROR" ]] || [[ "${logtype}" =~ "WARN" ]];then
        fncall="${FUNCNAME[@]/log/}"
        line="[line:$(caller 0 | awk '{print$1}')]"
    fi

    # 日志格式
    logformat="${datetime} ${logtype} ${fncall} ${line} ${logmsg}"

    # 日志输出
    [ "${nostd}" = "-nostd" ] && echo -e "${logformat}" >> "${LOGFILE}" || echo -e "${logformat}" | tee -a "${LOGFILE}"
}

##################################################
# Description:  Euler版本判断(V200R007/V200R008)
# Parameters:   无
##################################################
function check_euler_version() {
    log INFO "\n********** Start to execute the check_euler_version() **********\n"

    local version=""

    # 判断当前操作系统版本
    if [ ! -f "/etc/euleros-latest" ];then
        log ERROR "The euleros-latest file does not exist. The OS is not EulerOS"
        exit 0
    fi

    version=$(grep "eulerversion" "/etc/euleros-latest" |tr "=_" " " |xargs -n 1 |grep "^V[0-9].*$" |grep -P "V[0-9]{1,3}R[0-9]{1,3}" -o)

    if [ ! -n "${version}" ];then
        log ERROR "FATAL:Euler Version does not match"
        return 1
    fi


    EULER_VR="${version}"
    log INFO "Euler Version is ${EULER_VR}"

    [ "${EULER_VR}" ] && return 0
}

##################################################
# Description:  检查版本是否需要回滚
# Parameters:   无
##################################################
function check_version_older_than() {
    log INFO "\n********** Start to execute the check_version_older_than() **********\n"

    local currentEulerVersion="${EULER_VR}"
    local targetNCEVersion=$(echo "$1" |grep -P "R[0-9]{1,3}C[0-9]{1,3}" -o)

    log INFO "current Euler Version is ${currentEulerVersion}, target NCE Version is ${targetNCEVersion}"

    if [[ "${targetNCEVersion}" < "R019C00" || "${currentEulerVersion}" < "V200R007" ]]; then
            log ERROR "Error, The target version does not support rollback."
            return 1
    fi

    if [[ "${targetNCEVersion}" < "R020C10" && "${currentEulerVersion}" > "V200R008" ]]; then
            log ERROR "Error, The version does not match. Please roll back the OS first."
            return 1
    fi

    if [[ "${targetNCEVersion}" < "R021C10" && "${currentEulerVersion}" > "V200R009" ]]; then
            log ERROR "Error, The version does not match. Please roll back the OS first."
            return 1
    fi

    if [[ "${targetNCEVersion}" < "R023C00" && "${currentEulerVersion}" > "V200R010" ]]; then
        log ERROR "Error, The version does not match. Please roll back the OS first."
        return 1
    fi

    log INFO "Version matching, supporting rollback."
    return 0
}

##################################################
# Description:  主函数
# Parameters:   无
##################################################
function main_can_rollback() {
    define_public || return 1
    create_public || return 1
    check_euler_version || return 1
    check_version_older_than $@ || return 1
}
main_can_rollback $@
