#!/bin/bash

CUR_PATH=$(cd "$(dirname "$0")"; pwd)
OSS_USER=$(id -nu 3001)
SELF_FILE=$(basename $0)
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
[ ! -e "${LOG_PATH}" ] && mkdir -p "${LOG_PATH}"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"

##################################################
# 校验执行用户
# 脚本要求使用ossadm用户执行
##################################################
function check_user()
{
    local user
    user=$(whoami)
    if [ "${user}" != "${OSS_USER}" ]
    then
        echo "[$(date +'%Y-%m-%d %H:%M:%S')]| User have no permission to run this script.(easysuite.upgrade.cmd.fail)"
        return 1
    fi
}

# 更新容灾信息到配置文件用于后续重新建立容灾
function update_dr_info()
{
    if [ -f "${CUR_PATH}/../pyscripts/update_dr_info.pyc" ]
    then
        update_dr_file="${CUR_PATH}/../pyscripts/update_dr_info.pyc"
    else
        if [ -f "${CUR_PATH}/../pyscripts/update_dr_info.py" ]
        then
            update_dr_file="${CUR_PATH}/../pyscripts/update_dr_info.py"
        else
            echo "Error: Can not find dr operate file"
            return 1
        fi
    fi

    . /opt/oss/manager/bin/engr_profile.sh; python ${update_dr_file} ${DR_INFO_TMP_FILE}
    if [ "$?" == "0" ]
    then
        return 0
    else
        return 1
    fi
}

function get_drservicedb_val()
{
    # 参数1:数据库CONFIG_NAME字段
    local config_name=$1
    # 参数2:查询数据配置临时文件
    local arb_config=$2

    # 返回:数据库CONFIG_VALUE字段
    local config_value=$(grep "^${config_name} " "${arb_config}" | awk '{print $2}')
    echo "${config_value}" | egrep "[0-9]+\-[0-9]+\-[0-9]+" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo -n "${config_value}"
        return 0
    fi
    return 1
}

# 更新容灾拓展信息到配置文件用于后续重新建立容灾
function update_arbitration_info()
{
    if [ -f "${CUR_PATH}/../pyscripts/query_database.pyc" ]
    then
        query_database_file="${CUR_PATH}/../pyscripts/query_database.pyc"
    else
        if [ -f "${CUR_PATH}/../pyscripts/query_database.py" ]
        then
            query_database_file="${CUR_PATH}/../pyscripts/query_database.py"
        else
            echo "Error: Can not find query_database file"
            return 1
        fi
    fi

    # 初始容灾参数配置文件
    local arb_config="${DR_PATH}/pre_upgrade_query_arb.conf"
    if [ -f "${arb_config}" ]
    then
        rm -f "${arb_config}"
    fi

    # 获取drservicedb数据信息,输出到临时文件
    if [ ! -f "${arb_config}" ]
    then
        local result=$(. /opt/oss/manager/bin/engr_profile.sh; python ${query_database_file} 'get_db_info' 'managedbsvr' 'drservicedb' ${arb_config})
        echo "${result}" | grep "EXEC_SUCCESS" &>/dev/null
        if [ $? -ne 0 -o ! -f "${arb_config}" ]
        then
            echo "[backup_drservicedb_data] Error: ${result}"
            echo "[backup_drservicedb_data] Error: Failed to obtain the drservicedb data."
            return 1
        fi
    fi

    # 根据数据库查询的自动倒换开关来刷新容灾配置文件
    auto_switch_status=$(get_drservicedb_val "autoswitch_status" "${arb_config}")
    if [ "$?" == "0" ]
    then
        auto_switch=$(echo "${auto_switch_status}" | grep -w "ON")
        if [ -n "${auto_switch}" ]
        then
            sed -i "s/\"auto_switch\"/true/g" ${DR_INFO_TMP_FILE}
            # 当自动倒换开启时根据数据库查询的仲裁类型来刷新容灾配置文件
            arbiter_type_info=$(get_drservicedb_val "arbiterType" "${arb_config}")
            if [ "$?" == "0" ]
            then
                arbiter_type=$(echo "${arbiter_type_info}" | grep -w "AAMonitor")
                if [ -n "${arbiter_type}" ]
                then
                    arbiterType="AAMonitor"
                    sed -i "s/\${arbiterType}/${arbiterType}/g" ${DR_INFO_TMP_FILE}
                    # 当存在仲裁节点时时根据数据库查询的仲裁IP地址来刷新容灾配置文件
                    arbiter_ip_info=$(get_drservicedb_val "etcd_servers" "${arb_config}")
                    if [ "$?" == "0" ]
                    then
                        echo "$arbiter_ip_info" | fgrep -q .
                        if [ $? -eq 0 ]
                        then
                            # ipv4
                            arbiterIp=$(echo ${arbiter_ip_info} | awk '{print $1}' | sed 's/\:[0-9]\{4\}//g' | sed s/\,/\;/g)
                        else
                            # ipv6
                            arbiterIp=$(echo ${arbiter_ip_info} | awk '{print $1}' | sed 's/\[//g' | sed 's/]\:[0-9]\{4\}//g' | sed s/\,/\;/g)
                        fi
                        if [ -n "${arbiterIp}" ]
                        then
                            sed -i "s/\${arbiterIp}/${arbiterIp}/g" ${DR_INFO_TMP_FILE}
                        else
                            echo "Failed to get the arbitration IP."
                            return 1
                        fi
                    fi
                else
                    arbiter_type=$(echo "${arbiter_type_info}")
                    if [ -n "${arbiter_type}" ]
                    then
                        arbiterType=${arbiter_type}
                    else
                        arbiterType="doubleMonitor"
                    fi
                    sed -i "s/\${arbiterType}/${arbiterType}/g" ${DR_INFO_TMP_FILE}
                    sed -i "s/\${arbiterIp}//g" ${DR_INFO_TMP_FILE}
                fi
            fi
        else
            sed -i "s/\"auto_switch\"/false/g" ${DR_INFO_TMP_FILE}
            sed -i "s/\${arbiterType}//g" ${DR_INFO_TMP_FILE}
            sed -i "s/\${arbiterIp}//g" ${DR_INFO_TMP_FILE}
        fi
    else
        sed -i "s/\"auto_switch\"/false/g" ${DR_INFO_TMP_FILE}
        sed -i "s/\${arbiterType}//g" ${DR_INFO_TMP_FILE}
        sed -i "s/\${arbiterIp}//g" ${DR_INFO_TMP_FILE}
    fi

    # 根据数据库查询的心跳配置来刷新容灾配置文件
    hb_retry_times=$(get_drservicedb_val "hb_retry_times" "${arb_config}")
    if [ "$?" == "0" ]
    then
        if [ -z "${hb_retry_times}" ]
        then
            hb_retry_times="18"
        fi
        sed -i "s/\${hbRetryTimes}/${hb_retry_times}/g" ${DR_INFO_TMP_FILE}
    fi

    hb_retry_interval=$(get_drservicedb_val "hb_retry_interval" "${arb_config}")
    if [ "$?" == "0" ]
    then
        if [ -z "${hb_retry_interval}" ]
        then
            hb_retry_interval="10"
        fi
        sed -i "s/\${hbRetryInterval}/${hb_retry_interval}/g" ${DR_INFO_TMP_FILE}
    fi

    autoswitch_delaytime=$(get_drservicedb_val "autoswitch_delaytime" "${arb_config}")
    if [ "$?" == "0" ]
    then
        if [ -z "${autoswitch_delaytime}" ]
        then
            autoswitch_delaytime="2"
        fi
        sed -i "s/\${switchDelay}/${autoswitch_delaytime}/g" ${DR_INFO_TMP_FILE}
    fi

    local isAppArbitrationOn=$(get_drservicedb_val "isAppArbitrationOn" "${arb_config}")
    sed -i "s/\${isAppArbitrationOn}/${isAppArbitrationOn}/g" ${DR_INFO_TMP_FILE}
    local isNetworkArbitrationOn=$(get_drservicedb_val "isNetworkArbitrationOn" "${arb_config}")
    sed -i "s/\${isNetworkArbitrationOn}/${isNetworkArbitrationOn}/g" ${DR_INFO_TMP_FILE}
    local isDBArbitrationOn=$(get_drservicedb_val "isDBArbitrationOn" "${arb_config}")
    sed -i "s/\${isDBArbitrationOn}/${isDBArbitrationOn}/g" ${DR_INFO_TMP_FILE}
    local isRedisArbitrationOn=$(get_drservicedb_val "isRedisArbitrationOn" "${arb_config}")
    sed -i "s/\${isRedisArbitrationOn}/${isRedisArbitrationOn}/g" ${DR_INFO_TMP_FILE}
    local isAppArbiDelayTimeOn=$(get_drservicedb_val "isAppArbiDelayTimeOn" "${arb_config}")
    sed -i "s/\${isAppArbiDelayTimeOn}/${isAppArbiDelayTimeOn}/g" ${DR_INFO_TMP_FILE}
    local isFastSwitchOn=$(get_drservicedb_val "isFastSwitchOn" "${arb_config}")
    sed -i "s/\${isFastSwitchOn}/${isFastSwitchOn}/g" ${DR_INFO_TMP_FILE}

    local neArbitrationDomain=$(get_drservicedb_val "neArbitrationDomain" "${arb_config}")
    sed -i "s/\${neArbitrationDomain}/${neArbitrationDomain}/g" ${DR_INFO_TMP_FILE}

    return 0
}

####################################################
#   功能说明:检查容灾关系
####################################################
function check_dr_status()
{
    # 不存在容灾关系时，不刷新直接跳过
    bash ${CUR_PATH}/check_dr_status.sh
    if [ $? -ne 0 ]
    then
        LOG "WARN" "The DR relationship does not exist. Skip 'update dr info' step."
        return 1
    fi
}


####################################################
#   功能说明:保存容灾健康检查时间
####################################################
function update_dr_health_time()
{
    local save_file="${1}"
    . /opt/oss/manager/bin/engr_profile.sh
    python <<END
import os
import sys
import json
import time
from util import httpclient

retry_count = 10
# 最多尝试30秒
for _ in range(retry_count):
    status, response = httpclient.IRHttpClient().get('/rest/drmgrservice/v1/main/drmgr/scheduletask/config?isDraft=false')
    if status == 200 and response:
        break
    print(status, response)
    time.sleep(3)
else:
    print(f"Fail to send request to manager.(retry {retry_count})")
    sys.exit(1)
with os.fdopen(os.open("${save_file}", os.O_CREAT | os.O_WRONLY | os.O_TRUNC, mode=0o600), 'wb') as w_stream:
    w_stream.write(response)
sys.exit(0)
END
}


function main()
{
    check_user >/dev/null 2>&1
    if [ $? -ne 0 ]
    then
        echo "easysuite.upgrade.cmd.fail"
        return 1
    fi

    # 加载公共方法
    . ${CUR_PATH}/common.sh
    LOG "INFO" "Start to init ${LOG_FILE}."

    # 清空日志文件
    echo >"${LOG_FILE}"

    # 初始化目录和配置文件
    DR_PATH="$1"
    DR_INFO_TMP_FILE="${DR_PATH}/drinfo.json"
    [ ! -d "${DR_PATH}" ] && mkdir -p ${DR_PATH}

    # 容灾不存在跳过
    check_dr_status || return 0

    DR_INFO_FILE=${CUR_PATH}/../pyscripts/drinfo.json
    if [ ! -f "${DR_INFO_FILE}" ]
    then
        echo "The DR configuration file does not exist."
        LOG "WARN" "The DR configuration file does not exist."
        return 2
    else
        cp -rf ${DR_INFO_FILE} ${DR_PATH}
    fi

    update_dr_health_time "${DR_PATH}/dr_health_time.json" &>>"${LOG_FILE}" || return 1

    update_arbitration_info || return 1

    update_dr_info || return 1

}

main "$@"
exit $?