#!/bin/bash

SCRIPT_PATH=$(cd $(dirname $0);pwd)
SELF_FILE=$(basename $0)
LOG_PATH="/opt/oss/log/manager/easysuite_upgrade/scriptlog"
LOG_FILE="${LOG_PATH}/${SELF_FILE//.sh/}.log"
TMP_PATH="/opt/oss/log/manager/easysuite_upgrade_config"
CHECK_TMP_FILE="${TMP_PATH}/${SELF_FILE//.sh/}_$(date +%Y%m%d%H%M%S_%N)"
ERROR_CODE_FILE="${SCRIPT_PATH}/config/${SELF_FILE}.json"

function LOG()
{
    BASH_PID=$$
    if [ "$1" = "ERROR" -o "$1" = "error" ]
    then
        Level="ERROR"
    elif [ "$1" = "WARN" -o "$1" = "warn" ]
    then
        Level="WARN"
    else
        Level="INFO"
    fi
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${BASH_PID}] | ${Level} $2" | tee -a ${LOG_FILE}
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [${BASH_PID}] | ${Level} $2" 1>&2
}

function get_db_status_file() {
    LOG "INFO" "Start to check ${PRODUCTNAME} database status from ${node_ip}."
    SSH_CMD="ssh -o ConnectTimeout=3600 -o stricthostkeychecking=no -o ConnectionAttempts=3 -o ServerAliveInterval=10 ${node_ip} "
    ${SSH_CMD} "bash ${OSS_ROOT}/agent/bin/ipmc_adm -cmd statusnodes -tenant ${PRODUCTNAME}" >${CHECK_TMP_FILE} 2>/dev/null
    local ssh_result=$?
    if [ $ssh_result -eq 1 -o $ssh_result -eq 255 ]
    then
        LOG "ERROR" "Failed to connect ${node_ip}."
        # 返回255表示节点连接失败
        return 2
    elif [ $ssh_result -ne 0 ]
    then
        LOG "ERROR" "Failed to obtain the database status."
        return 1
    fi

    cat ${CHECK_TMP_FILE} >> ${LOG_FILE}

    # 检查结果文件有效性
    grep "RUNNING" ${CHECK_TMP_FILE} &>/dev/null
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "[${PRODUCTNAME}] The database status is abnormal."
        return 1
    fi
    return 0
}

function check_db_result() {
    # 获取DB节点IP
    node_ip=$(cat "${INPUTPATH}"/nodes_${PRODUCTNAME}.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['hostlist'][0]['nodemgrip'])" 2>/dev/null)
    if [ -z "${node_ip}" ]
    then
        LOG "ERROR" "Failed to query product IP."
        return 1
    fi

    # 检查DB状态
    local db_flag="False"
    local unknown_nodes_flag="False"
    for query_num in {1..6}
    do
        get_db_status_file || return $?
        LOG "INFO" "Query the database status for ${query_num} time"
        # 使用ipmc命令获取节点状态后，取第5列的DB节点状态
        local db_node_num=$(cat ${CHECK_TMP_FILE} | sed "1d" | awk '{print $5}' | grep -v NA | wc -l)
        local db_running_node_num=$(cat ${CHECK_TMP_FILE} | sed "1d" | awk '{print $5}' | grep -c RUNNING)
        LOG "INFO" "[${PRODUCTNAME}] DB Node sum: ${db_node_num}"
        LOG "INFO" "[${PRODUCTNAME}] DB Normal node: ${db_running_node_num}"

        # 检查DB业务节点的数据库状态是否全都正常(running状态)
        if [ ${db_node_num} -ne 0 -a ${db_node_num} -eq ${db_running_node_num} ]
        then
            local db_flag="True"
        else
            cat ${CHECK_TMP_FILE}
            [ ${query_num} -eq 6 ] || sleep 30s
            continue
        fi

        # 检查节点状态是否为UNKNOWN
        local unknown_num=$(cat ${CHECK_TMP_FILE} | grep UNKNOWN | wc -l)
        if [ ${unknown_num} -eq 0 ]
        then
           local unknown_nodes_flag="True"
           break
        else
            cat ${CHECK_TMP_FILE}
            [ ${query_num} -eq 6 ] || sleep 30s
        fi
    done
    if [ ${db_flag} == "False" ]
    then
        LOG "INFO" "Query info:"
        cat ${CHECK_TMP_FILE}
        LOG "ERROR" "[${PRODUCTNAME}] The database status is abnormal and the upgrade cannot be started."
        return 1
    fi
    
    if [ ${unknown_nodes_flag} == "False" ]
    then
        LOG "INFO" "Query info:"
        cat ${CHECK_TMP_FILE}
        LOG "ERROR" "[${PRODUCTNAME}] The current product has unknown status nodes."
        return 1
    fi
}

function check_services_status() {
    out_list=$(python ${SCRIPT_PATH}/pyscripts/pre_upgrade_check_service.py ${PRODUCTNAME})
    if [ $? -ne 0 ]
    then
        LOG "ERROR" "The service (${out_list}) for the ${PRODUCTNAME} is not running and the upgrade cannot be started."
        return 1
    fi
}

# 解析参数
function init_params() {
    PRODUCTNAME=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['productname'])" 2>/dev/null)
    if [ -z "${PRODUCTNAME}" ]
    then
        LOG "ERROR" "Productname is null."
        return 1
    fi
    IS_MGR_DOAMIN=$(cat ${INPUTPATH}/plandata.json 2>/dev/null | python -c "import json; import sys; obj=json.load(sys.stdin); print(obj['single_mgr_domain'])" 2>/dev/null)
    if [ -z "${IS_MGR_DOAMIN}" ]
    then
        LOG "ERROR" "Failed to query the scenario."
        return 1
    fi
}

# 如果是单管，跳过检查产品运行
function check_single_domain() {
    if [ ${IS_MGR_DOAMIN} == "yes" ]
    then
        return 0
    fi
    return 1
}

# 脚本失败生成错误码
function generate_error_code() {
    python <<EOF
import os
import json

with open("${ERROR_CODE_FILE}", "r") as f:
    error_code_details = json.loads(f.read())
error_code_detail = error_code_details.get("${main_result}", {})

error_code_json = os.path.join("${ERROR_PATH}", "${SELF_FILE}.json")
with open(error_code_json, "w") as f:
    f.write(json.dumps(error_code_detail))
EOF
}

function main()
{
    source /opt/oss/manager/bin/engr_profile.sh
    # 创建日志目录
    [ ! -d "${LOG_PATH}" ] && mkdir -p ${LOG_PATH}
    # 创建临时文件目录
    [ ! -d "${TMP_PATH}" ] && mkdir -p ${TMP_PATH}
    INPUTPATH="$2"
    ERROR_PATH="$4"
    init_params || return 1
    check_db_result || return $?
    check_single_domain && return 0
    check_services_status || return 1
}

main "$@"
main_result=$?
# 生成错误码
[ ${main_result} -ne 0 ] && generate_error_code
[ -f "${CHECK_TMP_FILE}" ] && rm -f "${CHECK_TMP_FILE}"
exit ${main_result}