# encoding=utf-8
"""
功 能：增量备份产品数据
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
脚本参数 :
# 1-product_name
# 2-ip
# 3-username
# 4-omp_version
# 5-upgrade_path
# 6-remote_backup
# 7-task_id
# 8-e_task_id
# 9-time_out
# 10-site
"""

import json
import os
import re
import sys
import time

from datetime import datetime
from backup_product_url import backup_product_service as backup_product_func
from backup_product_url import BackupProductData
from check_building_db import main as check_build_func
from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr
from taskmgr_util import Taskmgrutil
from upgrade_settings import TASK_PATH, BACKUP_TIME_STAMP, BACKUP_PATH, BACKUP_RUNNING_STATE, \
    BACKUP_FAILURE_STATE, BACKUP_DIFF_ACTION, BACKUP_FULL_ACTION


LOG = Logger().getinstance(sys.argv[0])


class BackupProductWithDiff:
    """
    增量备份
    """

    def __init__(self, e_task_path, time_out=14400, backup_action=BACKUP_DIFF_ACTION):
        self.task_path = e_task_path
        self.start_time = int(time.time())
        self.time_out = int(time_out)
        self.action = backup_action

    def get_response(self, response):
        """
        功能说明：请求响应
        :return:
        """
        if not response:
            return False
        try:
            json.loads(response)
        except Exception:
            LOG.error(f"Failed to load json from \"{response}\"")
            Taskmgrutil.set_task(self.task_path, msg=f'The task response is \"{response}\"')
            return False
        return True

    def query_backup_progress(self, url):
        """
        查询恢复产品数据任务进度
        :param url: 查询任务执行状态所需的url
        :return:
        """
        result = False
        details = str()
        Taskmgrutil.set_task(self.task_path, msg="Start to query backup product progress.")
        max_time = self.start_time + self.time_out
        while True:
            # 超时报错退出
            if int(time.time()) >= max_time:
                LOG.error("Time_out")
                Taskmgrutil.set_task(self.task_path, status='ERROR', progress='100',
                                     msg=f'The task execution timeout exceeds '
                                         f'{self.time_out} seconds.')
                break

            status, response = Unieptaskmgr().send_get_request(url)

            # 请求失败
            if not status or not response:
                Taskmgrutil.set_task(self.task_path, status='ERROR', progress='100',
                                     msg='Failed to send get request.')
                break

            # json解析失败重试
            if not self.get_response(response):
                Taskmgrutil.set_task(self.task_path, status='ERROR', progress='100',
                                     msg='Failed to get response.')
                break

            execute_result = json.loads(response)[0].get("currentState")
            progress = json.loads(response)[0].get('progress')
            details = json.loads(response)[0].get('details')
            if not execute_result or execute_result in BACKUP_FAILURE_STATE:
                Taskmgrutil.set_task(self.task_path,
                                     msg=f'Backup product is error, progress is {progress}%.')
                break
            elif execute_result in BACKUP_RUNNING_STATE:
                # 运行中状态
                Taskmgrutil.set_task(self.task_path, progress=f"{progress}",
                                     msg=f'Backup product is running, progress is {progress}%.')
                time.sleep(5)
            else:
                Taskmgrutil.set_task(self.task_path, progress='99',
                                     msg=f'Backup product is finish, progress is {progress}%.')
                result = True
                break
        if details:
            Taskmgrutil.set_task(self.task_path, msg=f'Detail:{details}.')
        return result

    @staticmethod
    def get_backup_time_stamp(backup_url):
        """
        功能说明:获取备份任务对应备份时间戳
        :param backup_url: 备份任务url
        :return:
        """
        status, response = Unieptaskmgr().send_get_request(backup_url)
        # 执行失败返回空串
        if not status:
            LOG.error(f"Failed to get time stamp from {backup_url}.response:{response}")
            return str()
        details = json.loads(response)[0].get("details")
        timestamp = re.search("([0-9]{17})", details).group(1)
        return timestamp

    def get_stamp_key(self, argv):
        """
        功能说明:获取备份时间戳key值
        :param argv:
        :return:
        """
        if self.action == "backup_product_diff":
            # [action]_[product]_[upgrade_path]
            return f"{self.action}_{argv[1]}_{argv[5]}"
        # [action]_[product]_[upgrade_path]
        return f"{self.action}_{argv[1]}_{argv[5]}"

    def get_backup_type(self):
        """
        功能说明:获取备份类型
        :return:
        """
        if self.action == BACKUP_DIFF_ACTION:
            return "PHYSICAL_DIFF_BACKUP"
        return ""

    def write_backup_timestamp_file(self, argv, backup_time_stamp):
        """
        写备份产品数据的时间戳到OMP后台文件
        :param argv:
        :param backup_time_stamp:
        :return:
        """
        LOG.info("Start to write backup timestamp file")
        try:
            date_sec_cmd = datetime.now().strftime("%s")
            # 时间戳
            timestamp_str = f"{self.get_stamp_key(argv)}:{backup_time_stamp}:{date_sec_cmd}"
            if not os.path.isdir(BACKUP_PATH):
                os.makedirs(BACKUP_PATH)
            with os.fdopen(os.open(BACKUP_TIME_STAMP, os.O_CREAT | os.O_WRONLY | os.O_APPEND,
                                   mode=0o640), 'a', encoding='utf-8') as w_stream:
                w_stream.write(f"{timestamp_str}{os.linesep}")
        except Exception as e:
            LOG.error(f"Write backup timestamp file throw an exception: {e}")
            return False
        LOG.info(f"Write backup timestamp file success, backup timestamp:{backup_time_stamp}.")
        return True

    def write_full_diff_timestamp_file(self, argv, backup_time_stamp):
        """
        将增量备份功能的全量时间戳和增量时间戳的对应关系记录到OMP后台文件
        :param argv:
        :param backup_time_stamp:
        :return:
        """
        LOG.info("Start to write full-diff timestamp")
        try:
            # 全量备份时间戳
            backup_full_timestamp = argv[11]
            if backup_full_timestamp == "NA":
                LOG.info(f"Backup mode is {argv[10]}. Skip write full-diff timestamp.")
                return True
            # 产品名
            product_name = argv[1]
            # 升级路径
            upgrade_path = argv[5]
            # 全量时间戳和增量时间戳的对应关系
            # 格式为backup_product_full_diff_{product_name}_{upgrade_path}: {full_timestamp}-{diff_timestamp}
            date_sec_cmd = int(time.time())
            full_diff_str = f"backup_product_full_diff_{product_name}_{upgrade_path}: " \
                            f"{backup_full_timestamp}-{backup_time_stamp}:" \
                            f"{date_sec_cmd}"
            with os.fdopen(os.open(BACKUP_TIME_STAMP, os.O_CREAT | os.O_WRONLY | os.O_APPEND,
                                   mode=0o640), 'a', encoding='utf-8') as w_stream:
                w_stream.write(f"{full_diff_str}{os.linesep}")
        except Exception as e:
            LOG.error(f"Write full-diff timestamp throw an exception: {e}")
            return False
        LOG.info(f"Write full-diff timestamp success, full-diff timestamp:{full_diff_str}.")
        return True

    def backup_product(self, argv):
        """
        功能说明: 备份总入口
        :param argv:
        :return:
        """
        product_name = argv[1]
        ssh_node = {
            "ip": argv[2],
            "username": argv[3]
        }
        omp_version = argv[4]
        upgrade_path = argv[5]
        remote_backup = argv[6]
        # 构造任务下发参数,并下发请求
        response = backup_product_func(ssh_node, product_name, omp_version,
                                       upgrade_path, remote_backup, self.get_backup_type())
        if response.get("status") != 200:
            return False

        # 查询请求结果，根据url获取任务执行进度
        back_url = response.get("url")
        if not self.query_backup_progress(back_url):
            return False

        backup_time_stamp = self.get_backup_time_stamp(back_url)

        # 记录备份url返回的备份时间戳到后台配置文件/opt/upgrade/backup/backup_timestamp.properties
        if not self.write_backup_timestamp_file(argv, backup_time_stamp) or \
                not self.write_full_diff_timestamp_file(argv, backup_time_stamp):
            return False
        return True

    def execute(self, argv):
        """
        功能说明:执行备份
        :param argv: 脚本入参
        :return:
        """
        # 检查数据库实例角色 入参__file__, product
        check_build_func(argv)

        # 备份产品数据
        if not self.backup_product(argv):
            return False

        # 关闭 BCT
        if self.action == BACKUP_DIFF_ACTION and not BackupProductData.gauss_db_bct(argv[1], "off"):
            return False
        return True


def main(argv):
    """
    功能说明：功能说明
    :return:
    """
    e_task_id = argv[8]
    backup_time_out = argv[9]
    backup_product_with_diff = argv[10]
    e_task_path = os.path.join(TASK_PATH, e_task_id)
    # 手工在管理面执行全量备份后，增量备份失效(走全量备份)
    backup_action = BACKUP_DIFF_ACTION if backup_product_with_diff == "true" else BACKUP_FULL_ACTION

    # 初始化任务
    Taskmgrutil().init_e_taskmgr(e_task_path)
    Taskmgrutil.set_task(e_task_path, status="RUNNING", progress="0")

    # 执行备份
    if not BackupProductWithDiff(e_task_path, backup_time_out, backup_action).execute(argv):
        # 执行失败
        Taskmgrutil.set_task(e_task_path, status="ERROR", progress="100")
        return False

    # 执行成功
    Taskmgrutil.set_task(e_task_path, status="FINISH", progress="100")
    return True


if __name__ == "__main__":
    if not main(sys.argv):
        sys.exit(1)
    sys.exit(0)
