# encoding=utf-8
"""
功 能：检查备份产品数据是否有效,无效直接进行备份操作
版权信息：华为技术有限公司，版本所有(C) 2020-2029
修改记录：2022-10-20 12:00 创建
脚本参数 :
# 1-product_name
# 2-ip
# 3-username
# 4-omp_version
# 5-upgrade_path
# 6-remote_backup
# 7-task_id
# 8-e_task_id
# 9-time_out
# 10-site
# 11-valid_time
"""
import os
import sys

from datetime import datetime

from backup_product_with_diff import BackupProductWithDiff
from upgrade_settings import BACKUP_TIME_STAMP, TASK_PATH
from taskmgr_util import Taskmgrutil
from commonlog import Logger

LOG = Logger().getinstance(sys.argv[0])


class BackupProduct(BackupProductWithDiff):
    def __init__(self, e_task_path, time_out=14400):
        super().__init__(e_task_path, time_out)
        self.action = "backup_product"


def get_backup_time_stamps():
    """
    功能说明: 读取备份数据时间戳并返回
    :return: 返回备份数据时间戳
    """
    if not os.path.isfile(BACKUP_TIME_STAMP):
        LOG.error(f"File({BACKUP_TIME_STAMP}) is not existed.")
        return list()
    with open(BACKUP_TIME_STAMP, 'r') as r_stream:
        file_data = r_stream.readlines()
    backup_time_stamps = list()
    for one_line in file_data:
        if not one_line or ':' not in one_line or one_line.count(':') < 1:
            continue
        backup_key = one_line.strip().split(':')[0].strip()
        backup_stamp = one_line.strip().split(':')[1].strip()
        if one_line.count(':') == 2:
            backup_finish_time = one_line.strip().split(':')[2].strip()
        else:
            backup_finish_time = str()
        backup_time_stamps.append([backup_key, backup_stamp, backup_finish_time])
    return backup_time_stamps


def get_backup_stamp(backup_key):
    """
    功能说明:返回最新匹配的时间戳数据进行返回
    :param backup_key: 备份key值
    :return:
    """
    backup_time_stamps = get_backup_time_stamps()
    backup_stamp = str()
    backup_finish_time = str()
    for time_stamp in backup_time_stamps:
        if backup_key == time_stamp[0]:
            # 遍历一直匹配到最后一个返回
            backup_stamp = time_stamp[1]
            backup_finish_time = time_stamp[2]
    return backup_stamp, backup_finish_time


def get_backup_product_data(*args):
    """
    功能说明:获取备份产品数据时间戳数据
    :param args:
    :return:
    """
    product = args[1]
    upgrade_path = args[5]
    key_name = f"backup_product_{product}_{upgrade_path}"
    # backup_product_NCE_R1C1-R2C2
    return get_backup_stamp(key_name)


def check_backup_product_valid(args):
    """
    功能说明:检查备份数据有效性
    :param args:可选参数
    :return:
    """
    valid_time = args[11]
    e_task_path = os.path.join(TASK_PATH, args[8])
    Taskmgrutil.set_task(e_task_path, msg="Start to check backup product data.")
    # backup_product_NCE_R1C1-R2C2
    backup_stamp, backup_finish_time = get_backup_product_data(*args)

    LOG.info(f"Time stamp:{backup_stamp}."
             f"Backup finish time:{backup_finish_time}."
             f"Backup data valid time:{valid_time}")
    if not backup_stamp or not backup_finish_time \
            or not backup_finish_time.isdigit() or not valid_time.isdigit():
        # 无对应时间戳,无结束时间记录,备份数据无效
        Taskmgrutil.set_task(e_task_path, msg=f"[WARNING] The Backup data is invalid. "
                                              f"No valid backup record found "
                                              f"in {BACKUP_TIME_STAMP}.")
        Taskmgrutil.set_task(e_task_path, msg="Back up product data again")
        return False

    # 检查备份时间戳结束时间和当前时间比对，备份数据是否在有效时间范围内
    valid_time_sec = int(valid_time) * 60
    if int(datetime.now().strftime("%s")) - int(backup_finish_time) > valid_time_sec:
        Taskmgrutil.set_task(e_task_path, msg=f"[WARNING] The Backup data is invalid. "
                                              f"The backup data exceededs more than {valid_time} "
                                              f"minutes and has expired.")
        Taskmgrutil.set_task(e_task_path, msg="Back up product data again")
        return False
    Taskmgrutil.set_task(e_task_path, msg=f"The Backup data is valid.")
    Taskmgrutil.set_task(e_task_path, msg=f"Finished to check backup product data. \n"
                                          f"Skip backing up the product data.")
    return True


def main(argv):
    e_task_id = argv[8]
    backup_time_out = argv[9]
    e_task_path = os.path.join(TASK_PATH, e_task_id)

    # 初始化任务
    Taskmgrutil().init_e_taskmgr(e_task_path)
    Taskmgrutil.set_task(e_task_path, status="RUNNING", progress="0")

    # 检查并且,执行备份
    if not check_backup_product_valid(argv) \
            and not BackupProduct(e_task_path, backup_time_out).execute(argv):
        # 执行失败
        Taskmgrutil.set_task(e_task_path, status="ERROR", progress="100")
        return False

    Taskmgrutil.set_task(e_task_path, status="FINISH", progress="100")
    return True


if __name__ == '__main__':
    if not main(sys.argv):
        sys.exit(1)
    sys.exit(0)
