# encoding=utf-8
"""
功 能：用于检查管理面是否存在正在运行的任务
版权信息：华为技术有限公司，版本所有(C) 2021-2029
修改记录：2021-07-5 12:00 创建
"""
import time
import json
import os
import sys
from datetime import datetime
from taskmgr_util import Taskmgrutil
from uniep_taskmgr import Unieptaskmgr
from commonlog import Logger

INFO = "INFO"
ERROR = "ERROR"
LOGGER = Logger().getinstance(sys.argv[0])


class CheckManagerStatus:

    def __init__(self, args):
        LOGGER.info("Init backup Uniep.")
        self.task_id = args.get("task_id", "")
        e_taskmgr_path = "/opt/upgrade/easysuite_upgrade/taskmgr"
        self.path = os.path.join(e_taskmgr_path, self.task_id)
        self.rest = Unieptaskmgr()
        self.taskmgr = Taskmgrutil()
        self.taskmgr.init_e_taskmgr(self.path)
        self.progress = 0

    def record_log(self, msg, level=INFO):
        """
        更新任务process、status、和日志
        :param msg:
        :param log_type:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        str_msg = "[%s] %s | %s" % (now_time, level, msg)
        self.taskmgr.set_msg_append(self.path, str_msg)
        if level == ERROR:
            self.taskmgr.set_e_taskstatus(self.path, "error")
            self.taskmgr.set_e_taskprogress(self.path, "100")
        return True

    def fresh_progress(self, progress):
        """
        根据输入的progress值，添加进度
        :param progress: int 增加的进度值
        """
        self.progress = self.progress + progress
        if self.progress > 100:
            self.progress = 100
        self.taskmgr.set_e_taskprogress(self.path, str(self.progress))

    def query_manager_info(self, tasks_list):
        running_list = []
        for num, one_task in enumerate(tasks_list):
            current_state = one_task.get('currentState')
            is_enabled = one_task.get('isEnabled')
            is_schedule_job = one_task.get('isScheduleJob')
            job_name = one_task.get('jobName')
            progress = one_task.get('progress')
            if current_state in ['RUNNING', 'TERMINATING']:
                running_list.append(
                    '{0} Name:{1} State:{2} Progress:{3}'.format(num + 1, job_name, current_state, progress))
                continue
            if current_state == "WAITING" and (
                    (is_schedule_job == "yes" and is_enabled == "yes") or is_schedule_job == "no"):
                running_list.append(
                    '{0} Name:{1} State:{2} Progress:{3}'.format(num + 1, job_name, current_state, progress))
        if running_list:
            return False, "\n".join(running_list)
        return True, ""

    def query_manager_status(self, time_out=120):
        """
        检查管理面任务状态，默认超时120分钟
        :param e_task_path:任务路径
        :param time_out:超时时间
        :return: True/False
        """
        start_time = int(time.time())
        self.record_log("Start to query the task status of the management plane. Timeout interval: {} min.".format(time_out))
        wait_timestamp = start_time + time_out * 60
        # 检查管理面任务分页接口
        query_omp_tasks_url = f'/rest/plat/omp/v1/main/common/taskmgr/pagingtasks?' \
                              'jobType=ui&page-length=20&page-size=1'
        count = 0
        while int(time.time()) < wait_timestamp:
            result, response = self.rest.send_get_request(query_omp_tasks_url)
            if not result or not response:
                self.record_log(
                    "Failed to query the task status on the management plane of the current site.", ERROR)
                return False
            omp_tasks = json.loads(response).get("taskList", [])
            result, out = self.query_manager_info(omp_tasks)
            if not result:
                if count == 0:
                    tips = self.taskmgr.get_res_dict().get("early_termination_tips")
                    self.taskmgr.set_msg_append(self.path, tips)
                    count = 1
                self.record_log("Tasks are running on the management plane. The task names are as follows:")
                self.taskmgr.set_msg_append(self.path, out)
                self.fresh_progress(1)
                time.sleep(10)
            else:
                self.record_log("No task is running on the management plane.")
                self.taskmgr.set_e_taskstatus(self.path, "success")
                self.taskmgr.set_e_taskprogress(self.path, "100")
                return True
        self.record_log("Waiting for completion of the management plane task times out.", ERROR)
        return False

def main(argv):
    """
    根据id和站点角色启动备份管理面
    :return:
    """
    if len(argv) < 2:
        print("Invalid parameter.")
        return False
    task_id = argv[1]
    if len(argv) == 3:
        time_out = argv[2]
    else:
        time_out = 120
        
    args = {}
    args.update({"task_id": task_id})
    check_function = CheckManagerStatus(args)
    if not check_function.query_manager_status(time_out=int(time_out)):
        return False
    return True

if __name__ == '__main__':
    main(sys.argv)