# encoding=utf-8
"""
功 能：记录日志
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import json
import os
import queue
import sys
import shlex
import subprocess
import threading
from datetime import datetime

from commonlog import Logger
from taskmgr_util import Taskmgrutil

logger = Logger().getinstance(sys.argv[0])
domain_script_path = "/opt/upgrade/easysuite_upgrade/scripts"
local_lang = Taskmgrutil.get_local_lang()
local_path = os.path.dirname(os.path.realpath(__file__))
error_code_file = os.path.join(os.path.dirname(local_path), "config/operate_check_error_code.json")


class OperateCheck:
    """
    功能描述:回滚健康检查，后续可以拓展其他步骤的检查
    入参：无
    """

    def __init__(self, script_id):
        """
        功能说明:初始化
        :param script_id:
        """
        self.check_msg = ""
        self.task_path = os.path.join("/opt/upgrade/easysuite_upgrade/taskmgr", script_id)
        self.task_mgr_function = Taskmgrutil()
        self.task_mgr_function.init_e_taskmgr(self.task_path)
        self.i18n_res = OperateCheck.read_i18n_res()

    @staticmethod
    def read_i18n_res():
        """
        功能说明:解析框架i18n国际化资源
        :return:
        """
        ret_res_dict = {}
        domain_path = os.path.join(domain_script_path, "common", "NCE-Common")
        domain_config_file = os.path.join(domain_path, "pyscripts", "easysuite_check.json")
        with open(domain_config_file, "r", encoding='utf-8') as file_obj:
            check_dict = json.load(file_obj)
            if check_dict.get("i18n", ""):
                for one_key in check_dict.get("i18n", ""):
                    ret_res_dict.update(
                        {one_key: check_dict.get("i18n", "").get(one_key).get(local_lang)})
        return ret_res_dict

    @staticmethod
    def read_check_config(action="rollback"):
        """
        功能描述:读取子域配置的检查脚本
        入参：无
        """
        check_list = []
        for one_domain in os.listdir(domain_script_path):
            domain_path = os.path.join(domain_script_path, one_domain)
            if one_domain == "common":
                domain_path = os.path.join(domain_script_path, "common", "NCE-Common")
            domain_config_file = os.path.join(domain_path, "pyscripts", "easysuite_check.json")
            if not os.path.isfile(domain_config_file):
                continue
            with open(domain_config_file, "r", encoding='utf-8') as file_obj:
                check_dict = json.load(file_obj)
                if not check_dict.get(action, ""):
                    continue
                # 子域存在对应配置项，直接覆盖框架配置
                check_list = check_dict.get(action, "")
                if one_domain != 'common':
                    break
        return check_list

    @staticmethod
    def skip_no_exist_script(check_list):
        """
        功能描述:过滤不存在的脚本
        入参：无
        """
        ret_check_list = []
        for one_check_script in check_list:
            script_abs_path = os.path.join(domain_script_path, one_check_script.get("script_name"))
            if os.path.isfile(script_abs_path):
                ret_check_list.append(one_check_script)
        return ret_check_list

    @staticmethod
    def execute_cmd(cmd, cmd_result):
        """
        功能描述：使用subprocess执行命令
        参数： cmd
        返回值: 返回码, 标准输出/标准错误
        """
        log_file_name = sys.argv[0].split("/")[-1]
        log_path = "/opt/oss/log/manager/easysuite_upgrade/%s.log" % log_file_name
        exe_cmd = "%s >>%s 2>&1" % (cmd, log_path)
        logger.info("cmd:{}".format(exe_cmd))
        process = subprocess.Popen(shlex.split(exe_cmd),
                                   shell=False, encoding='utf-8',
                                   stderr=subprocess.PIPE,
                                   stdout=subprocess.PIPE)
        stdout, stderr = process.communicate()
        ret_code = process.returncode
        logger.info(f"cmd:{exe_cmd}.stdout:{stdout}.stderr:{stderr}.ret_code:{ret_code}")
        cmd_result.put({"script_name": cmd, "code": ret_code})
        return ret_code

    def print_msg(self, msg):
        """
        功能说明:打印提示信息
        :param msg:
        :return:
        """
        now_time = datetime.strftime(datetime.now(), '%Y-%m-%d %H:%M:%S')
        str_msg = "[%s] [%s] %s" % (now_time, os.getpid(), msg)
        if self.check_msg == "":
            self.check_msg = str_msg
        else:
            self.check_msg = "%s\n%s" % (self.check_msg, str_msg)
        self.task_mgr_function.set_e_taskmsg(self.task_path, self.check_msg)
        return True

    def do_exit_error(self, action):
        """
        功能说明:失败退出
        :return:
        """
        Taskmgrutil.set_e_taskstatus(self.task_path, "error")
        Taskmgrutil.set_e_taskprogress(self.task_path, "100")
        with open(error_code_file, "r") as f:
            error_code_details = json.loads(f.read())
        error_code_detail = error_code_details.get(action, {})
        if error_code_detail:
            Taskmgrutil.set_e_task_errorcode(self.task_path, error_code_detail)
        return True

    def do_exit_finish(self):
        """
        功能说明:成功退出
        :return:
        """
        Taskmgrutil.set_e_taskstatus(self.task_path, "finish")
        Taskmgrutil.set_e_taskprogress(self.task_path, "100")
        return True

    def write_step_result(self, script_results, check_list, script_params, action):
        """
        功能说明:记录脚本执行结果
        :param script_results:
        :param check_list:
        :param script_params:
        :param action:
        :return:
        """
        is_fail = False
        # 1.描述  --成功
        for one_check in check_list:
            cmd = "bash %s" % os.path.join(domain_script_path, one_check.get("script_name"))
            for param in one_check.get("params", []):
                cmd += " " + script_params.get(param)
            for one_script_result in script_results:
                if one_script_result.get("script_name") != cmd:
                    continue

                ret_code = str(one_script_result.get("code"))
                if ret_code == "0":
                    self.print_msg(one_check.get("ret_msg").get(ret_code).get(local_lang))
                    continue

                if ret_code == "3" and "check_omp_task" in one_script_result.get("script_name"):
                    self.print_msg(one_check.get("ret_msg").get(ret_code).get(local_lang))
                    continue

                is_fail = True
                msg = one_check.get("ret_msg").get(ret_code)
                if msg:
                    self.print_msg("%s %s" % (self.i18n_res.get("easysuite.check.suggest"),
                                              msg.get(local_lang)))
        if is_fail:
            self.do_exit_error(action)
            return False
        self.do_exit_finish()
        return True

    def step_check(self, action="rollback", script_params=None):
        """
        功能说明:执行健康检查
        :param action:
        :param script_params:
        :return:
        """
        cmd_result = queue.Queue()
        check_list = OperateCheck.read_check_config(action)
        not_skip_check_list = OperateCheck.skip_no_exist_script(check_list)
        child_task_threads = []
        for one_check in not_skip_check_list:
            self.print_msg(one_check.get("check_name").get(local_lang))
            cmd = "bash %s" % os.path.join(domain_script_path, one_check.get("script_name"))
            for param in one_check.get("params", []):
                cmd += " " + script_params.get(param)
            child_task_threads.append(
                threading.Thread(target=OperateCheck.execute_cmd, args=(cmd, cmd_result,)))
        for child_task_thread in child_task_threads:
            child_task_thread.start()
        for child_task_thread in child_task_threads:
            child_task_thread.join()
        thread_result = []
        for _ in range(0, cmd_result.qsize()):
            q_result = cmd_result.get()
            if q_result:
                thread_result.append(q_result)
        self.write_step_result(thread_result, not_skip_check_list, script_params, action)


def main(argv):
    """
    功能说明:主入口
    :param argv:
    :return:
    """
    action = argv[1]
    script_id = argv[2]
    src_version = argv[3]
    upgrade_path = argv[4]
    script_params = {
        "script_id": script_id,
        "src_version": src_version,
        "upgrade_path": upgrade_path
    }
    if not action:
        action = "rollback"
    if not script_id:
        return False
    check_function = OperateCheck(script_id)
    check_function.step_check(action, script_params)
    return True


if __name__ == '__main__':
    main(sys.argv)
