# -*- coding:utf-8 -*-
"""
功 能：检查指定微服务是否正在运行
版权信息：华为技术有限公司，版本所有(C) 2021-2029
修改记录：2021-07-20 17:00 创建
"""
import json
import time
import sys

from commonlog import Logger
from uniep_taskmgr import Unieptaskmgr

logger = Logger().getinstance(sys.argv[0])

class CheckTools:
    @staticmethod
    def get_abnormal_services(process_list, status):
        """
        返回待过滤列表的状态是否都符合要求
        :param process_list:待确认列表
        :param status:目标状态列表
        :return:False,不匹配的列表。True，空列表
        """
        abnormal_process_list = []
        for one_process in process_list:
            if one_process.get("status") not in status:
                logger.info("The process is not in %s,detail: %s" % (status, one_process))
                abnormal_process_list.append(one_process.get("appName") + ",nodeIP:" + one_process.get("nodeIP"))
        if abnormal_process_list:
            return False, set(abnormal_process_list)
        return True, []

    @staticmethod
    def get_deploy_services(app_service_list, product_name):
        """
        获取指定产品，指定app列表中的服务状态
        :param app_service_list: 服务查询列表
        :param product_name: 产品名
        :return: True/False: 查询是否成功，process_list指定app状态的列表（不存在的app自动跳过）
        """
        taskmgr_request = Unieptaskmgr()
        query_app_process_url = '/rest/plat/productmonitorservice/v1/main/service/expstatus' \
                                '?productName=%s&appName=%s'
        status, response = taskmgr_request.send_get_request(
            query_app_process_url % (product_name, ",".join(app_service_list)))
        if not status:
            return False, []
        process_list = CheckTools.get_process_list_from_response(response)
        return True, process_list

    @staticmethod
    def get_process_list_from_response(response):
        """
        根据接口返回值获取服务状态，支持针对服务进行定制
        :param response:
        :return:
        """
        process_list = []
        hiroberservice_list = []
        # 判断定制服务是否需要加入返回值的标志
        need_return_flag = True
        result_dict = json.loads(response).get("result")
        status_dict = json.loads(result_dict).get("statuslist")
        for item in status_dict:
            # 针对HIROBERService进行特殊处理
            if item.get("appName") == "HIROBERService":
                hiroberservice_list.append(item)
            else:
                process_list.append(item)
            # 针对HIROBERService进行特殊处理，当子项中存在"RUNNING"状态时，所有服务项就不加入返回值
            if item.get("appName") == "HIROBERService" and item.get("status") == "RUNNING":
                need_return_flag = False
        if need_return_flag:
            process_list.extend(hiroberservice_list)
        return process_list

    @staticmethod
    def check_app_status(app_service_list, product_name, try_count=3, action="start"):
        """
        检查指定产品下，指定app列表的状态是否符合要求
        :param app_service_list: 待检查app列表
        :param product_name:产品名
        :param try_count:重试次数
        :param action:启动或失败
        :return: True/False,不符合要求的列表。
        """
        status = ["STOPPED"]
        if action == "start":
            status = ["RUNNING", "STOPPED(STANDBY)"]
        out_list = []
        process_list = []
        for time_count in range(0, try_count):
            result, process_list = CheckTools.get_deploy_services(app_service_list, product_name)
            if not result:
                logger.info("Failed to obtain the service list.")
                time.sleep(10)
                continue
            result, out_list = CheckTools.get_abnormal_services(process_list, status)
            if result:
                logger.info("True;Process_list:%s" % process_list)
                return True, []
        logger.info("False;Process_list:%s" % process_list)
        return False, out_list


    @staticmethod
    def check_necessary_services(product_name):
        """
        查询产品必要的服务是否在运行，如果服务不存在，会自动跳过
        :param product_name: 产品名称
        :return: True/False
        """
        app_service_list = ['RouterAgent', 'ServiceAwareWatchAgent', 'DRMgrAgent',
                            'DBAgent', 'CloudbProxyService', 'Etcd', 'HIROIRService',
                            'SSAMgrWebsite', 'DBProxyService', 'ServiceCenter', 'OMMHAService',
                            'HIROBERService', 'BusService', 'UniEPAgent', 'NodeAgent']
        return CheckTools.check_app_status(app_service_list, product_name, try_count=3, action="start")

def main(argv):
    if len(argv) < 2:
        print("Please enter a product name.")
        sys.exit(1)
    productname = argv[1]
    result, out_list = CheckTools.check_necessary_services(productname)
    if not result:
        print(out_list)
        sys.exit(1)
    sys.exit(0)

if __name__ == '__main__':
    main(sys.argv)