#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import json
import os
import shutil
import sys

from commonlog import Logger
from util import httpclient

DR_MGR_URL = "/rest/drmgrservice/v1/main/drmgr"
DR_ARB_URL = "/rest/drservice/v1/main/drmgr/custom/queryParam"

LOG = Logger().getinstance(sys.argv[0])


def update_dr_info_file(dr_info_file):
    """
    获取容灾信息
    """
    status, response = httpclient.IRHttpClient().get(DR_MGR_URL)
    if status != 200:
        LOG.warning(
            "[update_dr_info_file] Failed to invoke interface. retCode: %s." % str(status))
        return False
    if not json.loads(response).get("data"):
        LOG.warning("[update_dr_info_file] Response is null.")
        return False
    drs_info = json.loads(response)["data"][0]
    if not drs_info:
        return False

    data = get_dr_config_info(dr_info_file, drs_info)
    LOG.info("[update_dr_info_file] drinfo.json queryed python: %s" % str(data))

    with os.fdopen(os.open(dr_info_file,
                           os.O_CREAT | os.O_WRONLY | os.O_TRUNC,
                           mode=0o660), 'w', encoding='utf-8') as json_file:
        json_file.write(json.dumps(data))
    return True


def get_dr_config_info(dr_info_file, drs_info):
    """
    功能说明:获取容灾配置数据
    :param dr_info_file:
    :param drs_info:
    :return:
    """
    dr_rep_direction = []
    primary_products_name = []
    secondary_products_name = []
    primary_site_name = drs_info.get("primarySiteName")
    secondary_site_name = drs_info.get("secondarySiteName")
    primary_site_ip = drs_info.get("primaryHBIPs")
    secondary_site_ip = drs_info.get("secondaryHBIPs")
    master_site_ip = ";".join(primary_site_ip)
    dr_site_ip = ";".join(secondary_site_ip)
    products = drs_info.get("products")
    for product in products:
        primary_product_name = product.get("primaryProductName")
        primary_products_name.append(primary_product_name)
        secondary_product_name = product.get("secondaryProductName")
        secondary_products_name.append(secondary_product_name)
        primary_product_status = product.get("primaryProductStatus")
        if primary_product_status == "standby":
            rep_direction = "PRIMARY_TO_SECONDARY"
        else:
            rep_direction = "SECONDARY_TO_PRIMARY"
        dr_rep_direction.append(rep_direction)
    installed_time = get_install_time()
    shutil.copyfile(dr_info_file, "%s.bak" % dr_info_file)
    with open("%s.bak" % dr_info_file, 'r', encoding='utf-8') as file_dr:
        data = json.load(file_dr)
    data["drinfo"]["primarySiteProductName"] = primary_products_name
    data["drinfo"]["secondarySiteProductName"] = secondary_products_name
    data["drinfo"]["primarySiteName"] = primary_site_name
    data["drinfo"]["primarySiteIP"] = primary_site_ip
    data["drinfo"]["secondarySiteName"] = secondary_site_name
    data["drinfo"]["secondarySiteIP"] = secondary_site_ip
    data["drinfo"]["repDirection"] = dr_rep_direction
    ext_info = data["drinfo"]["extInfo"][0]["info"]
    dr_ext_config = {"masterSiteIP": master_site_ip,
                     "drSiteIP": dr_site_ip,
                     "primarySiteName": primary_site_name,
                     "secondarySiteName": secondary_site_name,
                     "installedTime": installed_time}
    for k_key in dr_ext_config:
        for info in ext_info:
            if info["key"] == k_key:
                info["value"] = dr_ext_config[k_key]
    return data


def get_install_time():
    """
    功能描述：获取安装时间
    返回： 安装时间
    修改记录：新增方法
    """
    url = DR_ARB_URL
    params = {'isEdit': 'false'}
    status, response = httpclient.IRHttpClient().post(url, params)
    if status != 200:
        LOG.warning(
            "[get_install_time] Failed to invoke interface. retCode: %s." % str(status))
        return False
    custom_info = json.loads(response)
    installed_time = custom_info.get("installedTime")
    return installed_time


def main(argv):
    """
    容灾主函数
    :param argv:必要入参
    :return:
    """
    dr_info_file = argv[1]

    return update_dr_info_file(dr_info_file)


if __name__ == '__main__':
    main(sys.argv)
