# encoding=utf-8
"""
功 能：错误码框架
版权信息：华为技术有限公司，版本所有(C) 2019-2029
修改记录：2019-12-11 12:00 创建
"""
import os
import sys

from commonlog import Logger
from os_utils import OsUtils

LOG = Logger().getinstance(sys.argv[0])
FILE_PATH = os.path.dirname(os.path.realpath(__file__))


class ErrorCode:
    def __init__(self):
        pass

    @staticmethod
    def check_error_details(error_details):
        """
        功能说明:检查错误码详情是否满足要求
        :param error_details: 错误码详情
        :return: True 正确, False 错误
        """
        # 1.入参是否合规
        if not error_details or not isinstance(error_details, dict):
            LOG.error(f"Parsing error|Params({error_details}) of check_error_details is invalid.")
            return False

        # 2.有关键字段在error_details中不存在代表关键字段缺失
        require_keys = ["error_code", "details", "reason", "suggest", "resource"]
        check_diff = set(require_keys).difference(set(list(error_details.keys())))
        if check_diff:
            LOG.error(f"Parsing error|Params({error_details})."
                      f"Require key is not exists:{check_diff}")
            return False
        return True

    @staticmethod
    def get_default_details():
        """
        功能说明:读取工具默认错误详情信息
        :return: default_details 错误码详情
        """
        default_details_file = os.path.join(os.path.dirname(FILE_PATH), 'config',
                                            'default_es_error_code.json')
        default_details = OsUtils.read_json(default_details_file)
        suggest = default_details.get('suggest', dict())
        log_file = LOG.logfile.replace('/opt/oss', '{OSS_ROOT}')
        suggest.update({'en_US': suggest.get('en_US', str()).replace('{LOG_FILE}', log_file),
                        'zh_CN': suggest.get('zh_CN', str()).replace('{LOG_FILE}', log_file)})
        return default_details

    @staticmethod
    def read_error_path_details(error_path):
        """
        功能说明:读取error_path数据
        :param error_path: 错误码路径
        :return: error_details 错误码详情
        """
        default_details = ErrorCode.get_default_details()
        error_details = {}
        for error_json in os.listdir(error_path):
            if not error_json.endswith('.json'):
                continue
            json_data = OsUtils.read_json(os.path.join(error_path, error_json))
            if not json_data or not ErrorCode.check_error_details(json_data):
                error_details.update({'easysuite': default_details})
                continue
            error_details.update({error_json: json_data})
        return error_details

    @staticmethod
    def set_task_error_details(task_path: str, error_path: list):
        """
        功能说明:汇聚升级任务错误码详情信息
        :param task_path: 任务工作目录
        :param error_path: 错误码文件保存目录列表
        :return:
        """
        # 入参类型校验
        if not isinstance(error_path, list):
            return

        # 读取目录下错误码汇总详情信息
        error_details = dict()
        for path_name in error_path:
            if not os.path.isdir(path_name):
                continue
            error_details.update(ErrorCode.read_error_path_details(path_name))

        # 无错误码详情信息直接返回
        if not error_details:
            return

        error_details_file = os.path.join(task_path, 'task_error.json')
        OsUtils.write_json(error_details_file, error_details)

        # 给sop用户添加权限保证文件可下载
        OsUtils.set_f_acl(error_details_file, user='sopuser', permit='r')

        # 清理临时目录,汇聚错误码详情后删除
        for path_name in error_path:
            OsUtils.rm(path_name)

    @staticmethod
    def error_path(work_path, pid=None):
        """
        功能说明:错误码存放目录
        :param work_path: 工作路径
        :param pid: 差异化ID
        :return:
        """
        return f"{work_path}/error_path/{pid if pid else os.getpid()}"

    @staticmethod
    def initial_error_path(work_path, pid=None):
        """
        功能说明:以线程ID和work_path作为路径区分
        :param work_path: 工作路径
        :param pid: 差异化ID
        :return:
        """
        path = ErrorCode.error_path(work_path, pid)
        if os.path.exists(path):
            # 目录存在清除
            OsUtils.rm(path)
        # 创建全新的目录
        OsUtils.mkdir(path)

    @staticmethod
    def convert_cloudsop_error_codes(task_path, error_code_file):
        """
        转换平台管理面和业务面升级错误码
        :param task_path:
        :param error_code_file:
        :return:
        """
        try:
            # 平台错误码详情
            cloudsop_error_codes = OsUtils.read_json(error_code_file)
            LOG.info(f"error_code_file: {error_code_file}, details: {cloudsop_error_codes}")
            if not cloudsop_error_codes:
                return

            # 转换错误码格式
            count = 0
            es_error_code_details = {}
            for cloudsop_error_code in cloudsop_error_codes:
                es_error_code = ErrorCode.format_es_error_code(cloudsop_error_code, count)
                if not es_error_code:
                    continue
                es_error_code_details.update(es_error_code)
                count += 1

            if not es_error_code_details:
                return

            error_details_file = os.path.join(task_path, 'task_error.json')
            OsUtils.write_json(error_details_file, es_error_code_details)

            # 给sop用户添加权限保证文件可下载
            OsUtils.set_f_acl(error_details_file, user='sopuser', permit='r')
        except Exception as e:
            LOG.error(f"Failed to format CloudSop error code, exception: {e}")

    @staticmethod
    def format_es_error_code(cloudsop_error_code, count):
        """
        将平台错误码转成ES格式
        :param cloudsop_error_code:
        :param count:
        :return:
        """
        # 平台错误码为空
        if not cloudsop_error_code.get("code", ""):
            return {}

        es_error_code = {
            count: {
                "error_code": cloudsop_error_code.get("code", ""),
                "details": {
                    "zh_CN": cloudsop_error_code.get("message", ""),
                    "en_US": cloudsop_error_code.get("message", "")
                },
                "reason": {
                    "zh_CN": cloudsop_error_code.get("reason", ""),
                    "en_US": cloudsop_error_code.get("reason", "")
                },
                "suggest": {
                    "zh_CN": cloudsop_error_code.get("suggest", ""),
                    "en_US": cloudsop_error_code.get("suggest", "")
                },
                "resource": {
                    "zh_CN": "CloudSop",
                    "en_US": "CloudSop"
                }
            }
        }
        return es_error_code


class ShErrorCode(ErrorCode):
    """
    功能说明:对外提供Shell直接调度总入口
    """
    @staticmethod
    def error_path(work_path, pid=None):
        """
        功能说明:提供Shell直接调用的error_path方法
        :param work_path:
        :param pid:
        :return:
        """
        print(ErrorCode.error_path(work_path, pid))

    @staticmethod
    def set_task_error_details(task_path: str, error_path: str):
        """
        功能说明:提供Shell直接调用的set_task_error_details方法
        :param task_path:
        :param error_path:
        :return:
        """
        ErrorCode.set_task_error_details(task_path, error_path.split(','))

    @staticmethod
    def convert_cloudsop_error_codes(task_path: str, error_code_file: str):
        """
        功能说明:提供Shell直接调用的convert_cloudsop_error_codes方法
        :param task_path:
        :param error_code_file:
        :return:
        """
        ErrorCode.convert_cloudsop_error_codes(task_path, error_code_file)


if __name__ == "__main__":
    params = sys.argv
    action = params[1]
    params.remove(action)
    params.remove(params[0])
    if hasattr(ShErrorCode, action):
        getattr(ShErrorCode, action)(*params)