#!/bin/bash

# 脚本名
SELF_FILE=$(basename $0)
# SSO认证配置文件路径
CONFIG_FILE="/opt/oss/easysuite/easysuite/easysuite/easysuite/config/upgrade_config.properties"

# 检查脚本执行用户
user=$(whoami)
if [ "${user}" != "ossadm" ]
then
    echo "The script must be run as the ossadm user."
    exit 1
fi

##################################
# 功能说明: 记录系统日志
##################################
function local_logger()
{
    local result="${1}"
    local action="${2}"
    if [ ${result} -eq 0 ]
    then
        logger -t ${USER} -p local0.info "${SELF_FILE}:action ${action} is finished." &>/dev/null
    else
        logger -t ${USER} -p local0.error "${SELF_FILE}:action ${action} is failed." &>/dev/null
    fi

}

##################################
# 功能说明: 打开SSO认证开关
##################################
function open_sso()
{
    sed -i "s/^SSO_SWITCH=.*/SSO_SWITCH=ON/g" "${CONFIG_FILE}"
    result=$?
    local_logger "${result}" "open_sso"
}

##################################
# 功能说明: 关闭SSO认证开关
##################################
function close_sso()
{
    sed -i "s/^SSO_SWITCH=.*/SSO_SWITCH=OFF/g" "${CONFIG_FILE}"
    result=$?
    local_logger "${result}" "close_sso"
}

##################################
# 功能说明: 设置SSO服务端IP
##################################
function set_sso_server()
{
    local sso_server="${1}"
    if [ -z "${sso_server}" ]
    then
        local_logger "1" "set_sso_server(sso_server is empty)"
        return 1
    fi

    sed -i "s/^SSO_SERVER_IP_PORT=.*/SSO_SERVER_IP_PORT=${sso_server}/g" "${CONFIG_FILE}"
    result=$?
    local_logger "${result}" "set_sso_server"
}

##################################
# 功能说明: 脚本使用帮助
##################################
function usage()
{
    echo "bash ${SELF_FILE} -action close_sso/open_sso/set_sso_server [-sso_server XXX.XXX.XXX.XXX]"
}

##################################
# 功能说明: 检查IP地址合规性
##################################
function check_ip()
{
    local ip_addr="${1}"
    # 待补充校验
    return 0
}

##################################
# 功能说明: 检查入参
##################################
function check_args()
{
    if [ $# -ne 2 -a $# -ne 4 ]
    then
        usage
        return 1
    fi

    # 检查入参一是否合规
    if [ -n "${1}" -a ${1} != "-action" ]
    then
        usage
        return 1
    fi

    # 检查入参二是否合规
    for scense in $(echo "close_sso open_sso set_sso_server" | xargs)
    do
        if [ ${scense} == "${2}" ]
        then
            break
        fi
    done

    if [ $# -eq 2 ]
    then
        # 两个参数场景下,非法入参
        if [ "${2}" == "set_sso_server" ]
        then
            usage
            return 1
        fi
    fi

    if [ $# -eq 4 ]
    then
        # 检查入参三是否合规
        if [ "${3}" != "-sso_server" ]
        then
            usage
            return 1
        fi

        # 检查入参四是否合规
        check_ip "${4}" && return 0
        usage
        return 1
    fi

}

##################################
# 功能说明: 主入口
#################################
function main()
{
    check_args $@ || return 1

    case ${scense} in
    "close_sso")
    close_sso
    ;;
    "open_sso")
    open_sso
    ;;
    "set_sso_server")
    set_sso_server "${4}"
    ;;
    esac
}

main $@
result=$?
[ ${result} -eq 0 ] && echo "success"
[ ${result} -ne 0 ] && echo "failed"
exit ${result}
