#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
# File name: start.sh
# Description: the shell to start easysuite

# 获取脚本工作路径
real_path=$(readlink -f ${0})
if [ $? -eq 0 ]
then
    base_path=$(cd $(dirname ${real_path}); pwd)
else
    base_path=$(cd $(dirname ${0}); pwd)
fi

# 获取脚本名
script_name=$(basename ${0})
# 切换工作路径到当前脚本所在路径
cd ${base_path}

############################################
# 功能说明: 解析脚本入参
############################################
function get_script_params()
{
    [ $# -eq 1 ] && solution_set="" || solution_set="${1}"
}

############################################
# 功能说明: 检查执行用户
############################################
function check_user_permit()
{
    user=$(whoami)
    if [ "${user}" != "ossadm" ]
    then
        echo "The script must be run as the ossadm user."
        exit 1
    fi
}

################################################################
# 功能说明: 补充加载操作系统环境变量,sudo缺少SSH_AUTH_SOCK环境变量
# SSH_AUTH_SOCK环境变量缺失会影响fabric站点内免密跳转
################################################################
function load_sys_environment()
{
    # Linux中，~是一个特殊的字符，表示当前用户的家目录。它不是一个变量，也不是一个可执行程序,不能当变量使用
    user_home=~
    # 操作系统默认/etc/profile内部会加载/etc/bashrc ~/.bash_profile会加载~/.bashrc
    sys_file="/etc/profile ${user_home}/.bash_profile"
    for file_name in ${sys_file}
    do
        [ -f "${file_name}" ] && source "${file_name}"
    done
}

############################################
# 功能说明: 设置脚本执行环境变量
############################################
function set_environment()
{
    current_path="${base_path}"
    tmp_access_ip_path="${current_path}/easysuite/var/temp/access_ip"
    python_path=$current_path'/easysuite/3rdparty/Python39'
    db_path=$current_path'/easysuite/db'
    certs_path=$current_path'/easysuite/certs'
    python_lib_path=$current_path'/easysuite/3rdparty/Python39/lib'
    sqlite_path=$current_path'/easysuite/3rdparty/sqlite3/lib'
    zip_tool_path=$current_path'/easysuite/3rdparty/7zip'
    verify_tool_path=$current_path'/easysuite/3rdparty/VerificationTools'
    easysuite_admin_key=$current_path'/var/.easysuite/admin_private.key'
    utils_path=$current_path'/easysuite/utils'
    zip_process_path=$current_path'/easysuite/utils/zip_process.py'
    zip_process_pyc_path=$current_path'/easysuite/utils/zip_process.pyc'
    unzip_cloudsop_py=$current_path'/easysuite/utils/unzip_cloudsop.py'
    unzip_cloudsop_pyc=$current_path'/easysuite/utils/unzip_cloudsop.pyc'
    expansion_path=$current_path'/var/software/escompontent'
    restart_flag=$current_path'/restart.flag'
    # 启动日志
    LOG_FILE="${current_path}/var/run/log/start.log"
    # 目录不存在创建
    [ ! -d "$(dirname ${LOG_FILE})" ] && mkdir -p "$(dirname ${LOG_FILE})"
    export LD_LIBRARY_PATH=$python_lib_path:$sqlite_path:$LD_LIBRARY_PATH

    python_bin_path=$current_path'/easysuite/3rdparty/Python39/bin/python3.9'
    python_start_file_path_py=$current_path'/easysuite/manage.py'
    python_start_file_path_pyc=$current_path'/easysuite/manage.pyc'
    python_os_reboot_path_py=$current_path'/easysuite/reboot_info.py'
    python_os_reboot_path_pyc=$current_path'/easysuite/reboot_info.pyc'
    python_start_file_path=''
}

#######################################################
# 功能说明: 重置脚本环境变量并以更纯粹的环境变量执行脚本
#######################################################
function reset_env_and_exe()
{
    if [ "${!#}" == "internal_call" ]
    then
        # 解析脚本入参
        get_script_params "$@"
    else
        # 借助sudo命令精简环境变量,避免环境变量污染
        sudo -u ossadm bash ${base_path}/${script_name} $@ internal_call </dev/null
        exit $?
    fi
}

############################################
# 功能说明: 解压3rdparty中的cloudsop组件包
############################################
function unzip_cloudsop()
{
    echo Start unzip osconfig and sudosconfig package ...
    chmod -R 750 "${python_path}" &>/dev/null
    if [ -f "$utils_path/unzip_cloudsop.py" ] || [ -f "$utils_path/unzip_cloudsop.pyc" ]
    then
        nohup $python_bin_path -c "import sys;sys.path.append('${utils_path}');import unzip_cloudsop; unzip_cloudsop.unzip_cloudsop()" --noreload >/dev/null 2>&1
        if [ $? -ne 0 ]
        then
            echo "Failed to unzip osconfig and sudoconfig package.Please check the current software package architecture."
            exit 1
        fi
        echo "unzip osconfig and sudoconfig package succeeded."
    else
        echo "Start failed, the file $current_path/easysuite/utils/unzip_cloudsop.py^(c^) does not exist!"
        exit 1
    fi
}

############################################
# 功能说明: 清理EasySuite tmp目录下升级临时文件
############################################
function clear_upgrade_tmp_taskmgr()
{
    # 最多保留目录个数
    max_file_num=80
    # MB, 不能小于1
    max_size=8
    # 最小保留目录个数
    min_file_num=8
    
    # 1.清理临时日志汇总目录 ${base_path}/var/temp/upgrade/taskmgr
    upgrade_tmp_path="${base_path}/var/temp/upgrade/taskmgr"
    if [ -d "${upgrade_tmp_path}" ]
    then
        dir_num=$(ls -l "${upgrade_tmp_path}"| wc -l)
        if [ ${dir_num} -gt ${max_file_num} ]
        then
            ls -tr "${upgrade_tmp_path}" | head -n $(expr ${dir_num} - ${max_file_num}) | xargs -i echo "'{}'" | xargs rm -rf
        fi
        # 大小不满足的直接按倍数删减, 最少保留
        dir_num=$(ls -l "${upgrade_tmp_path}"| wc -l)
        path_size=$(du -shm "${upgrade_tmp_path}" | awk '{print $1}')
        if [ ${path_size} -gt ${max_size} ]
        then
            multiple=$(expr ${path_size} / ${max_size})
            save_file_num=$(expr ${dir_num} / ${multiple})
            if [ ${save_file_num} -gt ${min_file_num} ]
            then
                ls -tr "${upgrade_tmp_path}" | head -n $(expr ${dir_num} - ${save_file_num}) | xargs -i echo "'{}'" | xargs rm -rf
            else
                ls -tr "${upgrade_tmp_path}" | head -n $(expr ${dir_num} - ${min_file_num}) | xargs -i echo "'{}'" | xargs rm -rf
            fi
        fi
    fi
}

############################################
# 功能说明: 清理EasySuite tmp目录下临时文件
############################################
function clear_tmp_file()
{
    # 1.清理升级临时文件
    clear_upgrade_tmp_taskmgr
}

#######################################
# 功能说明: 加载Solution
#######################################
function start_plugin_prepare_solution()
{

    echo Start loading Solution package ...
    if [ -f "$utils_path/start_plugin_prepare.py" ]
    then
        nohup $python_bin_path -c "import sys;sys.path.append('${utils_path}');import start_plugin_prepare; start_plugin_prepare.start_plugin_prepare()" --noreload >/dev/null 2>&1
    elif [ -f "$utils_path/start_plugin_prepare.pyc" ]
    then
        nohup $python_bin_path -c "import sys;sys.path.append('${utils_path}');import start_plugin_prepare; start_plugin_prepare.start_plugin_prepare()" --noreload >/dev/null 2>&1
    else
        echo "Start failed, the file $current_path/easysuite/utils/start_plugin_prepare.py^(c^) does not exist!"
        exit 1
    fi
    
    a=$?
    if [ ${a} -eq 2 ]
    then
        echo "Unzip Solution or Migrate zipfile failed, please check whether the zipfile is damaged or current path is too long."
        exit 2
    elif [ ${a} -eq 3 ]
    then
        echo "echo Copy template files to $current_path/'easysuite/plugin/' failed. because the current path is too long"
        exit 3
    elif [ ${a} -eq 4 ]
    then
        echo "Zip '*DeployTemplate.zip' or '*Expansion.zip' or 'Migrate.zip' failed."
        exit 4
    elif [ ${a} -eq 5 ]
    then
        echo "Solution or ESMigrate install package signature check failed."
        exit 5
    elif [ ${a} -eq 6 ]
    then
        echo "Make path '/var/software/Solution' or '/var/software/Migrate' failed."
        exit 6
    elif [ ${a} -eq 8 ]
    then
        echo "Solution or ESMigrate signature file cms or crl does not exist."
        exit 8
    elif [ ${a} -eq 0 ]
    then
        echo "Loading Solution product_apps success."
    fi
    if [ -d "${expansion_path}" ]
    then
        [ -d $expansion_path'/software/install/install_script' ] && find $expansion_path'/software/install/install_script' -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod --silent 550
    fi

    if [ -d "${var_path}" ]
    then
        find $var_path -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod --silent 550 2>/dev/null
        find $var_path -type f -name "*.json" | xargs -i echo '"{}"' | xargs chmod --silent 600 2>/dev/null
    fi

    plugin_path=$current_path'/easysuite/plugins'
    if [ -d "$plugin_path" ]
    then
        find $plugin_path -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod --silent 550 2>/dev/null
        find $plugin_path -type f -name "*.json" | grep -v site-package | cut -d : -f 1 |xargs chmod --silent 600 2>/dev/null
    fi
    if [ -d "$plugin_path" ]
    then
        find $plugin_path -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod --silent 550 2>/dev/null
        find $plugin_path -type f -name "*.json" | grep -v site-package | cut -d : -f 1 |xargs chmod --silent 600 2>/dev/null
    fi


    if [ -f "$easysuite_admin_key" ]
    then
        chmod -R --silent 600 $easysuite_admin_key
    fi
}

#######################################
# 功能说明: 生成证书
#######################################
function generate_certificate()
{
    if [ -f "$utils_path/cert/create_certificate.py" ] || [ -f "$utils_path/cert/create_certificate.pyc" ]
    then
        $python_bin_path -c "import sys; sys.path.append('${utils_path}'); sys.path.append('$current_path/easysuite'); from cert.create_certificate import CreateCertificate; CreateCertificate.exe()" &>>"${LOG_FILE}"
        if [ $? -ne 0 ]
        then
            echo "Failed to generate the self-signed certificate."
        fi
        find $current_path'/easysuite/certs' -type f -name "*.pem" -or -name "*.crt"| xargs -i echo '"{}"' | xargs chmod --silent 600
        # 刷新config文件和目录权限
        chmod --silent 600 ${current_path}/easysuite/easysuite/config/* &>/dev/null
        chmod --silent 700 ${current_path}/easysuite/easysuite/config &>/dev/null
        echo "Generated from signature certificate succeeded."
    else
        echo "Start failed, the file $current_path/easysuite/utils/cert/create_certificate.py^(c^) does not exist!"
        exit 1
    fi
}

#######################################
# 功能说明: 更新密钥
#######################################
function replace_key()
{
    if [ -f "$current_path/easysuite/replace_key.py" ]
    then
        replace_key_file="$current_path/easysuite/replace_key.py"
    elif [ -f "$current_path/easysuite/replace_key.pyc" ]
    then
        replace_key_file="$current_path/easysuite/replace_key.pyc"
    else
        echo "Start failed, the file $current_path/easysuite/replace_key.py^(c^) does not exist!"
        exit 1
    fi
 
    expect <<END >/dev/null 2>&1
        spawn $python_bin_path ${replace_key_file}
        set timeout -1
        expect {
        "*(Y/N)*" {send "Y\r";exp_continue}
        }
END
 
    if [ $? -ne 0 ]
    then
        echo "first start replace key failed."
        exit 1
    fi
    echo "first start replace key succeeded."
}

#######################################
# 功能说明: 检查IP地址是否合规
#######################################
function check_ip()
{
    local target_ip=$1
    VALID_CHECK=$(echo $target_ip|awk -F. '$1<=255&&$2<=255&&$3<=255&&$4<=255{print "yes"}')  
    if echo $target_ip|grep -E "^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$" >/dev/null
    then
        if [ $VALID_CHECK == "yes" ]
        then
            return 0 
        else
            return 1
        fi
    else  
        return 1  
    fi    
}

##############################
# 功能说明: 打印EasySuite进程信息
#############################
function print_es_processs()
{
    echo ---------------------------------------------------------------------------------------
    echo 'WARNING | Easysuite is running already.Please run stop.sh to stop the easysuite first.'
    es_pid=$(netstat -anp 2>/dev/null | grep "tcp.*${ES_PORT}.*LISTEN" | awk '{print $NF}' | awk -F/ '{print $1}')
    if [ -n "${es_pid}" -a "${es_pid}" != "-" ]
    then
        process_data=$(ps -ef | fgrep ${es_pid} | grep -v grep | xargs)
        echo "WARNING | Process run user: $(echo ${process_data} | awk '{print $1}')"
        echo "WARNING | Process pid: $(echo ${process_data} | awk '{print $2}')"
        echo "WARNING | Process start time: $(echo ${process_data} | awk '{print $5}')"
        echo "WARNING | Easysuite: $(echo ${process_data} | awk '{print $8}')"
    fi
    echo ---------------------------------------------------------------------------------------
}

################################################
# 功能说明: 保持原有IP选择机制,自动选择第一个IP
###############################################
function select_listen_address()
{
    ip_arr=()
    # Obtain the IP address of the environment, excluding the IPv6 local link address.
    for i in $(ip address show | grep inet | grep -v 127.0.0 | grep -v "scope link" | awk '{print $2}' | cut -d'/' -f1)
    do
        a=$i
        if [ $? == 0 -a $a != '127.0.0.1' -a $a != '::1' ]
        then
            ip_arr[${#ip_arr[@]}]=$a
        fi
    done
    listen_address=${ip_arr[0]}
    logger -t ${USER} -p local0.info "easysuite select_listen_address is ${listen_address}." &>/dev/null
    return 0
}

##############################
# 功能说明: 获取工具监听IP
#############################
function get_listen_address()
{
    cd ${base_path}
    script="${utils_path}/listen_address.py"
    if [ -f "${script}" ]
    then
        script="${script}"
    elif [ -f "${script}c" ]
    then
        script="${script}c"
    else
        echo "Start failed, the file ${script} does not exist!"
        exit 1
    fi
    listen_address=$(${python_bin_path} 2>/dev/null <<END
import sys
from easysuite.utils.listen_address import ListenAddress
ip_addr = ListenAddress("$(dirname ${LOG_FILE})").output_console()
if ip_addr:
    sys.exit(0)
sys.exit(1)
END
)
    result=$?
    logger -t ${USER} -p local0.info "easysuite listen_address is ${listen_address}." &>/dev/null
    if [ ${result} -ne 0 ]
    then
        return 1
    fi
    return 0
}

#######################################
# 功能说明: 拉起EasySuite Python进程
#######################################
function start_python_easysuite()
{
    if [ -f "${python_start_file_path_py}" ]
    then
        python_start_file_path=${python_start_file_path_py}
    elif [ -f "${python_start_file_path_pyc}" ]
    then
        python_start_file_path=${python_start_file_path_pyc}
    fi
    get_easysuite=$(ps -ef | grep ${ES_PORT} | grep easysuite | awk '{print $2}')
    if [ -z "${get_easysuite}" ]
    then
        listen_address=""
        # 计算监听IP,获取不到有效IP时选一个IP进行监听
        get_listen_address || select_listen_address
        if [ -z "${listen_address}" ]
        then
            logger -t ${USER} -p local0.err "easysuite listen_address is ${listen_address}." &>/dev/null
            exit 1
        fi
        # Check whether an IP address is an IPv6 or IPv4 address.
        [ "${solution_set}" == "-r" ] && export OS_RESTART_ES=TRUE
        if echo "${listen_address}" | grep : >/dev/null
        then
            echo "https://["${listen_address}"]:${ES_PORT}"
            nohup $python_bin_path $python_start_file_path runsslserver [${listen_address}]:${ES_PORT} --noreload --ipv6 >/dev/null 2>&1 &
        else
            echo "https://"${listen_address}":${ES_PORT}"
            nohup $python_bin_path $python_start_file_path runsslserver ${listen_address}:${ES_PORT} --noreload >/dev/null 2>&1 &
        fi
        [ "${solution_set}" == "-r" ] && unset OS_RESTART_ES

        sleep 5
        get_easysuite_again=`ps -ef | grep ${ES_PORT} | grep easysuite | awk '{print $2}'`
        count=1
        while true
        do
            if [ $count -gt 24 ]
            then
                echo "Error ocurred during start easysuite.Please check whether the file permission and storage space are available."
                logger -t $USER -p local0.err "start easysuite failed." >/dev/null 2>&1 &
                exit 1
            fi
            if [ $count -le 24 ]
            then
                get_easysuite_again=`ps -ef | grep ${ES_PORT} | grep easysuite | awk '{print $2}'`
                if [ -n "$get_easysuite_again" ]
                then
                    echo "start easysuite success."
                    logger -t $USER -p local0.err "start easysuite successfully." >/dev/null 2>&1 &
                    break
                else
                    echo "easysuite is starting."
                    count=`expr $count + 1`
                    sleep 5
                    continue
                fi
            fi
        done
    else
        logger -t $USER -p local0.err "easysuite process is already existed." >/dev/null 2>&1
        print_es_processs | tee -a "${LOG_FILE}"
        exit 1
    fi
    rm -f ${tmp_access_ip_path}
}

#######################################
# 功能说明: 打包Solution内安装扩容脚本
#######################################
function zip_solution_path()
{
    echo Start zip software package ...
    if [ -f "$zip_process_path" ]
    then
        nohup $python_bin_path $zip_process_path --noreload >/dev/null 2>&1
    elif [ -f "$zip_process_pyc_path" ]
    then
        nohup $python_bin_path $zip_process_pyc_path --noreload >/dev/null 2>&1
    else
        echo "Zip failed, the file $current_path/easysuite/utils/zip_process.py^(c^) does not exist!"
        exit 1
    fi

    a=$?
    if [ ${a} -eq 2 ]
    then
        echo "Unzip Solution or Migrate zipfile failed, please check whether the zipfile is damaged or the current path is too long."
        exit 2
    elif [ ${a} -eq 3 ]
    then
        echo "echo Copy template files to '/var/software/escompontent/software' failed."
        exit 3
    elif [ ${a} -eq 4 ]
    then
        echo "Zip '*DeployTemplate.zip' or '*Expansion.zip' or 'Migrate.zip' failed."
        exit 4
    elif [ ${a} -eq 0 ]
    then
        if [ -d "${expansion_path}" ]
        then
            [ -d $expansion_path'/software/install/install_script' ] && find $expansion_path'/software/install/install_script' -type f -name "*.sh" | xargs -i echo '"{}"' | xargs chmod --silent 550
        fi
        echo "Zip package success."
    fi
}

################################
# 功能说明:加载公共方法
################################
function load_common()
{
    common_file="${current_path}/common.sh"
    [ -f "${common_file}" ] && chmod u+x "${common_file}" && source "${common_file}"
    work_path="${current_path}"
}

#######################################
# 功能说明:启动刷新签名工具目录权限
#######################################
function chmod_sign_tool()
{
    # 检查是否有sudo权限,对应脚本无sudo提权可能导致需要交互密码导致用户锁定，提前检查
    sudo -l | grep -q '(root) NOPASSWD: /usr/local/uniepsudobin/execute.sh' || return 0
    sudo_pkg="${current_path}/easysuite/plugins/common/NCE-Common/upgrade_online/website/scripts/sudobin/sudobin_pkg.tar"
    
    # 提权包不存在不执行
    [ -f "${sudo_pkg}" ] || return 0
    
    # 提权包存在执行
    sudo /usr/local/uniepsudobin/execute.sh "${sudo_pkg}" /opt/check_verify_tools_${RANDOM} check_verify_tools.sh &>>"${LOG_FILE}"
    return 0
}

#######################################
# 功能说明:检查软连接目录是否正确
#######################################
function check_soft_link()
{
    if [ -d "/opt/oss/easysuite/easysuite" ]
    then
        soft_link_path=`cd /opt/oss/easysuite/easysuite;pwd -P`
        if [ ${soft_link_path} == ${current_path} ]
        then
            return 0
        fi
        return 1
    fi
    return 1
}
# 判断当前easysuite所在节点是否是0号节点
# 非0号节点启动easysuite报错
function check_local_node_id()
{
  local_node_id_file="/opt/oss/manager/var/agent/mcagentid.conf"
  if [ -f ${local_node_id_file} ]
  then
    local local_node_id=`cat ${local_node_id_file} | sed 's/=/ /g' | awk '{print $2}'`
    if [ ${local_node_id} -ne 0 ]
    then
      echo "Error occurred during start easysuite.The current node is not the 0 node of NCE. " >&2
      exit 1
    fi
  fi
}

########################################################################
# 功能说明:检查bash输入流方式是否可用,tmp分区不可用会影响bash管道调用
########################################################################
function check_bash_input()
{
    # 测试bash输入流方式是否可用
    bash <<END
echo -n ''
END
    # 返回码非零,异常退出
    [ $? -ne 0 ] && exit 1
    return 0
}

########################################################################
# 功能说明:设置当前进程的umask为正常的0027,不满足的情况下会导致进程默认创建文件异常
########################################################################
function set_umask()
{
    umask 0027
}

########################################################################
# 参照管理面微服务逻辑， 计算升级前的旧版本EasySuite目录
########################################################################
function get_old_es_path()
{
    ES_WORK_PATH="/opt/oss/easysuite"
    ES_LINK_PATH="${ES_WORK_PATH}/easysuite"
    for work_path in $(ls -t /opt/oss/easysuite)
    do
        # 跳过软链接或者非目录
        [ -L "${ES_WORK_PATH}/${work_path}" -o ! -d "${ES_WORK_PATH}/${work_path}" ] && continue
        # 过滤无启停脚本的无效目录
        [ ! -f "${ES_WORK_PATH}/${work_path}/start.sh" ] && continue
        real_path=$(readlink -f "${ES_WORK_PATH}/${work_path}")
        new_es_path=$(readlink -f "${ES_LINK_PATH}")
        [ "${real_path}" != "${new_es_path}" ] && break
    done
    old_es_path="${ES_WORK_PATH}/${work_path}"

    echo "old EasySuite version is ${old_es_path}." >>${LOG_FILE}
}

########################################################################
# 导出新老版本EasySuite的task_main表，比较数据是否一致
########################################################################
function export_db_data()
{
    # Euler R7版本，sqlite3数据库驱动不兼容导致无法导出数据（后续使用系统自带的sqlite）
    unset LD_LIBRARY_PATH

    old_db_path="${old_es_path}/easysuite/db/easysuite.sqlite3"
    new_db_path="${ES_LINK_PATH}/easysuite/db/easysuite.sqlite3"

    sqlite3 ${old_db_path} &>>${LOG_FILE} <<- EXIT_HERE
.output old_db.sql
SELECT * FROM task_main;
.quit
EXIT_HERE

    sqlite3 ${new_db_path} &>>${LOG_FILE} <<- EXIT_HERE
.output new_db.sql
SELECT * FROM task_main;
.quit
EXIT_HERE

    export LD_LIBRARY_PATH=$python_lib_path:$sqlite_path:$LD_LIBRARY_PATH
}

########################################################################
# 同步KMC密钥（解决低版本管理面服务无法同步kmc密钥问题）
########################################################################
function sync_kmc_keys()
{
    get_old_es_path
    export_db_data

    diff old_db.sql new_db.sql &>>${LOG_FILE}
    if [ $? -eq 0 ]
    then
        echo "need to sync kmc keys." >>${LOG_FILE}
        cp -fp "${old_es_path}"/var/.easysuite/* "${ES_LINK_PATH}"/var/.easysuite
    fi

    rm -f old_db.sql new_db.sql
}

#######################################
# 功能说明:主入口
#######################################
function main()
{
    # 检查脚本执行用户
    check_user_permit

    # 重置环境变量并且重新调用
    reset_env_and_exe "$@"

    # 打印开始提示信息
    echo 'Starting, please wait...'

    # 加载系统关键环境变量
    load_sys_environment

    # 设置脚本执行环境变量
    set_environment

    # 设置进程umask值
    set_umask

    check_soft_link
    if [ $? -ne 0 ]
    then
        echo "Error occurred during start easysuite.Please check whether the soft link are available. " >&2
        exit 1
    fi
    # 判断当前节点是否是OMP_01节点，如果不是则报错退出
    check_local_node_id

    # 检查bash输入流方式执行是否可用
    check_bash_input

    # OS升级回滚过程操作系统重启,不解压cloudsop内容
    [ "${solution_set}" != "-r" ] && unzip_cloudsop

    clear_tmp_file

    load_common

    # 加载es自定义环境变量
    load_es_env_file "${base_path}"

    # OS升级回滚过程操作系统重启,不解压Solution和刷权限
    if [ "${solution_set}" != "-r" ] && [ "${solution_set}" != "-s" ]
    then
        chmod_sign_tool &>/dev/null
        chmod_permit &>/dev/null
        start_plugin_prepare_solution
    fi

    # 首次启动同步kmc密钥(生成证书之前)
    if [ ! -f ${restart_flag} ]
    then
        echo "start EasySuite for the first time." >>${LOG_FILE}
        sync_kmc_keys
        touch ${restart_flag}
    fi

    generate_certificate

    check_ip
    start_python_easysuite
    if [ "${solution_set}" != "-r" ] && [ "${solution_set}" != "-s" ]
    then
        zip_solution_path
    fi
}

main "$@"
exit_code=$?
if [[ ${exit_code} -ne 0 ]]
then
    echo "Error occurred during start easysuite.Please check whether the file permission and soft link and storage space are available. " >&2
    exit 1
fi
exit 0

