#!/bin/bash
#get absolute path
FileName="`basename $0`"
RELATIVE_PATH="$(dirname $0)"
cd "${RELATIVE_PATH}"
ABSOLUTELY_PATH=`pwd`
cd - &> /dev/null

IS_UPGRADE="false"

USER=`whoami`
OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`
[ -z "${WORK_USER}" ] && WORK_USER='ossadm'
if [ "${USER}" != "ossadm" ]
then
    echo "Only ossadm can run this command."
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Only ossadm can run this command."
    exit 1
fi

if [ ! -d  /usr/local/osconfig ]
then
    echo "Uninstall osconfig package done."
    exit 0
fi


#Note:set global parameter
LOGPATH="/usr/local/osconfig/logs"
if [ "${1}" != "upgrade" ]
then
    mkdir -p /home/ossadm/osconfig_uninstall/log && LOGPATH="/home/ossadm/osconfig_uninstall/log"
fi
GLOBAL_BASH_PID=$$
CONSTANT_LOGFILE="${LOGPATH}/uninstall.log"
TMP_CONFIG_PATH="${LOGPATH}/tmp"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmpcontent_install_${RANDOM}.log"
mkdir -p ${LOGPATH} &> /dev/null
mkdir -p ${TMP_CONFIG_PATH} &> /dev/null
chmod 750 "${LOGPATH}" >/dev/null 2>&1

function FUNC_CLEAN_TMP_FILE()
{
    [ -f ${TMP_CONTENT_FILE} ] && rm -f ${TMP_CONTENT_FILE} &> /dev/null
    [ -d ${TMP_CONFIG_PATH} ] && rm -rf ${TMP_CONFIG_PATH}/* &> /dev/null
    return 0
}

# print text with color -- green
# parameters:   $1 -- text
function fn_echo_succ()
{
    FG=32 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- yellow
# parameters:   $1 -- text
function fn_echo_warn()
{
    FG=33 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# print text with color -- red
# parameters:   $1 -- text
function fn_echo_fail()
{
    FG=31 BG=40
    TEXT="$@"
    echo -ne "\E[${FG};${BG}m${TEXT}"
    echo -e "\E[0m"
    return 0
}

# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
function PRINT_FILE_TO_LOG()
{
    local file_name=$1

    LOGFILE="${LOGFILE:-${CONSTANT_LOGFILE}}"

    if [ -f "${file_name}" ]
    then
        local check_content=`cat ${file_name}`
        if [ -z "${check_content}" ]
        then
            LOG "This file <${file_name}> is empty."
            return 1
        else
            LOG "Print <${file_name}>"
            LOG "`cat \"${file_name}\" | sed 's/\r//g'`"
        fi
    else
        LOG "This file <${file_name}> does not exist."
        return 1
    fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "[Line-${BASH_LINENO}] ${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}

#*************************************************************#
# Name:        getSysTime                                     #
# Description: get the system time                            #
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}

#Note: get single log file size: unit M
function fn_get_shell_log_size()
{
    local log_size=""
    log_size=$(cat /usr/local/osconfig/os/conf/common.cfg 2>/dev/null | grep "MAX_SHELL_LOG_SIZE=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_size}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_size}"
    fi
}

function fn_get_shell_log_num()
{
    local log_num=""
    log_num=$(cat /usr/local/osconfig/os/conf/common.cfg 2>/dev/null | grep "MAX_SHELL_LOG_NUM=" | awk -F'=' '{print $2}' | xargs)
    echo "${log_num}" | egrep "^[0-9]+$" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo ""
    else
        echo "${log_num}"
    fi
}

#*************************************************************#
# Name:         LOG                                           #
# Description: record the message into the logfile            #
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSizeM=`fn_get_shell_log_size`
    [ -z "${maxSizeM}" ] && maxSizeM="2"
    local maxSize=$((${maxSizeM} * 1024 * 1024))
    local maxRecord=`fn_get_shell_log_num`
    [ -z "${maxRecord}" ] && maxRecord="5"
    local index=`expr $maxRecord - 1`
    local content="$*"

    if [  -f ${CONSTANT_LOGFILE} ]
    then
        curSize=`ls -lk ${CONSTANT_LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${CONSTANT_LOGFILE}.${index}.gz ]
                then
                    echo y | mv ${CONSTANT_LOGFILE}.${index}.gz ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                    chmod 440 ${CONSTANT_LOGFILE}.`expr ${index} + 1`.gz
                fi
                let index--
            done
            echo y | mv ${CONSTANT_LOGFILE} ${CONSTANT_LOGFILE}.1
            gzip ${CONSTANT_LOGFILE}.1
            chmod 440 ${CONSTANT_LOGFILE}.1.gz
        fi
        if echo "${content}" | grep -w "^WARN:" &> /dev/null
        then
            fn_echo_warn "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        elif echo "${content}" | grep -w "^ERROR:" &> /dev/null
        then
            fn_echo_fail "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] [${LINE_NUMBER}] ${content}" >> ${CONSTANT_LOGFILE}
        else
            echo -e "${content}"
            echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" >> ${CONSTANT_LOGFILE}
        fi
    else
        mkdir -p `dirname ${CONSTANT_LOGFILE}`
        chmod 750 `dirname ${CONSTANT_LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${CONSTANT_LOGFILE}
    fi
}

function handle_osconfig_agent()
{
    local osconfig_server_script="/usr/local/osconfig/os/bin/osconfigserver.sh"
    ps -ef |grep -v grep |  grep -q osconfigserver
    if [ $? -ne 0 ] || [ ! -f "${osconfig_server_script}" ]
    then
        PRINT_LOG "INFO" "Service not start or file <${osconfig_server_script}> not exist,no need to stop service." &> /dev/null
        return 0
    fi

    unregister_system_service  || return $?
    remove_cron || return $?

    bash ${osconfig_server_script} stop &> "${TMP_CONTENT_FILE}"
    if [[ $? -ne 0 ]]
    then
        PRINT_LOG "FATAL" "stop osconfig server failure"
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        return 1
    fi
    return 0
}

function unregister_system_service()
{
    . /usr/local/osconfig/os/lbin/env_profile.sh
    local python_file="/usr/local/osconfig/os/pyscripts/action/config_osconfig_service.py"
    if [[ ! -f "${python_file}" ]]
    then
        python_file="/usr/local/osconfig/os/pyscripts/action/config_osconfig_service.pyc"
    fi
    ${PYTHONHOME}/bin/python ${python_file} unregister
    if [[ $? -ne 0 ]]
    then
        PRINT_LOG "FATAL" "unregister osconfig system service failure"
        return 1
    fi
    return 0
}

function remove_cron()
{
    PRINT_LOG "INFO" "remove osconfig cron..."
    . /usr/local/osconfig/os/lbin/env_profile.sh
    local python_file="/usr/local/osconfig/os/pyscripts/action/cron_action.pyc"
    if [[ ! -f "${python_file}" ]]
    then
        python_file="/usr/local/osconfig/os/pyscripts/action/cron_action.py"
    fi

    PRINT_LOG "INFO" "Invoke ${python_file}..."
    ${PYTHONHOME}/bin/python ${python_file} -action delete
    if [[ $? -ne 0 ]]
    then
        PRINT_LOG "FATAL" "remove osconfig cron failure"
        return 1
    fi

    PRINT_LOG "INFO" "remove osconfig cron done"
    return 0
}

function fn_main()
{
    PRINT_LOG "INFO" "Start to uninstall osconfig package."

    #Note: upgrade or rollback phase, $1 == upgrade
    [ "${1}" == "upgrade" ] && IS_UPGRADE="true"

    #Note:get sudobin path
    local sudobinpath=$(cat /usr/local/osconfig/os/conf/common.cfg | grep "SUDOBIN_PATH=" | awk -F'=' '{print $2}' | xargs)
    if [ -z "${sudobinpath}" -o ! -d "${sudobinpath}" ]
    then
        PRINT_LOG "INFO" "Start to clear the old data of osconfig... "
        tmp_content=$(find /usr/local/osconfig | xargs -I {} stat -c "%A, %U:%G @@ access=%x, modify=%y,  change=%z %n" {} 2>&1)
        PRINT_LOG "INFO" "file list=${tmp_content}" &> /dev/null

        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} chmod -R 750 /usr/local/osconfig/{}
        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} rm -rf /usr/local/osconfig/{}
        if [ $? -ne 0 ]
        then
            PRINT_LOG "WARN" "The sudobin path <${sudobinpath}> does not exist."
            PRINT_LOG "FATAL" "Exec< rm -rf /usr/local/osconfig/files> failure."
            return 1
        fi
        [ "${IS_UPGRADE}" != "true" ] && rm -rf /usr/local/osconfig/*

        PRINT_LOG "INFO" "The osconfig old data is cleared. "
        return 0
    fi

    local installOstype=$(cat /usr/local/osconfig/os/conf/common.cfg | grep "INSTALL_OS_TYPE=" | awk -F'=' '{print $2}' | xargs)
    if [ -z "${installOstype}" ]
    then
        PRINT_LOG "FATAL" "The installOstype <${installOstype}> is empty."
        return 1
    fi
    PRINT_LOG "INFO" "The install ostype is ${installOstype}" &> /dev/null

    #Note: stop osconfig server
    handle_osconfig_agent || return $?
    bash /usr/local/osconfig/os/"${installOstype}"/uninstall.sh "${sudobinpath}" &> ${TMP_CONTENT_FILE}
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        return 1
    fi

    local sudotmpPath="/tmp/osconfig_${RANDOM}_$$"
    local uninstall_ret=0
    if [ "${IS_UPGRADE}" == "true" ]
    then
        sudo ${sudobinpath}/execute.sh "/usr/local/osconfig/os/bin/sudobin_pkg.tar" "${sudotmpPath}" "uninstall.sh" "${sudobinpath}" "upgrade" &> "${TMP_CONTENT_FILE}" || uninstall_ret=1
        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} chmod -R 750 /usr/local/osconfig/{}
        ls /usr/local/osconfig | grep -vw -e "logs" -e "etc" | xargs -I {} rm -rf /usr/local/osconfig/{} || uninstall_ret=1
    else
        sudo ${sudobinpath}/execute.sh "/usr/local/osconfig/os/bin/sudobin_pkg.tar" "${sudotmpPath}" "uninstall.sh" "${sudobinpath}" &> "${TMP_CONTENT_FILE}" || uninstall_ret=1
    fi

    if [ ${uninstall_ret} -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}" &> /dev/null
        PRINT_LOG "FATAL" "Uninstall framework sudo script failure."
        return 1
    fi

    echo "Uninstall osconfig package done."
    return 0
}

fn_main "$@"
RET_MAIN=$?
if [[ "$RET_MAIN" == "0" ]]
then
    logger -t $USER -p local0.info "$FileName;Successful;${OPERIP:-127.0.0.1};Excute execution $FileName to uninstall osconfig."
else
    logger -t $USER -p local0.err "$FileName;Failed;${OPERIP:-127.0.0.1};Excute execution $FileName to uninstall osconfig."
fi
[ -f "${TMP_CONTENT_FILE}" ] && rm -f "${TMP_CONTENT_FILE}" &> /dev/null
FUNC_CLEAN_TMP_FILE &> /dev/null
exit ${RET_MAIN}
