#!/bin/bash
# Name             : config_sudo.sh
# Description      : configure sudo in initial installing
# Platform         : Linux
# Creation date    : 2014-11-1
# Modification log :


# Global variable

export LANG=posix

MODULE_NAME="`basename $0 | sed 's/.sh//g'`"
RELATIVE_PATH="`dirname $0`"

#get absolute path
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`

#import common variable..
. ../common/common.inc

OSS_ADM_USER_NAME=ossadm
OSS_ADM_USER_ID=3001
OSS_ADM_GROUP_NAME=ossgroup
OSS_ADM_GROUP_ID=2000
SOP_GROUP_NAME=sopgroup
SOP_GROUP_ID=2002
OSS_DB_GROUP_NAME=dbgroup
OSS_DB_GROUP_ID=1999
OSS_ADM_HOME_DIR="/home/${OSS_ADM_USER_NAME}"
OSS_ADM_PATCH_PATH="/opt/patch_manager"

SUDO_CONFIG=/etc/sudoers
SUDOERSD_PATH=/etc/sudoers.d
OSS_SUDOERS="/etc/oss_sudoers"
INCLUDE_UNIEP_SUDO_CONFIG="${SUDOERSD_PATH}/oss_uniep_sudoers"

SYBASE_USER_NAME=dbuser
OSS_USER_NAME=ossuser

SUSE_OS_RELEASE_FILE="/etc/SuSE-release"
EULER_OS_RELEASE_FILE="/etc/euleros-release"
REDHAT_OS_RELEASE_FILE="/etc/redhat-release"
CENTOS_OS_RELEASE_FILE="/etc/centos-release"
ENCRYPT_PVALUE=""
PRE_SUDO_SHELL_NAME=execute.sh
GRUB_CFG_FILE="/boot/grub2/grub.cfg"
USER_FILE="/boot/grub2/user.cfg"
EULER_UEFI_GRUB_CFG_FILE="/boot/efi/EFI/euleros/grub.cfg"
EULER_UEFI_USER_FILE="/boot/efi/EFI/euleros/user.cfg"
GRUB_HEADER_FILE="/etc/grub.d/00_header"
GRUB_FILE="/etc/grub.d/42_password"

TMP_RESULT_FILE="${ABSOLUATELY_PATH}"/result_$$.txt
TMP_PYTHON_FILE="${ABSOLUATELY_PATH}/getpvalue.py"
PV_LEN=8
SOP_OS_PATH="/bin:/sbin:/usr/bin:/usr/sbin"
umask 0027

OPERATION_IP=`who am i | awk '{print $NF}' | sed -e 's/[()]//g' -e 's/:.*//g'`
OPERATION_IP=${OPERATION_IP:-"127.0.0.1"}

function fn_write_operation_Log()
{
    local log_level="$1"
    shift
    local log_info="$@"
    local result_info
    
    if [ "${log_level}" = "err" ]
    then
        result_info=Failed
    else
        log_level=info
        result_info=Successful
    fi
    
    logger -t "${USER_NAME}" -p local0.${log_level} "${MODULE_NAME};${result_info};${OPERATION_IP};${log_info}"
    
    return 0
    
}


#***************************************************************#
# Name        : fn_get_config                                   #
# Description : get the config from osmu                        #
# Parameters  : $1 the absoluately path of config file          #
# Parameter   :                                                 #
#***************************************************************#
function fn_get_config()
{
    local TMP_CONFIG_FILE="$1"
    local sudo_bin_dir

    PRINT_LOG "INFO" "Loading <${TMP_CONFIG_FILE}>.."
    fn_clean_reference_variable $@

    # check whether the variable has a value
    sudo_bin_dir=`fn_get_value "${TMP_CONFIG_FILE}" "SUDO_BIN_DIR"`
    sudo_bin_dir=$(readlink -m ${sudo_bin_dir})
    if [ -z "${sudo_bin_dir}" ]
    then
        PRINT_LOG "WARN" "The param SUDO_BIN_DIR<${sudo_bin_dir}>could not found!"
        return ${ERROR_PARAM}
    fi

    # check whether it is an absolute path
    sudo_bin_dir="`echo $sudo_bin_dir | sed -r 's#/+#/#g'`"
    echo "${sudo_bin_dir}" |grep "^/" &> /dev/null
    if [ $? -ne 0 ]
    then
        PRINT_LOG "WARN" "The param SUDO_BIN_DIR<${sudo_bin_dir}> is invalid."
        return ${ERROR_PARAM}
    fi

    # check whether it is a directory in the root directory
    local tmp_sudo_bin="`echo $sudo_bin_dir | sed -r 's#^/+##g'`"
    ls / | grep -w "${tmp_sudo_bin}" &> /dev/null
    if [ $? -eq 0 ]
    then
        PRINT_LOG "FATAL" "Get parameter SUDO_BIN_DIR<${sudo_bin_dir}> failed!"
        return ${ERROR_PARAM}
    fi

    eval $2="\${sudo_bin_dir}"
    PRINT_LOG "INFO" "Loading <${TMP_CONFIG_FILE}>.. OK!!!"
    return 0;
}

function modify_cfg_file()
{
    local file_path="${1}"
    if grep -q 'Sudoconfig: Start adding records' "${file_path}"
    then
        sed -i '/Sudoconfig: Start adding records/,/Sudoconfig: End adding record/s/^PATH=.*/PATH="${PATH}:${SOP_OS_PATH}"/g' "${file_path}"
        return 0
    fi
    echo -e "\n# Sudoconfig: Start adding records.\nSOP_OS_PATH=${SOP_OS_PATH}\nPATH=\"\${PATH}:\${SOP_OS_PATH}\"\nexport PATH\n# Sudoconfig: End adding record\n" >> "${file_path}"
}

function cfg_path()
{
    [ -f "/etc/bashrc" ] && { modify_cfg_file "/etc/bashrc"; return 0; }
    modify_cfg_file "/etc/profile"
}

#*************************************************************#
# Name:        fn_change_user_mode                            #
# Description: Modify the file permission.                    #
#*************************************************************#
function fn_change_user_mode()
{
    local user_name="$1"
    local tmp_home_path="$2"
    local file_name
    local file_list=".bash_profile .bash_logout .bashrc .tcshrc .cshrc"
    for file_name in ${file_list}
    do
        [ -f "${tmp_home_path}/${file_name}" ] || continue
        sudo -u "${user_name}" chmod 0640 "${tmp_home_path}/${file_name}" || { echo "FATAL: Can not change mode of ${tmp_home_path}/${file_name}"; return 1; }
    done
    return 0
}


function fn_create_user_and_modify_files()
{
    local user_name="$1"
    local home_path="$2"
    local pvalue="$3"
    local user_uid="$4"
    local master_group="$5"
    local slave_1_group="$6"
    local slave_2_group="$7"
    local bash_script="$8"

    local tmp_pvalue=""
    [ "${pvalue}" = "" ] || tmp_pvalue="-p ${pvalue}"

    useradd ${tmp_pvalue} -u "${user_uid}" -g "${master_group}" -G "${slave_1_group},${slave_2_group}" -m -d "${home_path}" -k /etc/skel -s "${bash_script}" "${user_name}" &>"${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_LOG "FATAL" "exec cmd<useradd -u ${user_uid} -g ${master_group} -G '${slave_1_group},${slave_2_group}' -m -d ${home_path} -k /etc/skel -s "${bash_script}" ${user_name}> failure"
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        return ${ERR_EXECSYS}
    else
        PRINT_LOG "INFO" "exec cmd<useradd -u ${user_uid} -g ${master_group} -G '${slave_1_group},${slave_2_group}' -m -d ${home_path} -k /etc/skel -s "${bash_script}" ${user_name}> success"
    fi

    fn_change_user_mode "${user_name}" "${home_path}" || return $?
    if [ -f "${home_path}"/.bashrc ]
    then
        echo "umask 0027" >> "${home_path}"/.bashrc
        PRINT_LOG "INFO" "Set umask to ${home_path}/.bashrc success"
    fi

    if [ -f "${home_path}"/.profile ]
    then
        echo "umask 0027" >> "${home_path}"/.profile
        PRINT_LOG "INFO" "Set umask ${home_path}/.profile success"
    fi

    find "${home_path}" -type d |  xargs -I {} chmod 750 {}
    return 0
}


function fn_create_group()
{
    local group_name="$1"
    local group_id="$2"

    PRINT_LOG "INFO" "exec cmd<groupadd -g ${group_id} ${group_name}>..."
    groupadd -g "${group_id}" "${group_name}" &>"${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_LOG "FATAL" "exec cmd<groupadd -g ${group_id} ${group_name}> failure."
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        return ${ERR_EXECSYS}
    fi
    PRINT_LOG "INFO" "exec cmd<groupadd -g ${group_id} ${group_name}> success."

    return 0
}


#*************************************************************#
# Name:        fn_create_user                                 #
# Description: create OS user                                 #
# Parameters:  NULL                                           #
#*************************************************************#
function fn_create_user()
{
    PRINT_LOG "INFO" "Begin to create os user<${OSS_ADM_USER_NAME}>..."

    getent group "${OSS_ADM_GROUP_NAME}" &> /dev/null || { fn_create_group "${OSS_ADM_GROUP_NAME}" "${OSS_ADM_GROUP_ID}" || return $?; }
    getent group "${SOP_GROUP_NAME}" &> /dev/null || { fn_create_group "${SOP_GROUP_NAME}" "${SOP_GROUP_ID}" || return $?; }

    id "${OSS_ADM_USER_NAME}" &>/dev/null || { fn_create_user_and_modify_files "${OSS_ADM_USER_NAME}" "${OSS_ADM_HOME_DIR}" "${ENCRYPT_PVALUE}" "${OSS_ADM_USER_ID}" "${OSS_ADM_GROUP_NAME}" "${OSS_ADM_GROUP_NAME}" "${SOP_GROUP_NAME}" "/bin/bash"  || return $?; }

    PRINT_LOG "INFO" "Create os user<${OSS_ADM_USER_NAME}> success"
    return 0
}

function fn_clean_opasswd()
{
    userdel -rf "${OSS_ADM_USER_NAME}" &> /dev/null
    sed -i "/^${OSS_ADM_USER_NAME}:/d" /etc/security/opasswd
    return $?
}


function get_other_pvalue()
{
    local tmp_pv="$1"
    fn_check_data || return $?
    fn_clean_opasswd || return $?
    useradd "${OSS_ADM_USER_NAME}" || return 1
    { echo "${tmp_pv}"; echo "${tmp_pv}"; } | passwd "${OSS_ADM_USER_NAME}"
    tmp_encrypt_pvalue=$(grep "^${OSS_ADM_USER_NAME}:" /etc/shadow | awk -F':' '{print $2}')
    fn_clean_opasswd || return $?
    [ -z "${tmp_encrypt_pvalue}" ] && return 2
    eval $2="\${tmp_encrypt_pvalue}"
    return 0
}

function init_ossadm_pvalue()
{
    read -t 180 -srp "Please input ossadm password:" pvalue
    [[ -z "${pvalue}" ]] && {
        echo -e "\nWaiting for input timed out or input data was incorrect.\n"
        return 1
    }
    echo ""
    fn_check_blank "${pvalue}" || return $?

    if [ ! -f /etc/SuSE-release ]
    then
        grep -qw 'sm3' /etc/pam.d/password-auth && { get_other_pvalue "${pvalue}" ENCRYPT_PVALUE ; return $?; }
    fi

    get_pvalue_by_python ENCRYPT_PVALUE || return $?
    if [[ -z "${ENCRYPT_PVALUE}" ]]
    then
        PRINT_LOG "FATAL" "The input value is invalid."
        return ${ERR_EXECSYS}
    fi
    return 0
}


function fn_check_blank()
{
    local param="$1"
    #Note: 判断参数中是否包含换行符、制表符
    no_blank_param=$(echo "${param}" | sed ':a;$!N;s/[\n\t\r]//g;ta')

    if [[ "${param}" != "${no_blank_param}" ]]
    then
        PRINT_LOG "FATAL" "Error: The param:<${param}> can not contain <\n|\t|\r>."
        return 5
    fi

    return 0
}

function fn_check_data()
{
        cat <<EOF > "${TMP_PYTHON_FILE}"
import sys
import string

def check_passwd(passwd):
    special_char = "\"#$%&\'\(\)\*+,-./:;<=>\?@\[\]^\`{_|}~!"
    try:
        passwd_len = len(passwd)
        if passwd_len < ${PV_LEN} or passwd_len > 64:
            print("The value length is invalid.")
            return 1

        has_digit = 0
        has_lowercase = 0
        has_uppercase = 0
        has_special_char = 0
        last_char = passwd[0]
        last_char_count = 0
        invalid = False
        for ch in passwd:
            if ch in string.digits:
                has_digit = 1
            elif ch in string.ascii_lowercase:
                has_lowercase = 1
            elif ch in string.ascii_uppercase:
                has_uppercase = 1
            elif ch in special_char:
                has_special_char = 1
            else:
                invalid = True
            if last_char == ch:
                last_char_count += 1
            else:
                last_char = ch
                last_char_count = 1
            if invalid or last_char_count > 2:
                print("The value char is invalid.")
                return 1
        if has_digit + has_lowercase + has_uppercase + has_special_char == 4:
            return 0
        else:
            print("The value not match digit, lowwercase, uppercase and special_char")
            return 2
    except Exception:
        print("Check value rule throw exception")
        return 3

def main():
    try:
        tmp_str = sys.stdin.readlines()
        passwd_value = ''.join(tmp_str[0][:-1])
        if check_passwd(passwd_value) != 0:
            return 1
    except Exception:
        print("Get value failure.")
        return 1

if __name__ == "__main__":
    sys.exit(main())
EOF
    local python_alias="python3"
    which python &> /dev/null && python_alias="python"
    echo "${pvalue}" | "${python_alias}" "${TMP_PYTHON_FILE}" > "${TMP_RESULT_FILE}"
    local ret_code="$?"
    if [[ ${ret_code} -ne 0 ]]
    then
        echo "Create value failure. $(cat ${TMP_RESULT_FILE})"
        return 1
    fi
    return 0
}


function get_pvalue_by_python()
{
    systemctl enable haveged.service
    systemctl restart haveged
    local salt=$(head -c 1000 /dev/random | tr -dc 'a-zA-Z0-9' | cut -c 1-16)
    fn_check_data || return $?

    cat <<EOF > "${TMP_PYTHON_FILE}"
import crypt
import sys
def main():
    try:
        tmp_str = sys.stdin.readlines()
        passwd_value = ''.join(tmp_str[0][:-1])
        print(crypt.crypt(passwd_value, "\$6\$${salt}\$"))
    except Exception:
        print("Get value failure.")
        return 1

if __name__ == "__main__":
    sys.exit(main())
EOF
    local python_alias="python3"
    which python &> /dev/null && python_alias="python"
    echo "${pvalue}" | "${python_alias}" "${TMP_PYTHON_FILE}" > "${TMP_RESULT_FILE}"
    local ret_code="$?"
    local tmp_encrypt_pvalue=$(cat ${TMP_RESULT_FILE})
    if [[ ${ret_code} -ne 0 || -z "${tmp_encrypt_pvalue}" ]]
    then
        echo "Create value failure. $(cat ${TMP_RESULT_FILE})"
        return 1
    fi

    eval $1="\${tmp_encrypt_pvalue}"
    return 0
}


function harden_system_sudo()
{
    local tmp_euler_msg=$(grep '^PRETTY_NAME="EulerOS 2.0' /etc/os-release | egrep -o 'SP[0-9]+')
    tmp_euler_msg="${tmp_euler_msg//SP/}"
    tmp_euler_msg="${tmp_euler_msg:-0}"
    [ "${tmp_euler_msg}" -lt 8 ] && { PRINT_LOG "INFO" "Current ostype is not euler arm, skip current step" &>/dev/null; return 0; }

    local sudo_config_file="/etc/pam.d/sudo"
    local sudo_config_tmp_file="/etc/pam.d/sudo_bak"
    local content_file="/opt/account_${RANDOM}.json"

    [ -f "${sudo_config_tmp_file}" ] && rm -f "${sudo_config_tmp_file}"
    [ ! -f "${sudo_config_file}" ] && { PRINT_LOG "FATAL" "sudo config file <${sudo_config_file}> is not exist, please check it"; return 1; }

    local account_content=$(egrep "^\s*account\s+include\s+system-auth" "${sudo_config_file}")
    [ -z "${account_content}" ] && { PRINT_LOG "INFO" "Sudo config file <${sudo_config_file}> is already modified" &>/dev/null; return 0; }

    PRINT_LOG "INFO" "Before modify sudo config file <${sudo_config_file}> content is: `cat ${sudo_config_file}`" &>/dev/null
    cp -a "${sudo_config_file}" "${sudo_config_tmp_file}"

    sed -ri "/^\s*account\s+required\s+pam_unix.so/d" "${sudo_config_tmp_file}"
    sed -ri "/^\s*#*account\s+required\s+pam_faillock.so/d" "${sudo_config_tmp_file}"
    sed -ri "/^\s*account\s+sufficient\s+pam_localuser.so/d" "${sudo_config_tmp_file}"
    sed -ri "/^\s*account\s+sufficient\s+pam_succeed_if.so uid < 1000 quiet/d" "${sudo_config_tmp_file}"
    sed -ri "/^\s*-account\s+\[default=bad success=ok user_unknown=ignore\]\s+pam_sss.so/d" "${sudo_config_tmp_file}"
    sed -ri "/^\s*account\s+required\s+pam_permit.so/d" "${sudo_config_tmp_file}"

    grep "^\s*-*account" /etc/pam.d/system-auth | grep -v "account.*pam_faillock.so" > "${content_file}"
    local auth_line=$(cat -n "${sudo_config_tmp_file}"| egrep "\s*account\s+include\s+system-auth" | awk '{print $1}')
    if [ ! -z "${auth_line}" ]
    then
        sed -i "${auth_line} r ${content_file}" "${sudo_config_tmp_file}"
        sed -i "${auth_line}"d "${sudo_config_tmp_file}"
        mv "${sudo_config_tmp_file}" "${sudo_config_file}"
        PRINT_LOG "INFO" "After modify sudo config file <${sudo_config_file}> content is: `cat ${sudo_config_file}`" &>/dev/null
    fi

    [ -f "${content_file}" ] && rm -f "${content_file}"
    [ -f "${sudo_config_tmp_file}" ] && rm -f "${sudo_config_tmp_file}"

    return 0
}


function fn_stop_crond_mail()
{
    local get_os_type=$(uname -m)
    [ "${get_os_type}" = "aarch64" ] || return 0
    if [ ! -f /etc/crontab -o ! -f /etc/sysconfig/crond ]
    then
        echo "WARN: File /etc/crontab or /etc/sysconfig/crond is not found. "
        return 0
    fi

    grep -q "^\s*MAILTO=['\"]*root['\"]*\s*$" /etc/crontab || { echo "WARN: The MAILTO value does not meet the requirements. "; return 0; }
    grep -q "^\s*CRONDARGS=.*-m off" /etc/sysconfig/crond && { echo "is already set. "; return 0; }
    local args_centont=$(egrep "^\s*CRONDARGS=" /etc/sysconfig/crond | sed -e 's/^\s*CRONDARGS=//' -re 's/^\s*"|"\s*$//g' -e "s/^\s*'|'\s*$//g")

    if [ -z "${args_centont}" ]
    then
        sed -ri '/^\s*CRONDARGS=/c\CRONDARGS="-m off"' /etc/sysconfig/crond
    else
        sed -ri "/^\s*CRONDARGS=/c\CRONDARGS=\"-m off ${args_centont}\"" /etc/sysconfig/crond
    fi

    sed -ri '/^\s*MAILTO=root/c\MAILTO=""' /etc/crontab
    systemctl restart crond

    return 0
}


function fn_config_sudoers_file()
{
    local sudo_user="${OSS_ADM_USER_NAME}"
    [ ! -z "${REF_RUN_USER}" ] && sudo_user="${REF_RUN_USER}"

    #configure uniep_sudoers sudo
    sed -i "\#.*/etc/uniep_sudoers.*#d" ${SUDO_CONFIG} || return $?
    sed -i "\#.*${INCLUDE_UNIEP_SUDO_CONFIG}.*#d" ${SUDO_CONFIG} || return $?

    sudoContext="${sudo_user} ALL=(root) NOPASSWD: ${SUDO_BIN_DIR}"
    if grep "^${sudoContext}" "${SUDO_CONFIG}" &>/dev/null
    then
        sed -i "\#${sudoContext}/execute.sh#d" "${SUDO_CONFIG}"  || echo "WARN: remove ${sudoContext} from sudoers failed."
        sed -i "\#${sudoContext}//execute.sh#d" "${SUDO_CONFIG}"  || echo "WARN: remove ${sudoContext} from sudoers failed."
    fi

    #modify mode for ${INCLUDE_UNIEP_SUDO_CONFIG}
    if [ ! -f "${INCLUDE_UNIEP_SUDO_CONFIG}" ]
    then
        touch "${INCLUDE_UNIEP_SUDO_CONFIG}" || return $?
    fi
    chown root:root ${INCLUDE_UNIEP_SUDO_CONFIG} || return $?
    chmod 400 ${INCLUDE_UNIEP_SUDO_CONFIG} || return $?

    local user_list="${OSS_ADM_USER_NAME}"
    [ "${sudo_user}" != "${OSS_ADM_USER_NAME}" ] && user_list=("${OSS_ADM_USER_NAME}" "${REF_RUN_USER}")
    for run_user in "${user_list[@]}"
    do
        if ! grep "^${run_user} ALL=(root) NOPASSWD: ${SUDO_BIN_DIR}/execute.sh" "${INCLUDE_UNIEP_SUDO_CONFIG}" &>/dev/null
        then
            echo "${run_user} ALL=(root) NOPASSWD: ${SUDO_BIN_DIR}/execute.sh" >> "${INCLUDE_UNIEP_SUDO_CONFIG}" || return $?
        fi
        for user_name in ${SYBASE_USER_NAME} ${OSS_USER_NAME} ${OSS_ADM_USER_NAME}
        do
            [ "${run_user}" = "${user_name}" ] && continue
            if [ "${user_name}" = "${OSS_ADM_USER_NAME}" -a "${run_user}" = "${REF_RUN_USER}" ]
            then
                if ! grep "^${user_name} ALL=(${run_user}) NOPASSWD: ALL" "${INCLUDE_UNIEP_SUDO_CONFIG}" &>/dev/null
                then
                    echo "${user_name} ALL=(${run_user}) NOPASSWD: ALL" >> "${INCLUDE_UNIEP_SUDO_CONFIG}" || return $?
                fi
                continue
            fi

            if ! grep "^${run_user} ALL=(${user_name}) NOPASSWD: ALL" "${INCLUDE_UNIEP_SUDO_CONFIG}" &>/dev/null
            then
                echo "${run_user} ALL=(${user_name}) NOPASSWD: ALL" >> "${INCLUDE_UNIEP_SUDO_CONFIG}" || return $?
            fi
        done
    done
    return 0
}


#*************************************************************#
# Name:        fn_config_sudo                                 #
# Description: configure sudo                                 #
# Parameters:  NULL                                           #
#*************************************************************#
function fn_config_sudo()
{
    local SUDO_BIN_DIR="$1"
    PRINT_LOG "INFO" "Being to configure sudo..."

    if [ ! -d "${SUDOERSD_PATH}" ]
    then
        mkdir "${SUDOERSD_PATH}" || return $?
        chmod 750 "${SUDOERSD_PATH}" || return $?
        chown root:root "${SUDOERSD_PATH}" || return $?
    fi

    if ! egrep -r "#includedir\s+${SUDOERSD_PATH}" "${SUDO_CONFIG}"
    then
        echo "#includedir ${SUDOERSD_PATH}" >> "${SUDO_CONFIG}"
        if [ $? -ne 0 ]
        then
            PRINT_LOG "FATAL" "Failed to modify the ${SUDO_CONFIG} configuration!"
            return "${ERROR_PARAM}"
        fi
    fi

    #create sudo bin directory
    mkdir -p ${SUDO_BIN_DIR} || return $?
    #copy interface execute shell
    cp ./execute.sh ${SUDO_BIN_DIR} || return $?
    chown -R root:root ${SUDO_BIN_DIR} || return $?

    chmod -R 500 ${SUDO_BIN_DIR} || return $?
    #modify mode for ${SUDO_CONFIG} || return $?
    chown root:root ${SUDO_CONFIG} || return $?
    chmod 400 ${SUDO_CONFIG} || return $?

    # Note: create patch management directory
    mkdir -p "${OSS_ADM_PATCH_PATH}" || return $?
    chown "${OSS_ADM_USER_NAME}":"${OSS_ADM_GROUP_NAME}" "${OSS_ADM_PATCH_PATH}" || return $?
    chmod 750 "${OSS_ADM_PATCH_PATH}" || return $?

    SUDO_BIN_DIR=$(readlink -m ${SUDO_BIN_DIR})

    #Note:delete the oss_sudoers config
    is_null_file=$(grep -v '^\s*$' "${OSS_SUDOERS}" 2>/dev/null)
    if [ -z "${is_null_file}" ]
    then
        sed -i "\#include.*${OSS_SUDOERS}.*#d" "${SUDO_CONFIG}"  &> /dev/null
        if [ $? -ne 0 ]
        then
            echo "Delete the oss_sudoers config failure."
            return 1
        fi
        rm -f "${OSS_SUDOERS}"
    fi

    fn_config_sudoers_file || return $?
    harden_system_sudo || return $?

    PRINT_LOG "INFO" "Configure Sudo Successfully"
    return 0
}

function is_exist_grub()
{
    if is_efi_mode
    then
        grep -e "^\s*password_pbkdf2.*grub.pbkdf2.sha512.*" "${EULER_UEFI_GRUB_CFG_FILE}" &> /dev/null && return 0
    else
        grep -e "^\s*password_pbkdf2.*grub.pbkdf2.sha512.*" "${GRUB_CFG_FILE}" &> /dev/null && return 0
    fi
    return 1
}

function is_efi_mode()
{
    [[ -d /boot/efi/EFI/euleros && -d /sys/firmware/efi ]] && return 0
    return 1
}

function get_grub_in_user_cfg()
{
    local grub_pvalue=""
    if is_efi_mode
    then
        grep "grub.pbkdf2.sha512" ${EULER_UEFI_USER_FILE} &> /dev/null && grub_pvalue=`grep "grub.pbkdf2.sha512" ${EULER_UEFI_USER_FILE} | awk -F "=" '{print $2}'`
    else
        grep "grub.pbkdf2.sha512" ${USER_FILE} &> /dev/null && grub_pvalue=`grep "grub.pbkdf2.sha512" ${USER_FILE} | awk -F "=" '{print $2}'`
    fi
    eval "${1}"="\${grub_pvalue}"
}

function config_pvalue()
{
    local encrypt_pvalue=""
    is_exist_grub && return 0

    get_grub_in_user_cfg encrypt_pvalue
    if [[ -z ${encrypt_pvalue} ]]
    then
        get_grub_pvalue encrypt_pvalue || return $?
        set_grub_pvalue ${encrypt_pvalue} || return $?
    else
        set_grub_pvalue ${encrypt_pvalue} || return $?
        clean_user_cfg
    fi
    return 0
}

function clean_user_cfg()
{
    if is_efi_mode
    then
        sed -i /"GRUB2_PASSWORD="/d ${EULER_UEFI_USER_FILE}
    else
        sed -i /"GRUB2_PASSWORD="/d ${USER_FILE}
    fi
}

function set_grub_pvalue()
{
    grub_pvalue=$1
    if [[ -f "/etc/euleros-release" ]]
    then
        set_euler_config_file ${grub_pvalue}
    else
        set_suse_config_file ${grub_pvalue}
    fi
    clean_header_file
    config_grub "${GRUB_CFG_FILE}" || return $?
    config_grub "${EULER_UEFI_GRUB_CFG_FILE}" || return $?
    return 0
}

function set_euler_config_file()
{
    local encry_pvalue=$1
    if [[ ! -f "${GRUB_FILE}" ]]
    then
        touch "${GRUB_FILE}"
        chmod 700 "${GRUB_FILE}"
    fi
    cat <<EOG > "${GRUB_FILE}"
cat <<EOF
set superusers="root"
password_pbkdf2 root ${encry_pvalue}
export superusers
set unrestricted_menu="y"
export unrestricted_menu
EOF
EOG
}

function set_suse_config_file()
{
    local encry_pvalue=$1
    if [[ -f "${GRUB_FILE}" ]]
    then
        sed -ri "/^\s*exec\s+tail.*/d" "${GRUB_FILE}"
        sed -i '1i\exec tail -n +2 \$0' "${GRUB_FILE}"

        sed -i "/^\s*password_pbkdf2.*/d" "${GRUB_FILE}"
        echo "password_pbkdf2 root ${encry_pvalue}" >> "${GRUB_FILE}"

        sed -ri "/^\s*set\s+superusers.*/d" "${GRUB_FILE}"
        echo "set superusers=\"root\"" >> "${GRUB_FILE}"

        sed -ri "/^\s*export\s+superusers\s*/d" "${GRUB_FILE}"
        echo "export superusers" >> "${GRUB_FILE}"

        sed -ri "/^\s*set\s+unrestricted_menu.*/d" "${GRUB_FILE}"
        echo "set unrestricted_menu=\"y\"" >> "${GRUB_FILE}"

        sed -ri "/^\s*export\s+unrestricted_menu\s*/d" "${GRUB_FILE}"
        echo "export unrestricted_menu" >> "${GRUB_FILE}"
    else
        touch "${GRUB_FILE}"
        chmod 700 "${GRUB_FILE}"
        cat <<EOF > "${GRUB_FILE}"
exec tail -n +2 \$0
set superusers="root"
password_pbkdf2 root ${encry_pvalue}
export superusers
set unrestricted_menu="y"
export unrestricted_menu
EOF
    fi
}

function config_grub()
{
    config_file="$1"
    if [[ -f "${config_file}" ]]
    then
        grub2-mkconfig -o "${config_file}" &> /dev/null || return 3
    fi
    return 0
}

function clean_header_file()
{
    sed -ri '/^\s*set\s+superusers.*/d' "${GRUB_HEADER_FILE}"
    sed -ri '/^\s*export\s+superusers\s*/d' "${GRUB_HEADER_FILE}"
    sed -ri '/^\s*password_pbkdf2.*/d' "${GRUB_HEADER_FILE}"
}

function get_grub_random()
{
    local lowe_char=0
    local upper_char=0
    local num_char=0
    local special_char=0
    local dict='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789~!@#$%^&*()-_=+\\|[{}];:,<.>/?"'
    local dict_count=94
    local len=16
    local count=0
    local random_pvalue=""
    while [ ${count} -le ${len} ]
    do
        random_num=$(head -c 500 /dev/random | tr -dc '0-9' |cut -c 1-6)
        echo "${random_num}" |  grep -qw '^[1-9][0-9]*$'
        if [ $? -ne 0 ]
        then
            continue
        fi
        index=$((${random_num} % dict_count))
        if [ ${index} -eq 0 ]
        then
            continue
        fi
        char=$(echo ${dict} | cut -c ${index}-${index})
        let count++
        random_pvalue="${random_pvalue}${char}"
        #Note: check num
        echo "${char}" | grep -qw '[0-9]'
        if [ $? -eq 0 ]
        then
            num_char=1
        #Note: check lowwer case
        elif echo "${char}" | grep -qw '[a-z]'
        then
            lowe_char=1
        #Note: check higher case
        elif echo "${char}" | grep -qw '[A-Z]'
        then
            upper_char=1
        else
        #Note: it is special char
            special_char=1
        fi
        if [ ${count} -ne 16 ]
        then
            continue
        fi

        charCount=$((${lowe_char} + ${upper_char} + ${num_char} + ${special_char}))
        if [ ${count} -eq 16 -a ${charCount} -lt 4 ]
        then
            random_pvalue=$(echo ${random_pvalue} | cut -c 2-16)
            let count--
        fi
    done
    eval "${1}"=\${random_pvalue}
}

function get_grub_pvalue()
{
    get_grub_random grub_value
    {
        sleep 1
        echo "${grub_value}"
        sleep 1;
        echo "${grub_value}"
    } | grub2-mkpasswd-pbkdf2 &> "${TMP_CONTENT_FILE}"
    ret_code=$?
    tmp_encrypt_pvalue=$(cat ${TMP_CONTENT_FILE} |  grep grub.pbkdf2 | awk '{print $NF}' | xargs)

    eval "${1}"="\${tmp_encrypt_pvalue}"
    return ${ret_code}
}


function fn_check_param()
{
    if [ "${REF_SUDO_CFG_FLAG}" = "True" ]
    then
        fn_check_config_file "${CONFIG_FILE}" || return $?
        fn_get_config "${TMP_CONFIG_FILE}" "REF_SUDO_BIN_DIR" || return $?
    else
        [ -z "${REF_SUDO_BIN_DIR}" ] && return 1
        ls / | grep  "${REF_SUDO_BIN_DIR//\//}" &> /dev/null && return 1
    fi

    [ "${PV_LEN}" -le 8 ] && PV_LEN=8
    [ "${PV_LEN}" -ge 64 ] && PV_LEN=64

    [ -z "${REF_SUDO_BIN_DIR}" ] && return 1
    return 0
}


function fn_get_opt()
{
    while [ $# -gt 0 ]
    do
        if [ -f "${1}" ]
        then
            REF_SUDO_CFG_FLAG="True"
            CONFIG_FILE="$1"
            shift
            continue
        elif [ "$1" = '-d' ]
        then
            REF_SUDO_PATH_FLAG="True"
            REF_SUDO_BIN_DIR="$2"
            shift
            shift
        elif [[ "${1}" == "-userpwdlength" ]]
        then
            PV_LEN="$2"
            expr "${PV_LEN}" / 1 &>/dev/null || { PRINT_LOG "FATAL" "The value of userpwdlength must be an integer greater than 0."; return 1; }
            shift
            shift
            continue
        else
            return 1
        fi
        
    done

    return 0
}


function fn_usage()
{
    PRINT_LOG "WARN" "Usage: $0 < [ config file ] | [ -d <obj path> ]> [ -userpwdlength <int Number> ]"
    PRINT_LOG "WARN" "e.g:   $0 /tmp/sample.properties or $0 -d /usr/local/uniepsudobin/"
    return ${ERROR_PARAM}
}


#*************************************************************#
# Name:         fn_main                                       #
# Description:  main function                                 #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_main()
{
    PRINT_LOG "INFO" "bash $0 $*" &> /dev/null
    [ $# -lt 1 ] && { fn_usage; return $?; }

    #only root run this script
    if [ `id -u` -ne 0 ]
    then
        PRINT_LOG "FATAL" "Only root user can run this script."
        return ${ERROR_PARAM}
    fi

    fn_get_opt "$@" || { fn_usage; return $?; }

    if id "${OSS_ADM_USER_NAME}" &>/dev/null
    then
        fn_change_user_mode "${OSS_ADM_USER_NAME}" "${OSS_ADM_HOME_DIR}" || return $?
    else
        init_ossadm_pvalue || return $?
    fi
    fn_check_param  || { fn_usage; return $?; }
    echo "${REF_SUDO_BIN_DIR}" | grep '/$' &> /dev/null || REF_SUDO_BIN_DIR="${REF_SUDO_BIN_DIR}/"

    #Note:check sudo path. the sudo path does not /usr/local/osconfig floder
    local tmpSudoPath="$(readlink -m ${REF_SUDO_BIN_DIR})"
    if [ "${tmpSudoPath}" == "/usr/local/osconfig" ]
    then
        PRINT_LOG "FATAL" "the path </usr/local/osconfig> does not be config as sudobin path."
        return ${ERR_FILEEXIST}
    fi

    #checking pre sudo shell and /etc/sudoers exist?
    cd ${ABSOLUATELY_PATH}
    if [ ! -f "${PRE_SUDO_SHELL_NAME}" -o ! -f "${SUDO_CONFIG}" ]
    then
        PRINT_LOG "FATAL" "the file<${SUDO_CONFIG}> or the script<`pwd`/${PRE_SUDO_SHELL_NAME}> is not exist"
        return ${ERR_FILEEXIST}
    fi

    cfg_path

    #create os user ossadm
    fn_create_user || return $?

    #configure sudo
    fn_config_sudo "${REF_SUDO_BIN_DIR}" || return $?
    fn_stop_crond_mail
    config_pvalue || return $?

    sudo_ver_file="${ABSOLUATELY_PATH}/../sudoconfig_version"
    if [ ! -f "${sudo_ver_file}" ]
    then
        PRINT_LOG "FATAL" "the file<${sudo_ver_file}>  is not exist"
        return ${ERR_FILEEXIST}
    fi
    [ -f "/etc/sudoconfig_version" ] && rm -f /etc/sudoconfig_version
    cp -f "${sudo_ver_file}" /etc/sudoconfig_version || return $?
    chmod 640 /etc/sudoconfig_version

    return 0
}


# main entrance

fn_main "$@"
MAIN_RET=$?
operation_info=$(ls "${REF_SUDO_BIN_DIR}/execute.sh")
msg_level=info
[ "${MAIN_RET}" -ne 0 ] && msg_level=err
fn_write_operation_Log "info" "msg=${operation_info}"
FUNC_CLEAN_TMP_FILE
[ -d "${TMP_CONFIG_PATH}" ] && rm -rf "${TMP_CONFIG_PATH}"/tmp*
[[ -f "${TMP_RESULT_FILE}" ]] && rm -f "${TMP_RESULT_FILE}"
[[ -f "${TMP_PYTHON_FILE}" ]] && rm -f "${TMP_PYTHON_FILE}"
PRINT_LOG "INFO" "MAIN_RET=${MAIN_RET}"
exit ${MAIN_RET}
