#!/bin/bash
# Name             : install_ipsi.sh
# Description      : install ipsi tools
# Platform         : Linux
# Creation date    : 2014-12-5
# Modification log :

# Global variable

export LC_ALL=POSIX

MODULE_NAME="`basename $0 | sed 's/.sh//g'`"
RELATIVE_PATH="`dirname $0`"

#get absolute path
cd "${RELATIVE_PATH}"
ABSOLUATELY_PATH=`pwd`

#import common variable..
. ../common/common.inc

SIGN_KEY="${ABSOLUATELY_PATH}/KEYS"
SIGN_NAME="${ABSOLUATELY_PATH}/verifyPkgSign"
SIGN_INSTALL_DIR="/opt/signtool"
USR_LOCAL_SIGN_INSTALL_DIR="/usr/local/signtool"
CA_ROOT="/etc/ssl/certs"

CMS_BIN="${ABSOLUATELY_PATH}/hwcmstool"
CMS_CA="${ABSOLUATELY_PATH}/HuaweiRootCAsipCRLs.crl"
P7S_CA="${ABSOLUATELY_PATH}/HuaweiRootCAsipCRLs_Release.crl"

CMS_BIN_TARGET="/usr/bin/hwcmstool"
CMS_CA_TARGET="${CA_ROOT}/HuaweiRootCAsipCRLs.crl"
P7S_CA_TARGET="${CA_ROOT}/HuaweiRootCAsipCRLs_Release.crl"
BAK_CMS_CA_TARGET="${USR_LOCAL_SIGN_INSTALL_DIR}/HuaweiRootCAsipCRLs.crl"
BAK_P7S_CA_TARGET="${USR_LOCAL_SIGN_INSTALL_DIR}/HuaweiRootCAsipCRLs_Release.crl"


function fn_install_gpg()
{
    PRINT_LOG "INFO" "Begin to install GPG..."
    
    #check if package file is exist?
    if [ ! -f "${SIGN_KEY}" -o ! -f "${SIGN_NAME}" ]
    then
        PRINT_LOG "WARN" "the sign key file<${SIGN_KEY}> or the sign program<${SIGN_NAME}> does not exist,skip."
        return 0
    fi
      
    #check if open gpg tools is installed on the system
    gpg --version | grep "gnupg" | awk -F"." '{print $2}' | grep -w "gnupg" &>"${TMP_CONTENT_FILE}"
    if [ $? -ne 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "FATAL" "The system does not install GPG tool"
        return 1
    fi
    
    #import sign key
    chmod 600 ${SIGN_KEY}
    export LC_ALL=POSIX

    local tmp_content=$(
/usr/bin/expect <<EOF
set timeout 30
spawn gpg --import ${SIGN_KEY}
expect {
     "*update the preferences*" {
        send -- "y\r"
        expect {
           "*processed*" { send "\r" }
           default { send_user "\r"; exit 0;}
        }
     }
     "*processed*" { send "\r" }
     default { send_user "\r"; exit 0;}
}
expect eof
EOF
)

    echo "${tmp_content}" | egrep -q "<support@huawei.com>\" (imported|not changed)"
    if [ $? -ne 0 ]
    then
        PRINT_LOG "FATAL" "Import sign key<${SIGN_KEY}> failure"
        return 1
    fi
    
    #create sign program store directory and copy sign program to it
    mkdir -p "${SIGN_INSTALL_DIR}" && chmod 700 "${SIGN_INSTALL_DIR}"
    copy_and_chmod "${SIGN_NAME}" "${SIGN_INSTALL_DIR}/verifyPkgSign" "500" || return $?
    
    #create usr local sign program store directory and copy sign program to it
    mkdir -p "${USR_LOCAL_SIGN_INSTALL_DIR}" && chmod 700 "${USR_LOCAL_SIGN_INSTALL_DIR}"
    copy_and_chmod "${SIGN_NAME}" "${USR_LOCAL_SIGN_INSTALL_DIR}/verifyPkgSign" "500" || return $?
    
    PRINT_LOG "INFO" "Install GPG success."
    return 0    
}

function fn_install_cms()
{
    PRINT_LOG "INFO" "Begin to install CMS..."
    local ret=0
    if [ ! -f "${CMS_BIN}" -o ! -f "${CMS_CA}" -o ! -f "${P7S_CA}" ]
    then
        PRINT_LOG "WARN" "the hwcmstool file<${CMS_BIN}> or ca <${CMS_CA}> or <${P7S_CA}> does not exist, skip."
        return 0
    fi

    check_os_arch || return $?

    copy_and_chmod "${CMS_BIN}" "${CMS_BIN_TARGET}" "550" || let ret++
    
    mkdir -p "/etc/ssl/certs" && chmod 755 "/etc/ssl" && chmod 750 "/etc/ssl/certs"

    copy_and_chmod "${CMS_CA}" "${CMS_CA_TARGET}" "600" || let ret++
    copy_and_chmod "${P7S_CA}" "${P7S_CA_TARGET}" "600" || let ret++
    copy_and_chmod "${CMS_CA}" "${BAK_CMS_CA_TARGET}" "600" || let ret++
    copy_and_chmod "${P7S_CA}" "${BAK_P7S_CA_TARGET}" "600" || let ret++

    return "${ret}"
}

function check_os_arch()
{
    local os_type=$(uname -m | awk -F'_' '{print $1}')
    file "${CMS_BIN}" | grep "${os_type}" &>/dev/null || {
        PRINT_LOG "FATAL" "The sudoconfig package does not match system, please check."
        return 1
    }
    return 0
}

#usage: copy_and_chmod <src> <target> <mode>
function copy_and_chmod()
{
    for i in {1..5}
    do
        cp -f "$1" "$2" && {
            chmod $3 $2
            PRINT_LOG "INFO" "copy ${1} to ${2} success."
            return 0
        }
        sleep 2
    done
    PRINT_LOG "FATAL" "copy ${1} to ${2} failed."
    return 1
}

function fn_create_shareliblink()
{
    PRINT_LOG "INFO" "Begin to create share lib link in env..."

    local ncurses_path="/lib64/libncurses.so.5"
    
    if [ ! -f "${ncurses_path}" ]
    then
        local ncureses_real_path="`ls /lib64/libncurses.so.* | sort -r | head -n1`"
        if [ -z "${ncureses_real_path}" ]
        then
            PRINT_LOG "FATAL" "There is no /lib64/libncurses.so.* in the env."
            return 1
        fi
        ln -sf "`basename ${ncureses_real_path}`" "${ncurses_path}"

    else
        PRINT_LOG "INFO" "The file ${ncurses_path} is exist in env, no need to create link."
    fi

    PRINT_LOG "INFO" "create share lib link in env success."

    return 0
}

function fn_check_target_files()
{
    if [ ! -f "${CMS_BIN_TARGET}" ] || [ ! -f "${SIGN_INSTALL_DIR}/verifyPkgSign" ] || [ ! -f "${CMS_CA_TARGET}" ] || [ ! -f "${USR_LOCAL_SIGN_INSTALL_DIR}/verifyPkgSign" ] || [ ! -f "${P7S_CA_TARGET}" ]
    then
        PRINT_LOG "FATAL" "Install ipsi failed."
        return 1
    fi

    return 0
}

function fn_check_path()
{
    local tmp_msg=$(lsattr -d /usr/bin /etc/ /opt/signtool /usr/local/signtool 2> /dev/null | awk '$1~/i/ {print $0}')
    [ -z "${tmp_msg}" ] && return 0

    PRINT_LOG "FATAL" "The directory is locked: ${tmp_msg}"
    return 1
}


#*************************************************************#
# Name:         fn_main                                       #
# Description:  main function                                 #
# Parameters:   NULL                                          #
#*************************************************************#
function fn_main()
{
    RET_VAL=0
    PRINT_LOG "INFO" "Begin to install ipsi ..."
    if [[ "$1" != "upgrade" ]]
    then
        [[ -d "${SIGN_INSTALL_DIR}" ]] && rm -rf "${SIGN_INSTALL_DIR}"
        [[ -d "${USR_LOCAL_SIGN_INSTALL_DIR}" ]] && rm -rf "${USR_LOCAL_SIGN_INSTALL_DIR}"
    fi
    
    fn_create_shareliblink || return $?
    fn_check_path || return $?
    fn_install_gpg || RET_VAL=$?
    fn_install_cms || RET_VAL=$?
    fn_check_target_files || RET_VAL=$?
    
    PRINT_LOG "INFO" "Install ipsi end"
    return "${RET_VAL}"
}

# main entrance

fn_main "$@"
MAIN_RET=$?
FUNC_CLEAN_TMP_FILE
[ -d "${TMP_CONFIG_PATH}" ] && rm -rf "${TMP_CONFIG_PATH}"/tmp*
PRINT_LOG "INFO" "MAIN_RET=${MAIN_RET}"
exit ${MAIN_RET}
