#!/usr/bin/env bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.
# File name     : execute_upgrade.sh
# Description   : 欧方升级回滚入口脚本
###########################################################

# 当前工作目录
CURRENT_DIR=$(cd $(dirname $0);pwd)
. "${CURRENT_DIR}/upgrade_rollback_func.sh"
LOG_FILE=""
# 类型：upgrade/rollback/single-upgrade/single-rollback
TYPE=""
# 版本，依据欧方包名的版本
VERSION=""
# 单点升级时，传入的欧方包路径
PACKAGE_PATH=""
# 升级回滚路径，用于单点升级时定制;非单点升级时，当前目录即为升级目录
OSM_PATH="${CURRENT_DIR}"
# /opt/dpc_uniupg_pkg目录创建标志，默认不创建
BUILD_FLAG="not-build"
# MAE与NCE融合共管场景，是否为此场景的标志
WIRELESS="no"
# 升级类型列表
TYPE_LIST=("upgrade" "rollback" "single-upgrade" "single-rollback")
# 执行超时时间
TIME_OUT=1200


############################################################
# Description: 初始化变量
############################################################
function init_param() {
    TYPE="${1}"
    echo "${TYPE_LIST[@]}" | grep "${TYPE}" >/dev/null 2>&1
    if [ $? -ne 0 ];then
        print_log "ERROR" "param:${TYPE} is invalid."
        return 1
    fi
    VERSION="${2}"
    [ ! -z "${3}" ] && PACKAGE_PATH="${3}"
    # 单点升级场景，会传入一个顶层的目录参数，因为仲裁和MAE不使用/opt/oss，会定制这个目录
    [ ! -z "${4}" ] && OSM_PATH="${4}/NCEEngr/O2_path/upgrade_config_${VERSION}"
    # 获取定制的 BUILD_FLAG，WIRELESS参数
    get_custom_para "$@"
    if [[ "${TYPE}" =~ "upgrade" ]];then
        LOG_FILE="${CURRENT_DIR}"/upgrade.log
    else
        LOG_FILE="${CURRENT_DIR}"/rollback.log
    fi
}

##################################################
# Description: 校验执行用户
#              脚本要求使用ossadm用户执行
##################################################
function check_user() {
    local user=$(whoami)
    oss_user=$(id -nu 3001)
    if [ "${user}" != "${oss_user}" ];then
        print_log "ERROR" "${user} is not allowed to execute this operation, please execute by ossadm(uid:3001)." >> "${LOG_FILE}"
        return 1
    fi
    return 0
}

############################################################
# Name:        unzip_pre_upgrade
# Description: 单点升级，将包解压至升级目录
############################################################
function unzip_pre_upgrade() {
    print_log "INFO" "start to unzip ${PACKAGE_PATH} to ${OSM_PATH}." >> "${LOG_FILE}"
    mkdir -p "${OSM_PATH}"
    if [ "${TYPE}" = "single-upgrade" ];then
        [ -d "${OSM_PATH}" ] && rm -rf "${OSM_PATH}"/*
        local cmd="unzip -q -o ${PACKAGE_PATH} -d ${OSM_PATH}"
        ${cmd}
        if [ $? -ne 0 ];then
            print_log "ERROR" "Failed to execute: ${cmd}." >> "${LOG_FILE}"
            return 1
        else
            print_log "INFO" "Success to execute: ${cmd}." >> "${LOG_FILE}"
            return 0
        fi
    fi
}

############################################################
# Name:        execute_operation
# Description: 升级或者操作入口函数，根据TYPE，执行upgrade.sh或者
#              rollback.sh脚本
############################################################
function execute_operation() {
    print_log "INFO" "get params:${TYPE},${VERSION},${PACKAGE_PATH},${OSM_PATH},${BUILD_FLAG},${WIRELESS}." > "${LOG_FILE}"
    if [[ "${TYPE}" =~ "single" ]];then
        # 单点升级，先解压包至指定的目录
        unzip_pre_upgrade
    fi
    local type=$(echo "${TYPE}" | sed "s/single-//g")
    print_log "INFO" "start to execute ${type}.sh" >> "${LOG_FILE}"
    local cmd="timeout ${TIME_OUT} bash ${OSM_PATH}/${type}.sh ${VERSION} ${OSM_PATH} ${BUILD_FLAG} ${WIRELESS}"
    ${cmd}
    local res=$?
    if [ "${res}" -ne 0 ];then
        # 超时执行结果为124
        print_log "ERROR" "Failed to execute: ${cmd}, result:${res}." >> "${LOG_FILE}"
        return 1
    else
        print_log "INFO" "Success to execute: ${cmd}." >> "${LOG_FILE}"
        return 0
    fi
}


############################################################
# Name:        main
# Description: 主函数，入口
#
############################################################
function main() {
    init_param "${@}" || return 1
    # 执行升级或者回滚脚本
    execute_operation
    return $?
}

main "${@}"
exit $?