#!/bin/bash

function show_ntp_status()
{
    systemctl status chronyd
}

########################################
#
# 查询当前NTP是否正常运行
#
########################################
function check_ntp_status()
{
    systemctl status chronyd | grep "Active" | grep "running" &>/dev/null
    if [ $? -ne 0 ]
    then
        echo "chronyd is not running."
        return 1
    fi
    echo "chronyd is running"
    return 0
}

########################################
#
# 查询NTP跟踪是否正常
#
########################################
function check_ntp_track()
{
    chronyc sources | grep '\^\*' &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "chronyd is tracking"
        return 0
    fi
    chronyc sources | grep '\^\?' | grep -w 127.127.1.0 &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "chronyd is tracking"
        return 0
    fi
    echo "chronyd is not tracking"
    return 1
}

########################################
#
# 查询NTP跟踪是否正常，返回结果为ns
#
########################################
function get_ntp_offset()
{
    local tracking_server=$(using_server)
    if [ $? -ne 0 ]
    then
       return 1
    fi
    local offset=$(chronyc sourcestats | grep -w "${tracking_server}" | awk {'print $7'})
    offset=${offset#-}
    offset=${offset#+}
    echo "${offset}" | grep "ns$" &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "${offset}" | sed 's/ns//g'
        return 0
    fi
    echo "${offset}" | grep "us$" &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "${offset}000" | sed 's/us//g'
        return 0
    fi
    echo "${offset}" | grep "ms$" &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "${offset}000000" | sed 's/ms//g'
        return 0
    fi
    echo "${offset}" | grep "s$" &>/dev/null
    if [ $? -eq 0 ]
    then
        echo "${offset}000000000" | sed 's/s//g'
        return 0
    fi
}

function start_ntp()
{
    systemctl start chronyd
    return $?
}

function stop_ntp()
{
    systemctl stop chronyd
    return $?
}

function restart_ntp()
{
    systemctl restart chronyd
    return $?
}

function enable_ntp()
{
    systemctl disable ntpd &>/dev/null
    systemctl stop ntpd &>/dev/null
    systemctl enable chronyd
    return $?
}

function using_server() {
    check_ntp_track &>/dev/null || return 1
    local using_ip=$(chronyc sources | grep '\^\*' | awk {'print $2'})
    echo ${using_ip}
}

function main()
{
    [[ ${1#-} == "cmd" ]] && { cmd=$2; shift 2; }
    if [ "${cmd}" == "show_ntp_status" ]
    then
        show_ntp_status
        return $?
    elif [ "${cmd}" == "check_ntp_status" ]
    then
        check_ntp_status
        return $?
    elif [ "${cmd}" == "check_ntp_track" ]
    then
        check_ntp_track
        return $?
    elif [ "${cmd}" == "offset" ]
    then
        get_ntp_offset
        return $?
    elif [ "${cmd}" == "start" ]
    then
        start_ntp
        return $?
    elif [ "${cmd}" == "stop" ]
    then
        stop_ntp
        return $?
    elif [ "${cmd}" == "restart" ]
    then
        restart_ntp
        return $?
    elif [ "${cmd}" == "enable" ]
    then
        enable_ntp
        return $?
    elif [ "${cmd}" == "using_server" ]
    then
        using_server
        return $?
    fi
}

main $@
exit $?