#! /bin/bash
#########################################################
# Copyright © Huawei Technologies Co., Ltd. 2020-2020. All rights reserved.
# File name: check_arbitration_omp.sh
# Description: 检查用户Easysuite界面输入的omp节点信息是否正确
###########################################################
function get_params()
{
    site_role=$1
    omp_node_ip=$2
    omp_node_uname=$3

    echo "es_input: master_omp_node_pwd"
    read -s master_omp_node_pwd
    echo "es_input: master_omp_node_ossadm_pwd"
    read -s master_omp_node_ossadm_pwd

    echo "es_input: slave_omp_node_pwd"
    read -s slave_omp_node_pwd
    echo "es_input: slave_omp_node_ossadm_pwd"
    read -s slave_omp_node_ossadm_pwd

    if [ "${site_role}" = "-master" ]
    then
        omp_node_pwd=${master_omp_node_pwd}
        omp_node_ossadm_pwd=${master_omp_node_ossadm_pwd}
    else
        omp_node_pwd=${slave_omp_node_pwd}
        omp_node_ossadm_pwd=${slave_omp_node_ossadm_pwd}
    fi

    # 特殊字符转义
    omp_node_pwd=$(echo ${omp_node_pwd} | sed 's/\\/\\\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\\\\"/g')
    omp_node_ossadm_pwd=$(echo ${omp_node_ossadm_pwd} | sed 's/\\/\\\\\\/g' | sed 's/{/\\{/g' | sed 's/}/\\}/g' | sed 's/\[/\\[/g' | sed 's/\$/\\\\\\$/g' | sed 's/`/\\`/g' | sed 's/\"/\\\\\\"/g')
}

function check_ip()
{
    # 检查用户输入的IP是否属于当前站点
    cat /opt/oss/manager/etc/sysconf/nodelists.json | grep -w "${omp_node_ip}" >/dev/null 2>&1
    if [ $? -ne 0 ]
    then
        echo "The omp node ip ${omp_node_ip} does not belong to the current NCE site."
        arbitration_log ERROR "CHECK_OMP_INFO" "The omp node ip ${omp_node_ip} does not belong to the current NCE site."
        return 1
    fi
}

function check_login()
{
    /usr/bin/expect << ENF >/dev/null 2>&1
set timeout 30
spawn /usr/bin/ssh ${omp_node_uname}@${omp_node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send "${omp_node_pwd}\r"}
}

expect {
    "*No route to host*" { exit 101 }
    "*Permission denied*" { exit 102 }
    "*please try again*" { exit 102}
    "*${omp_node_uname}@*" {
                            send "su - ossadm\r"
                            expect {
                                "*yes/no*" {send "yes\r";exp_continue}
                                "*assword:" {send "${omp_node_ossadm_pwd}\r"}
                                "*ossadm@*" {send "\r"}
                            }
    
                            expect {
                                "*ossadm@*" { exit 100 }
                                "*Permission denied*" { exit 103 }
                            }
    }
}


expect eof

catch wait result
exit [lindex $result 3]

ENF
    check_login_result=$?
    if [ -z ${check_login_result} ]
    then
        echo "The user name or password of the ${site_role#*-} OMP node cannot be used to log in."
        arbitration_log ERROR "CHECK_OMP_INFO" "The user name or password of the ${site_role#*-} OMP node cannot be used to log in."
        return 1
    fi
    if [ ${check_login_result} -ne 100 ]
    then
        if [ ${check_login_result} -eq 101 ]
        then
            echo "The connection to the ${site_role#*-} OMP IP address is rejected."
            arbitration_log ERROR "CHECK_OMP_INFO" "The connection to the ${site_role#*-} OMP IP address is rejected."
            return 1
        elif [ ${check_login_result} -eq 102 ]
        then
            echo "The user name or password of the ${site_role#*-} OMP node is incorrect."
            arbitration_log ERROR "CHECK_OMP_INFO" "The user name or password of the ${site_role#*-} OMP node is incorrect."
            return 1
        elif [ ${check_login_result} -eq 103 ]
        then
            echo "Failure to log in to the ${site_role#*-} OMP node as user ossadm."
            arbitration_log ERROR "CHECK_OMP_INFO" "Failure to log in to the ${site_role#*-} OMP node as user ossadm."
            return 1
        else
            echo "Unknow error,please contact Huawei technical support."
            arbitration_log ERROR "CHECK_OMP_INFO" "Unknow error,please contact Huawei technical support."
            return 1
        fi
    fi
}

function check_dr_service()
{
    /usr/bin/expect << ENF >/dev/null 2>&1
set timeout 30
spawn /usr/bin/ssh ${omp_node_uname}@${omp_node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send "${omp_node_pwd}\r"}
}

expect {
    "*${omp_node_uname}@*" {
                            send "su - ossadm\r"
                            expect {
                                "*yes/no*" {send "yes\r";exp_continue}
                                "*assword:" {send "${omp_node_ossadm_pwd}\r"}
                                "*ossadm@*" {send "\r"}
                            }

                            expect {
                                "*ossadm@*" {
                                send "\`test -d /opt/oss/manager/apps/DRService/;echo result=\$?\`\r"
                                expect {
                                	"result=0" { exit 100 }
                                	"result=1" { exit 255 }
                                	}
                                }
                            }
    }
}


expect eof

catch wait result
exit [lindex $result 3]

ENF
    check_dr_result=$?
    if [ -z ${check_dr_result} ]
    then
        echo "Checking Whether the DRService Installation Fails on node ${omp_node_ip}."
        arbitration_log ERROR "CHECK_OMP_INFO" "Checking Whether the DRService Installation Fails on node ${omp_node_ip}."
        return 1
    fi
    if [ ${check_dr_result} -ne 100 ]
    then
        echo "The entered node ${omp_node_ip} is not installed with DRService."
        arbitration_log ERROR "CHECK_OMP_INFO" "The entered node ${omp_node_ip} is not installed with DRService."
        return 1
    fi

}

function check_ip_float()
{
    /usr/bin/expect << ENF >/dev/null 2>&1
set timeout 30
spawn /usr/bin/ssh ${omp_node_uname}@${omp_node_ip}
expect {
    "*yes/no*" {send "yes\r";exp_continue}
    "*assword:" {send "${omp_node_pwd}\r"}
}

expect {
    "*${omp_node_uname}@*" {
                            send "su - ossadm\r"
                            expect {
                                "*yes/no*" {send "yes\r";exp_continue}
                                "*assword:" {send "${omp_node_ossadm_pwd}\r"}
                                "*ossadm@*" {send "\r"}
                            }

                            expect {
                                "*ossadm@*" {
                                send "\`ip addr | grep -w \"inet ${omp_node_ip}\" | egrep \"eth\[0-9\]+:\[0-9\]+\" > /dev/null 2>&1;echo result=\$?\`\r"
                                expect {
                                	"result=0" { exit 255 }
                                	"result=1" { exit 100 }
                                	}
                                }
                            }
    }
}


expect eof

catch wait result
exit [lindex $result 3]

ENF
    check_float_ip_result=$?
    if [ -z ${check_float_ip_result} ]
    then
        echo "Checking Whether the IP is float ip Fails on node ${omp_node_ip}."
        arbitration_log ERROR "CHECK_OMP_INFO" "Checking Whether the IP is float ip Fails on node ${omp_node_ip}."
        return 1
    fi
    if [ ${check_float_ip_result} -ne 100 ]
    then
        echo "The ip address ${omp_node_ip}(Management Node IP Address) is float ip, please modify your input."
        arbitration_log ERROR "CHECK_OMP_INFO" "The ip address ${omp_node_ip}(Management Node IP Address) is float ip, please modify your input."
        return 1
    fi

}

function main()
{
    . ./arbitration_opertate_lib

    arbitration_log INFO "CHECK_OMP_INFO" "Start to check the OMP node information entered by the user."

    # 获取参数
    get_params $*

    # 检查当前输入的OMP IP是否属于当前NCE站点
    check_ip || return 1

    # 检查当前的用户名密码是否可以登录成功
    check_login || return 1

    # 检查当前输入的节点是否安装DRService
    check_dr_service || return 1

    # 检查输入的IP是否是浮动IP
    check_ip_float || return 1

    arbitration_log INFO "CHECK_OMP_INFO" "Finish to check the OMP node information entered by the user."
}

main $*
exit $?


