#!/bin/bash

if [[ ${USER} != "root" ]]; then
    echo "only the root user can run this script, current the user is ${USER}"
    exit 1
fi

SOURCE="${BASH_SOURCE[0]}"
SCRIPT_FILE_PATH=`/usr/bin/readlink -e ${SOURCE}`
SCRIPT_DIR=`/usr/bin/dirname ${SCRIPT_FILE_PATH}`
APP_PATH=`/usr/bin/dirname ${SCRIPT_DIR}`

help()
{
cat << HELP
modify_hosts -- Modify the "/etc/hosts" file after changing
                the FusionInsight Manager's IP address.

USAGE: modify_hosts.sh [OPTIONS]
Options are:
 -h         display this help text and exit

HELP
exit 0
}

error()
{
echo "$1"
exit 1
}

if [ $# -gt 1 ]; then
    echo "error: unsupported option. -h for help"
    exit 1
fi

while [ -n "$1" ]; do
case $1 in
-h) help;shift 1;;
*) error "error: unsupported option $1. -h for help";;
esac
done

###################################################
#
# 检测ip
#
###################################################
function check_ip()
{
    for IP in $(echo $1 | sed 's/,/ /g')
    do
        valid_ip "${IP}"
        if [ $? -ne 0 ];then
            return 1
        fi
    done

    return 0
}

###################################################
#
# 检查配置的ip是否合法
#
###################################################
function valid_ip()
{
    local ipdot=0
    local ipcount=0

    if [ -z "$1" ]; then
        echo "INFO" $LINENO "IP is null."
        return 1
    fi

    #检查有多少个.号
    ipdot=$(echo $1 | awk -F '.' '{print NF-1}')
    if [ "${ipdot}" -ne 3 ]; then
        echo "INFO" $LINENO "Ip:$1 is invalid."
        return 1
    fi

    for ipvar in $(echo $1 | awk -F. '{print $1, $2, $3, $4}')
    do
        echo "${ipvar}" | grep -E "^0$|^[1-9][0-9]*$" >/dev/null
        if [ $? -ne 0 ]; then
            echo "INFO" $LINENO "Ip:$1 is invalid."
            return 1
        fi

        #逐个检查是否在0-255范围内，注意：该检查包含0.0.0.0和255.255.255.255合法
        if [ "${ipvar}" -ge 0 -a "${ipvar}" -le 255 ] ; then
            ((ipcount=ipcount+1))
            continue
        else
            echo "INFO" $LINENO "Ip:$1 is invalid."
            return 1
        fi
    done

    #如果没有四个数字，认为非法
    if [ ${ipcount} -ne 4 ]; then
        echo "INFO" $LINENO "Ip:$1 is invalid."
        return 1
    fi

    return 0
}

###################################################
#
# 恢复NTP配置文件，规避NTP问题
#
###################################################
function rec_ntp_conf() {
    if [ ! -e /tmp/ntp_conf_backup/ntp.conf ]; then
        return 0
    fi
    /bin/cp -f /tmp/ntp_conf_backup/ntp.conf /etc/ntp.conf
    if [ $? -ne 0 ]; then
        echo "Recovery ntp conf fail. Error code is $?."
        return 1
    fi
    service ntpd restart
    rm -rf /tmp/ntp_conf_backup
    return 0
}

###################################################
#
# 记录操作到系统日志
#
###################################################
function sys_log()
{
    declare source="$0"
    [ -e "$source" ] && source=`/usr/bin/readlink -e $source`
    if [ $# -gt 1 ]; then
        shift 1
        source="$source $@"
    fi

    OPERIP=`who am i | awk '{print $NF}' | sed 's/[()]//g'`
    if [[ "$1" == "0" ]]
    then
        logger -t $USER -p local0.info "[$source] return code=[$1], execute success from ${OPERIP:-127.0.0.1}"
    else
        logger -t $USER -p local0.info "[$source] return code=[$1], execute failed from ${OPERIP:-127.0.0.1}"
    fi
}
###################################################
#
# ODAE提供一个脚本用来合并/etc/hosts中相同IP的两行，产品在修改IP后用root用户在有etcd的节点执行该脚本即可。
#
#实现逻辑：
#1、	可以用*cluster.local来判断是cloudsop的域名
#2、	如果cloudsop域名对应的ip有两行配置，就删除cloudsop这行配置，把cloudsop的域名添加到另外一行最后面，实现合并操作
#
###################################################
function main()
{
    rec_ntp_conf
    #查找出含有cloudsop域名，并且以该域名结尾的，且只有两列的行
    result=$(awk -F " " '$2~/cluster.local$/{print;}' /etc/hosts | grep -w 'cluster.local$')
    if [ -z "$result" ]; then
        echo "No modification is required."
        return 0
    fi
    local cloudsop_host_array=()
    local index=0
    for item in ${result[@]}
    do
        cloudsop_host_array[$index]=$item
        ((index=index+1))
    done
    if [ ${#cloudsop_host_array[@]} -eq 0 ]; then
        echo "No modification is required."
        return 0
    fi

    local avoid_success=1
    for ((i=0;i<${#cloudsop_host_array[@]};i=i+2))
    do
        let j=$i+1
        cloudsop_ip=${cloudsop_host_array[${i}]}
        check_ip "${cloudsop_ip}"
        if [ $? -ne 0 ];then
            continue
        fi
        cloudsop_domain=${cloudsop_host_array[${j}]}
        #查找以cloudsop_ip开始的，并且不包含cloudsop_domain的行，然后在该行最后增加上cloudsop_domain
        fi_host=$(cat /etc/hosts | grep -v "${cloudsop_domain}" | grep -w "^${cloudsop_ip}" )
        if [ -z "$fi_host" ]; then
            continue
        fi
        local fi_item="$fi_host $cloudsop_domain"
        #删除cloudsop的行
        sed -i '/'"^${cloudsop_ip}.[ ]*${cloudsop_domain}$"'/d' /etc/hosts
        #替换fi的行
        sed -i 's/'"${fi_host}"'/'"${fi_item}"'/g' /etc/hosts
        avoid_success=0
    done
    if [ ${avoid_success} -eq 0 ]; then
        echo "Modification successful."
    fi
    return 0
}

main
result=$?
sys_log $result $@
exit $result
