#!/bin/bash

# Name             : common.inc
# Description      : a common include shell
# Creation date    : 16-Nov-2009
# Modification log :
#

#=================================================
# Global variable

GLOBAL_ROOT_PARTITION="/dev/sda1"
GLOBAL_BOOT_PARTITION="/dev/sda2"
GLOBAL_BASH_PID=$$

#=================================================


if [ -d "${ABSOLUATELY_PATH}" ]
then
    CUR_PATH="${ABSOLUATELY_PATH}"
else
    CUR_PATH=`pwd`
fi

if [ -z "${MODULE_NAME}" ]
then
    MODULE_NAME="default"
fi

OS_TYPE=`uname | tr '[:upper:]' '[:lower:]'`
if [ "${OS_TYPE}" = "linux" ]
then
    MGR_ROOT=${CUR_PATH%/*/*}
else
    MGR_ROOT="/opt"
fi

#export env
. ${MGR_ROOT}/bin/engr_profile.sh

MGR_SCRIPT_PATH="${MGR_ROOT}/tools/shscript"
OSMU_SCRIPT_COMMON_PATH="${MGR_SCRIPT_PATH}/common"
DLDEP_ROOT="${MGR_ROOT}/q_deployer"
DLDEP_INSTANCE_PATH="${DLDEP_ROOT}/instances"
DLDEP_PXECFG_PATH="${DLDEP_ROOT}/tftpboot/pxelinux.cfg"
DLDEP_PXE_DEFAULT_CONFIG="${DLDEP_PXECFG_PATH}/virtualos"
DLDEP_REPOSITORY_PATH="${DLDEP_ROOT}/repository"
DLDEP_WWN_RESULT_PATH="${DLDEP_REPOSITORY_PATH}/WWN"
DLDEP_HOOK_PATH=
LOGIN_MESSAGE="${MGR_ROOT}/tmp/OSMUMessage.log"
CMMD=`bash ${MGR_ROOT}/agent/tools/shscript/manager/suswitch.sh "su"`
DATE_PLUS_INFO="`date '+%Y%m%d%H%M%S'`"
RANDOM_INFO="${DATE_PLUS_INFO}${RANDOM}"
LOGPATH="$_APP_LOG_DIR"
THELOGFILE="${LOGPATH}/${MODULE_NAME}.log"

TMP_CONFIG_PATH="${LOGPATH}/tmp"
TMP_CONFIG_FILE="${TMP_CONFIG_PATH}/tmpconfig_${RANDOM_INFO}.log"
TMP_CONTENT_FILE="${TMP_CONFIG_PATH}/tmpcontent_${RANDOM_INFO}.log"
OPERATION_CODE_FILE="${OSMU_SCRIPT_COMMON_PATH}"/operation_code.inc
TMP_BUFFER_PATH="${LOGPATH}/tmp/tmpbuffer_${RANDOM_INFO}"
[ -f /etc/profile ] && . /etc/profile > /dev/null 2>&1


if [ "${OS_TYPE}" = "linux" ]
then
    DLDEP_HOOK_PATH="${DLDEP_ROOT}/hook/atca"
    #maintenance log folder, the format of log file should be like this "XXXX_timestamp"
    MAX_LOG_COUNT="12"
    ALL_PREFIX_NAME=`ls -td "${TMP_CONFIG_PATH}"/* 2>/dev/null |xargs -I '{}' -s 100000 basename '{}' | awk -F"_" '{print $1}' | sort -u`
    for TMP_PREFIX_NAME in ${ALL_PREFIX_NAME}
    do
        ls -td "${TMP_CONFIG_PATH}"/* | grep "${TMP_PREFIX_NAME}_.*" | tail -n+${MAX_LOG_COUNT} | xargs -I '{}' -s 100000 rm -r '{}' > /dev/null 2>&1
    done
    [ -f "${OSMU_SCRIPT_COMMON_PATH}/error_code.inc" ] && . "${OSMU_SCRIPT_COMMON_PATH}/error_code.inc" > /dev/null 2>&1
else
    DLDEP_HOOK_PATH="${DLDEP_ROOT}/hook/initrd"
    [ -f "${OSMU_SCRIPT_COMMON_PATH}/error_code_qi.inc" ] && . "${OSMU_SCRIPT_COMMON_PATH}/error_code_qi.inc" > /dev/null 2>&1
fi


function FUNC_CLEAN_TMP_FILE()
{
    rm -f "${TMP_CONFIG_FILE}"
    rm -f "${TMP_CONTENT_FILE}"
    rm -rf "${TMP_BUFFER_PATH}"
    fn_remove_tmp_file > /dev/null 2>&1
    return 0;
}

if [ ! -d "${TMP_BUFFER_PATH}" ]
then
    mkdir -p "${TMP_BUFFER_PATH}"
fi

#*************************************************************#
# Name:        getSysTime                                     #
# Description: get the system time                            #
# Inout:                                                      #
# Ouput:   Time: 2009-11-03 09:09:09                          #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function fn_getSysTime()
{
    date "+%Y-%m-%d %T"
}


#*************************************************************#
# Name:        LOG_NO_SED                                     #
# Description: record the message into the logfile without sed#
# Inout:                                                      #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2013-01-26                                         #
#*************************************************************#
function LOG_NO_SED()
{
	local strTime=`fn_getSysTime` 
	local curSize
	local maxSize=10240
	local maxRecord=3
	local index=2
	local content="$*"
	
	content=`echo "${content}" | sed -e "s#LICENSE.*=.*#LICENSE=#g" \
	    -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" \
        -e "s#SR_SApswd.*=.*#SR_SApswd_pvalue=#g" \
        -e "s#SR_UPword.*=.*#SR_UPword_pvalue=#g"`

	if [ -z ${LOGFILE} ]
	then
		echo  -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}"
	elif [  -f ${LOGFILE} ]
	then
		curSize=`ls -lk ${LOGFILE} | awk -F" " '{print $5}'`
		if [ ${curSize} -ge ${maxSize} ]
		then
			while [ ${index} -gt 0 ]
			do
				if [ -f ${LOGFILE}.${index} ] 
				then
					mv ${LOGFILE}.${index} ${LOGFILE}.`expr ${index} + 1`
					
				fi
				let index--
			done
			mv ${LOGFILE} ${LOGFILE}.1
		fi
		echo -e  "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
	else
		mkdir -p `dirname ${LOGFILE}`
        chmod 750 `dirname ${LOGFILE}`
		echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
	fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
# Inout:  warning message                                     #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2013-01-26                                         #
#*************************************************************#
function WARN_NO_SED()
{
    LOG_NO_SED "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
# Inout:  error message                                       #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2013-01-26                                         #
#*************************************************************#
function ERROR_NO_SED()
{
    LOG_NO_SED "ERROR: $*"
}

#*************************************************************#
# Name:         LOG                                           #
# Description: record the message into the logfile            #
# Inout:  log message                                         #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function LOG()
{
    local strTime=`fn_getSysTime`
    local curSize
    local maxSize=10240
    local maxRecord=3
    local index=2
    local content="$*"
	
    content=`echo "${content}" | sed -e "s#LICENSE.*=.*#LICENSE=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd].*=.*#pvalue=#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" \
        -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" \
        -e "s#SR_SApswd.*=.*#SR_SApswd_pvalue=#g" \
        -e "s#SR_UPword.*=.*#SR_UPword_pvalue=#g" \
        -e "s#SNMP_COMMUNITY.*=.*#SNMP_COMMUNITY=#g" \
        -e "s#[Pp][Ww][Dd].*=.*#pvalue=#g"`

    if [ -z ${LOGFILE} ]
    then
        echo  -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}"
    elif [  -f ${LOGFILE} ]
    then
        curSize=`ls -lk ${LOGFILE} | awk -F" " '{print $5}'`
        if [ ${curSize} -ge ${maxSize} ]
        then
            while [ ${index} -gt 0 ]
            do
                if [ -f ${LOGFILE}.${index} ]
                then
                    mv ${LOGFILE}.${index} ${LOGFILE}.`expr ${index} + 1`

                fi
                let index--
            done
            mv ${LOGFILE} ${LOGFILE}.1
        fi
        echo -e  "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
        sed -i -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" ${LOGFILE}	
    else
        mkdir -p `dirname ${LOGFILE}`
        chmod 750 `dirname ${LOGFILE}`
        echo -e "[${strTime}] [${GLOBAL_BASH_PID}] ${content}" | tee -a ${LOGFILE}
        sed -i -e "s#[Pp][Aa][Ss][Ss][Ww][Oo][Rr][Dd]#pvalue#g" -e "s#[Pp][Aa][Ss][Ss][Ww][Dd]#pvalue#g" ${LOGFILE}
    fi
}

#*************************************************************#
# Name:        WARN                                           #
# Description: record the warning message into the logfile    #
# Inout:  warning message                                     #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function WARN()
{
    LOG "WARN: $*"
}

#*************************************************************#
# Name:        ERROR                                          #
# Description: record the error message into the logfile      #
# Inout:  error message                                       #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function ERROR()
{
    LOG "ERROR: $*"
}

# Name        : PRINT_LOG
# Description : print log
# Parameters  : $1 the print level
# Parameters  : $2 the log
function PRINT_LOG()
{
    local LOG_LEVEL=$1
    local LOG_INFO=$2

    if [ -z "${LOGFILE}" ]
    then
        LOGFILE="${THELOGFILE}"
    fi

    case ${LOG_LEVEL} in
        "INFO")
            LOG "${LOG_INFO}"
        ;;
        "WARN")
            WARN "${LOG_INFO}"
        ;;
        "FATAL")
            ERROR "${LOG_INFO}"
        ;;
        *)
            WARN " The LOG_LEVEL must be <INFO|WARN|FATAL>, it will be set to WARN by default.."
            WARN "${LOG_INFO}"
        ;;
    esac
}

# Name        : PRINT_FILE_TO_LOG
# Description : print file content into log file
# Parameters  : $1 the file to be printed
function PRINT_FILE_TO_LOG()
{
    local FILE_NAME=$1

    if [ -z "${LOGFILE}" ]
    then
        LOGFILE="${THELOGFILE}"
    fi

    if [ -f "${FILE_NAME}" ]
    then
        CHECK_CONTENT=`cat ${FILE_NAME}`
        if [ -z "${CHECK_CONTENT}" ]
        then
            LOG "This file FILE_NAME<${FILE_NAME}> is empty!!!"
            return ${ERR_FILEISNULL};
        else
            LOG "Print <${FILE_NAME}>"
			LOG "`cat \"${FILE_NAME}\" | sed 's/\r//g'`"
        fi
    else
        LOG "This file FILE_NAME<${FILE_NAME}> is not exist!!!"
        return ${ERR_FILEEXIST};
    fi
}

# Name        : fn_check_status
# Description : check the return result
# Parameters  : $1 return value
function fn_check_status()
{
    local ERROR_CODE=$1

    echo -en "\\033[65G"
    if [ ${ERROR_CODE} = 0 ]
    then
        echo -en "\\033[1;32m[done]"
        echo -e "\\033[0;39m"
    else
        echo -en "\\033[1;31m[fail]"
        echo -e "\\033[0;39m"
        FUNC_CLEAN_TMP_FILE
        exit ${ERROR_CODE};
    fi
}

# Name        : fn_title
# Description : print the title in white color
# Parameters  : $1 the content that you want to display
#             :
function fn_title()
{
    echo -en "\033[1m"
    echo -e  "======================================================"
    echo -en $@
    echo -en "\033[0m"

    PRINT_LOG "INFO" "$@"
    PRINT_LOG "INFO" " "
}

# Name        : fn_validate_ip
# Description : check the ip is validate or not
# Parameters  : $1 ip address
function fn_validate_ip()
{
    local IP_ADDR=$1

    echo $IP_ADDR | grep -E "^([0-9]{1,3}\.){3}[0-9]{1,3}$" >/dev/null 2>&1
    if [[ $? -eq 0 ]]
    then
        IFS_TMP=$IFS
        IFS='.'
        IP_ADDR=($IP_ADDR)
        IFS=$IFS_TMP
        if [[ ${IP_ADDR[0]} -ge "0" && ${IP_ADDR[0]} -le "255" \
            && ${IP_ADDR[1]} -ge "0" && ${IP_ADDR[1]} -le "255" \
            && ${IP_ADDR[2]} -ge "0" && ${IP_ADDR[2]} -le "255" \
            && ${IP_ADDR[3]} -ge "0" && ${IP_ADDR[3]} -le "255" ]]
        then
            PRINT_LOG "INFO" "The IP_ADDR<${IP_ADDR[0]}.${IP_ADDR[1]}.${IP_ADDR[2]}.${IP_ADDR[3]}> is OK!"
            return 0;
        fi
    fi

    PRINT_LOG "WARN" "The IP_ADDR<${IP_ADDR[0]}.${IP_ADDR[1]}.${IP_ADDR[2]}.${IP_ADDR[3]}> is invalid!"
    return ${ERROR_PARAM};
}

# Name        : fn_check_ip
# Description : check the ip can be connected or not
# Parameters  : $1 a ip address
function fn_check_ip()
{
    local IP_ADDR=$1
    local TMP_RET

    if [ -z "${IP_ADDR}" ] || [ "${IP_ADDR}" = "NULL" ]
    then
        PRINT_LOG "WARN" "The IP_ADDR<${IP_ADDR}> is null!"
        return ${ERR_IPREACH};
    fi

    if [ "${OS_TYPE}" = "linux" ]
    then
        ping -c3 "${IP_ADDR}" > /dev/null 2>&1
        TMP_RET=$?
    else
        ping -n 3 "${IP_ADDR}" -w 5 > /dev/null 2>&1
        TMP_RET=$?
    fi

    if [ "${TMP_RET}" != 0 ]
    then
        PRINT_LOG "WARN" "The IP_ADDR<${IP_ADDR}> is unreachable!"
        return ${ERR_IPREACH};
    fi

    PRINT_LOG "INFO" "The IP_ADDR<${IP_ADDR}> is reachable!"
    return 0;
}

# Name        : fn_get_value
# Description : get the config from osmu
# Parameters  : $1 the absoluately path of config file
#             : $2 the key name
function fn_get_value()
{
    local TMP_CONFIG_FILE=$1
    local TMP_KEY_NAME=$2
    local TMP_RET
    [ ! -f "${TMP_CONFIG_FILE}" ] && { ERROR "The file TMP_CONFIG_FILE<${TMP_CONFIG_FILE}> is not exist" &>/dev/null; return ${ERROR_PARAM}; }
    cat "${TMP_CONFIG_FILE}" | grep -m1 "^${TMP_KEY_NAME}=" | awk -F'=' '{print $1}' > "${TMP_CONTENT_FILE}_tmp"
     egrep -i ".*password.*|.*pwd.*|.*passwd.*|.*pswd.*|.*pword.*|.*snmp_community.*" "${TMP_CONTENT_FILE}_tmp" > /dev/null 2>&1
    if [ $? -eq 0 ]
    then
        rm -f "${TMP_CONTENT_FILE}_tmp" &>/dev/null
        . "${OSMU_SCRIPT_COMMON_PATH}/translate.sh"
        TMP_RET=`cat "${TMP_CONFIG_FILE}" | grep -m1 "^${TMP_KEY_NAME}=" | awk 'BEGIN {FS="="}{print $2}'`
        [ -z "${TMP_RET}" ] || fn_translate "${TMP_RET}"
    else
        rm -f "${TMP_CONTENT_FILE}_tmp" &>/dev/null
        cat "${TMP_CONFIG_FILE}" | grep -m1 "^${TMP_KEY_NAME}=" | awk 'BEGIN {FS="="}{print $2}'
    fi

}

# Name        : fn_check_config_file
# Description : get the config from osmu
# Parameters  : $1 the absoluately path of config file
function fn_check_config_file()
{
    local CONFIG_FILE=$1

    if [ ! -f "${CONFIG_FILE}" ]
    then
        PRINT_LOG "FATAL" "The config file <${CONFIG_FILE}> could not found!"
        return ${ERR_FILEEXIST};
    fi

    PRINT_LOG "INFO" "loading config file <${CONFIG_FILE}>..."

    cp -f "${CONFIG_FILE}" "${TMP_CONFIG_FILE}"
    cd "${LOGPATH}"

    dos2unix "${TMP_CONFIG_FILE}" "${TMP_CONFIG_FILE}" > /dev/null 2>&1
    if [ $? != 0 ]
    then
        PRINT_LOG "WARN" "loading failure <dos2unix ${TMP_CONFIG_FILE} ${TMP_CONFIG_FILE}> <${LOGPATH}>"
        return ${ERR_DECODE}
    fi

    sed 's/= */=/g' "${TMP_CONFIG_FILE}" > "${TMP_CONFIG_FILE}_tmp"
    mv "${TMP_CONFIG_FILE}_tmp" "${TMP_CONFIG_FILE}"
    PRINT_FILE_TO_LOG "${TMP_CONFIG_FILE}"
    fn_check_status $?
    PRINT_LOG "INFO" "loading config file <${CONFIG_FILE}>... OK!"
    return 0;
}

# Name        : fn_dldep_service
# Description : start or stop dldep serivce
# Parameters  : $1 the absoluately path of config file
function fn_dldep_service()
{
    local SERVICE_FLAG=$1

    local EXEC_SCRIPT

    if [ "${SERVICE_FLAG}" != "start" ] && [ "${SERVICE_FLAG}" != "stop" ]
    then
        PRINT_LOG "WARN" "The SERVICE_FLAG should be <start> or <stop>!!!"
        return 1;
    fi

    if [ "${SERVICE_FLAG}" = "start" ]
    then
        EXEC_SCRIPT="${MGR_SCRIPT_PATH}/tools/startdldep.sh"
    elif [ "${SERVICE_FLAG}" = "stop" ]
    then
        EXEC_SCRIPT="${MGR_SCRIPT_PATH}/tools/stopdldep.sh"
    fi

    bash "${EXEC_SCRIPT}" > "${TMP_CONTENT_FILE}" 2>&1
    if [ $? != 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Excute<bash \"${EXEC_SCRIPT}\"> failure!!!"
        return ${ERR_EXECSHELL};
    else
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "INFO" "Excute<bash \"${EXEC_SCRIPT}\"> OK!!!"
    fi

    return 0;
}

# Name        : fn_print_process
# Description : print process of a task
# Parameters  : $1 the parent process id
function fn_print_process()
{
    local TOTAL_COUNT=$1
    local INDEX_COUNT=$2
    local LASTTIME_TMP_POS=$3
    local PRINT_TYPE=$5

    local BEGIN_NUM
    local END_NUM
    local ROW_NUM

    if [ "${PRINT_TYPE}" = "newline" ]
    then
        printf "\033[m\n"
    else
        BEGIN_NUM=1
        END_NUM=50
        ROW_NUM=100

        POS2=`expr $END_NUM - 1`
        POS1=`echo "scale=2; ($INDEX_COUNT/$TOTAL_COUNT)*${POS2} + 1" | bc `

        TMP_CHECK=`echo "scale=2; $POS2 - $POS1 + 1" | bc`
        if [ "${TMP_CHECK}" -lt 2 ]
        then
            POS1=$POS2
        fi

        POS1=`printf "%.0f" ${POS1}`

        TMP_POS="${LASTTIME_TMP_POS}"
        if [ "${TMP_POS}" = 0 ]
        then
            TMP_POS=`expr ${BEGIN_NUM} + 1`
        fi

        if [ "${POS1}" -lt 2 ]
        then
            mid=`echo "($BEGIN_NUM+$END_NUM)/2-2" |bc`
            pro=0
            printf "\033[${ROW_NUM};${mid}H$pro%%\033[${ROW_NUM};2H:"
        else
            while [ "${TMP_POS}" -lt "${POS1}" ]
            do
                mid=`echo "($BEGIN_NUM+$END_NUM)/2-2" |bc`
                pro=`echo "scale=0; ($TMP_POS - $BEGIN_NUM)*100/($POS2 - $BEGIN_NUM)" | bc`
                echo -en "\033[1m"
                printf "\033[${ROW_NUM};${mid}H$pro%%\033[${ROW_NUM};${TMP_POS}H:"
                echo -en "\033[0m"
                let TMP_POS++
            done
        fi
        eval $4=${TMP_POS}
    fi
}

# Name        : fn_kill_ptree
# Description : invoke fn_kill_ptree function iterative to kill the by pid throught ps tree
# Parameters  : $1 the parent process id
function fn_kill_ptree()
{
    local PARENT_PROCESS_ID=$1
    local CHILD_PROCESS_LIST

    CHILD_PROCESS_LIST=`ps -ef |grep " ${PARENT_PROCESS_ID} " | grep -v grep | awk 'BEGIN {FS=" "}{print $2}' | grep -v "${PARENT_PROCESS_ID}"`
    for PARENT_PROCESS_ID in ${CHILD_PROCESS_LIST}
    do
        fn_kill_ptree "${PARENT_PROCESS_ID}"
        kill -2 "${PARENT_PROCESS_ID}"
        kill -9 "${PARENT_PROCESS_ID}"

        PRINT_LOG "INFO" "Exec <kill -2 \"${PARENT_PROCESS_ID}\" && kill -2 \"${PARENT_PROCESS_ID}\"> OK!"
    done
    return 0;
}

# Name        : fn_exec_remote_cmd
# Description : execute command on remote server
# Parameters  : $1 the ip address of remote server
#             : $2 the login user name
#             : $3 the login user password
#             : $4 the command for execute
function fn_exec_remote_cmd()
{
    local DISK_IP=$1
    local USER_NAME=$2
    local USER_PASSWORD=$3
    local TMP_COMMAND=$4
    local CHECK_RESULT

    cd "${GLOBAL_ARRAY_EXPECT_PATH}"
    bash exeRemote_cmd.sh "${DISK_IP}" "${USER_NAME}" "${USER_PASSWORD}" "${TMP_CONTENT_FILE}" "${TMP_COMMAND}"
    if [ $? != 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Exec<bash exeRemote_cmd.sh \"${DISK_IP}\" \"${USER_NAME}\" \"${TMP_CONTENT_FILE}\" \"${TMP_COMMAND}\"> failure!"
        return ${ERROR_CLI_FAILED};
    fi

    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
    echo "${GLOBAL_ARRAY_EXPECT_PATH}" | grep "S3900"
    if [ $? = 0 ]
    then
        CHECK_FLAG="CLI-Error"
    else
        CHECK_FLAG="Error"
    fi

    CHECK_RESULT=`cat "${TMP_CONTENT_FILE}" | grep -i "${CHECK_FLAG}"`
    if [ -n "${CHECK_RESULT}" ]
    then
        PRINT_LOG "WARN" "Exec<bash exeRemote_cmd.sh \"${DISK_IP}\" \"${USER_NAME}\" \"${TMP_CONTENT_FILE}\" \"${TMP_COMMAND}\"> failure!"
        return ${ERROR_CLI_FAILED};
    fi

    PRINT_LOG "INFO" "Exec<bash exeRemote_cmd.sh \"${DISK_IP}\" \"${USER_NAME}\" \"${TMP_CONTENT_FILE}\" \"${TMP_COMMAND}\"> OK!"
    return 0;
}

# Name        : fn_create_result_file
# Description : create a result file while it is not exist or restore the result file while it is exist
# Parameters  : $1 the absoluately path of config file
function fn_create_result_file()
{
    local RESULT_FILE=$1
    if [ -z "`echo "${RESULT_FILE}" | xargs`" ] 
    then
        PRINT_LOG "FATAL" "The param RESULT_FILE<${RESULT_FILE}> could not found!"
        return ${ERROR_PARAM}
    fi
	
    local RESULT_PATH

    if [ ! -f "${RESULT_FILE}" ]
    then
        RESULT_PATH=`dirname "${RESULT_FILE}"`
        if [ ! -d "${RESULT_PATH}" ]
        then
            mkdir -p "${RESULT_PATH}"
        fi
    fi

    echo "" > "${RESULT_FILE}"
    if [ ! -f "${RESULT_FILE}" ]
    then
        PRINT_LOG "INFO" "Create result file... Fail!"
        return ${ERR_MKDIR};
    fi
    PRINT_LOG "INFO" "Create result file... OK!"
    return 0;
}

# Name        : fn_check_result_to_file
# Description : print result to result file
# Parameters  : $1 the error code
#             : $2
function fn_check_result_to_file()
{
    local ERROR_CODE=$1
    local MODULE_NAME=$2
    local RESULT_FILE=$3

    local TMP_CONTENT

    echo -en "\\033[65G"
    if [ "${ERROR_CODE}" = "0" ]
    then
        echo -en "\\033[1;32m[done]"
        echo -e "\\033[0;39m"
    else
        echo -en "\\033[1;31m[fail]"
        echo -e "\\033[0;39m"
    fi

    if [ -n "${MODULE_NAME}" ]
    then
        TMP_CONTENT=`cat "${RESULT_FILE}" | grep "^${MODULE_NAME}=.*"`
        if [ -z "${TMP_CONTENT}" ]
        then
            echo "${MODULE_NAME}=${ERROR_CODE}" >> "${RESULT_FILE}"
        else
            TMP_CONTENT=`cat "${RESULT_FILE}" | grep "^${MODULE_NAME}=0\>"`
            if [ -n "${TMP_CONTENT}" ]
            then
                #it need to make sure the return result is accord to the final procedure
                if [ "${ERROR_CODE}" -ne "0" ]
                then
                    sed "s/^${MODULE_NAME}=.*/${MODULE_NAME}=${ERROR_CODE}/g" "${RESULT_FILE}" > "${RESULT_FILE}.${RANDOM_INFO}"
                    TMP_CONTENT=`cat "${RESULT_FILE}.${RANDOM_INFO}"`
                    if [ -n "${TMP_CONTENT}" ]
                    then
                        mv "${RESULT_FILE}.${RANDOM_INFO}" "${RESULT_FILE}"
                    fi
                fi
            fi
        fi
    fi

    return ${ERROR_CODE};
}

# Name        : fn_clean_reference_variable
# Description : clean the reference variable before get it from configuration file
# Parameters  : $@ ALL the variable name
function fn_clean_reference_variable()
{
    local VARIABLE_NAME

    local TMP_CONTENT

    for VARIABLE_NAME in $@
    do
        TMP_CONTENT=`echo "${VARIABLE_NAME}" | grep "^REF_"`
        if [ -n "${TMP_CONTENT}" ]
        then
            eval ${VARIABLE_NAME}=""
        fi
    done

    return 0;
}

function fn_Modify_known_hosts_file()
{
    local IP_ADDR=$1

    if [ "${OS_TYPE}" = "linux" ]
    then
    sed -i -e "s/${IP_ADDR}.*//g" -e "/^$/d" /root/.ssh/known_hosts
    else
        sed -i -e "s/${IP_ADDR}.*//g" -e "/^$/d" "${HOME}/.ssh/known_hosts"
        chmod 600 "${HOME}/.ssh/known_hosts"
    fi
}

function fn_LoginSMM()
{
    local server=$1
    local user=$2
    local passwd=$3
    local tempfile="${ABSOLUATELY_PATH}/autossh_${RANDOM}.sh"
    local result
    shift
    shift
    shift
    cmd=$@

    fn_Modify_known_hosts_file "${server}"

    cat << EOF > "${tempfile}"
#!/usr/bin/expect
set username "${user}"
set passwd "${passwd}"
set server "${server}"
set cmd "$@"
set EXT "exit"
spawn ssh \${username}@\${server} "[lindex \$argv 0]" ;
expect {
    "*(yes/no)?*"
    {
        send "yes\n"
        exp_continue
    }
    "*Account locked*"
    {
        exit ${ERR_SMM_LOGIN}
    }
    "*assword:*"
    {
        sleep 1
        send  "\${passwd}\r"
    }
}

sleep 2
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*#"
    {
        send "\${cmd}\n"
    }
    "*>"
    {
        send  "\${cmd}\n"
    }
}
expect {
    "*#"
    {
        send "\${EXT}\n"
    }
    "*Y/N]:"
    {
        send  "Y\n"
    }
}
expect {
    "*#"
    {
        send "\${EXT}\n"
    }
    "*>"
    {
        send "\${EXT}\n"
    }
}
EOF
    chmod 750 ${tempfile}    
    ${tempfile}
    result=$?
    if [ "${result}" != 0 ]
    then
        PRINT_LOG "WARN" "Remote login ${server} failed."
    fi
    rm -f "${tempfile}" > /dev/null 2>&1
    return ${result}
}

function fn_check_blade_status()
{
    local TMP_SLOT=$1
    local TMP_SMM_IP=$2
    local STATUS

    ssh -n ${TMP_SMM_IP} "/smm/smmget -l blade${TMP_SLOT}:fru1 -d hotswapstate" > "${TMP_CONTENT_FILE}"
    STATUS=`grep "Blade${TMP_SLOT}:fru1 hot swap state is M[0-9]." "${TMP_CONTENT_FILE}" | awk '{print $NF}' | awk -F"." '{print $1}'`
    #WARNING: we should check the content of status while reboot the slot ...
    if [ -n "${STATUS}" -a "${STATUS}" != "M4" ]
    then
        PRINT_LOG "INFO" "SMM<${TMP_SMM_IP}> SLOT<${TMP_SLOT}> STATUS<${STATUS}> rear blade abnormal, restart it..."
        ssh -n "${TMP_SMM_IP}" "/smm/smmset -l blade${TMP_SLOT} -d powerstate -v poweroff"
        sleep 10
        ssh -n "${TMP_SMM_IP}" "/smm/smmset -l blade${TMP_SLOT} -d powerstate -v poweron"
        sleep 20
        return 1;
    fi
    PRINT_LOG "INFO" "SMM<${TMP_SMM_IP}> SLOT<${TMP_SLOT}> rear blade OK!"
    return 0;
}


#*************************************************************#
# Name:   fn_setHosts                                         #
# Description:  add hostname and ip addre to /etc/hosts       #
# Inout:  host list and ip list                               #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-12-08                                         #
#*************************************************************#
function fn_setHosts()
{
    local ipList=($1)
    local node

    cp ${HOSTSFILE} ${TMPDIR}/hosts
    #delete the host in the hosts file
    i=0
    for node in ${SLS_SYSTEMS}
    do
        cat ${TMPDIR}/hosts | sed "/ ${node}$/d" | sed "/    ${node}$/d" > ${TMPDIR}/hosts1
        echo "${ipList[${i}]} ${node}" >> ${TMPDIR}/hosts1
        i=$((i+1))
        mv ${TMPDIR}/hosts1 ${TMPDIR}/hosts
    done

    mv ${TMPDIR}/hosts ${HOSTSFILE}
    if [ $? -ne 0 ]
    then
        ERROR "setHosts fail";
        return ${ERR_EXECSYS};
    else
        return 0;
    fi
}

#*************************************************************#
# Name:   fn_FileCheck                                        #
# Description:                                                #
# Inout:                                                      #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-12-03                                         #
#*************************************************************#
function fn_FileCheck()
{
    File=$1
    LOG "Checking file ${File}."
    if [ -f ${File} ]
    then
        LOG "The file ${File} is exist."
        return 0;
    else
        ERROR "The file ${File} is not exist."
        return ${ERR_FILEEXIST};
    fi
}

#*************************************************************#
# Name:        fn_getTotalMem                                 #
# Description: get the total system physics memory            #
# Inout:                                                      #
# Ouput:   physics memory size MB                             #
# Author:                                                     #
# Date:    2009-11-10                                         #
#*************************************************************#
function fn_getTotalMem()
{
    local physicsmemory=`grep MemTotal /proc/meminfo | awk -F' ' '{print $2}'`
    local physicsM=$((${physicsmemory}/1024))

    echo ${physicsM}
}

#*************************************************************#
# Name:        getSwapMem                                     #
# Description: get the total system swap memory               #
# Inout:                                                      #
# Ouput:   swap memory size MB                                #
# Author:                                                     #
# Date:    2009-11-10                                         #
#*************************************************************#
function fn_getSwapMem()
{
    local swapmemory=`grep SwapTotal /proc/meminfo | awk -F' ' '{print $2}'`
    local swapM=$((${swapmemory}/1024))
    echo ${swapM}
}

#*************************************************************#
# Name:        getTmpDirFreeSize                              #
# Description: get the temp directory available size          #
# Inout:                                                      #
# Ouput:   temp directory available size MB                   #
# Author:                                                     #
# Date:    2009-11-10                                         #
#*************************************************************#
function fn_getTmpDirFreeSize()
{
    local tmpsize=`df -B 1M /tmp | sed -n '2p' | awk '{print $4}'`
    echo ${tmpsize}
}

#*************************************************************#
# Name:        SafeExec                                       #
# Description: execute the command and record log             #
# Inout:  error message                                       #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function fn_SafeExec()
{
    local cmd=$1
    local errno=$2
    locallog=/tmp/ataetmp_$$.log
    eval ${cmd} >${locallog} 2>&1
    if [ $? -ne 0 ]
    then
        ERROR "Exec ${cmd} FAILED."
        strTmp="`cat ${locallog}`"
        ERROR "${strTmp}"
        exit ${errno}
    fi
}

function fn_execRemoteSilentCmd()
{
    if [ $# -lt 3 ]
    then
        return ${ERROR_PARAM}
    fi
    local_user=$1
    remote_host=$2
    shift;shift
    command=`echo $@`
    fn_SafeExec "su - ${local_user} ${CMMD} \"ssh ${local_user}@${remote_host} ${command}\"" ${ERR_CMDEXEC}
}

function fn_copyRemoteToRemote()
{
    if [ $# -ne 5 ]
    then
        ERROR "Incorrect copy parameters."
        return ${ERROR_PARAM}
    fi
    local_user=$1
    source_host=$2
    source_fd=$3
    target_host=$4
    target_fd=$5
    su - ${local_user} ${CMMD} "scp -rp ${local_user}@${source_host}:${source_fd} ${local_user}@${target_host}:${target_fd}"
    if [ $? -ne 0 ]
    then
        ERROR "FUNC:fn_copyRemoteToRemote scp fail."
        return ${ERROR_SCP};
    else
        return 0;
    fi
}

function fn_copyLocalToRemote()
{
    if [ $# -ne 4 ]
    then
        ERROR "Incorrect copy parameters."
        return ${ERROR_PARAM}
    fi
    local_user=$1
    source_fd=$2
    target_host=$3
    target_fd=$4
	
	# if source_fd is dir and you want to copy the files in source_fd to remote_source_fd, the para input must as:/opt/osmu/, you cant use:/opt/osmu
    ssh ${local_user}@${target_host} "[ ! -d ${target_fd%/*} ] && mkdir -p ${target_fd%/*}"
	if [ -d ${source_fd} ]
	then
		source_fd_list=`ls ${source_fd}`
		for one_source_fd in ${source_fd_list}
		do
			su - ${local_user} ${CMMD} "scp -rp ${source_fd}/${one_source_fd} ${local_user}@${target_host}:${target_fd}"
			if [ $? -ne 0 ]
			then
				ERROR "FUNC:fn_copyLocalToRemote scp fail."
				return ${ERROR_SCP};
			fi
		done
		return 0;
	else
		su - ${local_user} ${CMMD} "scp -rp ${source_fd} ${local_user}@${target_host}:${target_fd}"
		if [ $? -ne 0 ]
		then
			ERROR "FUNC:fn_copyLocalToRemote scp fail."
			return ${ERROR_SCP};
		else
			return 0;
		fi
	fi
}

function fn_copyRemoteToLocal()
{
    if [ $# -ne 4 ]
    then
        ERROR "Incorrect copy parameters."
        return ${ERROR_PARAM}
    fi
    local_user=$1
    source_host=$2
    source_fd=$3
    target_fd=$4
    su - ${local_user} ${CMMD} "scp -rp ${local_user}@${source_host}:${source_fd} ${target_fd}"
    if [ $? -ne 0 ]
    then
        ERROR "FUNC:fn_copyRemoteToLocal scp fail."
        return ${ERROR_SCP};
    else
        return 0;
    fi
}

#*************************************************************#
# Name:   buildSSH                                            #
# Description:  establish the crediation between the nodes    #
#               the same user must be the same password       #
# Inout:   nodelist user remote_node_password SSHYype         #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2009-11-03                                         #
#*************************************************************#
function fn_buildSSH()
{
    c_nodes=$1
    c_user=$2
    c_passwd=$3
    c_sshtype=$4

    chmod 750 ${HOME_PATH}/common/enable_auto_ssh.exp

    if [ "${c_sshtype}" = "IP" ]
    then
        localIPlist=`ifconfig |grep "inet addr:"|awk -F: '{print $2}'|awk '{print $1}'`
        local_node=""
        for ip in ${localIPlist}
        do
            if [ "${local_node}" != "" ]
            then
                break
            else
                for node in ${c_nodes}
                do
                    if [ "${ip}" = "${node}" ]
                    then
                        local_node=${node}
                        break
                    fi
                done
            fi
        done
        if [ "${local_node}" = "" ]
        then
            PRINT_LOG "WARN" "This host is not a cluster member."
        fi
    else
        local_node=`hostname`
    fi

    for c_node1 in ${c_nodes}
    do
        cp -rpf ${HOME_PATH}/common/enable_auto_ssh.exp ${TMPDIR}/enable_auto_ssh.exp
        fn_SafeExec "su - ${c_user} ${CMMD} \"expect ${TMPDIR}/enable_auto_ssh.exp ${c_user}@${c_node1} ${c_passwd}\"" ${ERR_CMDEXEC}
        PRINT_LOG "INFO" "Build SSH ${local_node} --------> ${c_node1} success."
        if [ "${local_node}" != "${c_node1}" ];then
            fn_execRemoteSilentCmd ${c_user} ${c_node1} "mkdir -p ${TMPDIR}"
            fn_copyLocalToRemote ${c_user} "${TMPDIR}/enable_auto_ssh.exp" ${c_node1} ${TMPDIR}
            fn_execRemoteSilentCmd ${c_user} ${c_node1} "chmod 750 ${TMPDIR}/enable_auto_ssh.exp"
            PRINT_LOG "INFO" "SCP enable_auto_ssh.exp from ${local_node} to ${c_node1}:${TMPDIR} success."
        else
            continue
        fi

        for c_node2 in ${c_nodes}
        do
            fn_execRemoteSilentCmd ${c_user} ${c_node1} "expect ${TMPDIR}/enable_auto_ssh.exp ${c_user}@${c_node2} ${c_passwd}"
            PRINT_LOG "INFO" "Build SSH ${c_node1} --------> ${c_node2} success."
        done
    done
}

#*************************************************************#
# Name:   getNICByName                                        #
# Description:  get the NIC name by the NIC type              #
# Inout:   nodelist user remote_node_password                 #
# Ouput:   NIC_type                                           #
# Author:                                                     #
# Date:    2009-11-09                                         #
#*************************************************************#
function fn_getNICByName()
{
    NICType=$1
    HB_ETH=`./inc/eth_alias.sh | grep ${NICType} | awk '{print $1}'`
    HB_ETH=`echo ${HB_ETH}`
    if [ -z "${HB_ETH}" ]
    then
        PRINT_LOG "WARN" "Failed to get the name of eth for ${NICType}"
        exit 1
    fi

    echo "${HB_ETH}"
}

function fn_execRemoteCmd()
{
    if [ $# -lt 3 ]
    then
        ERROR "Incorrect command parameters."
        return ${ERROR_PARAM}
    fi
    local_user=$1
    remote_host=$2
    shift;shift
    command=`echo $@`
    su - ${local_user} ${CMMD} "ssh ${local_user}@${remote_host} ${command}"
    if [ $? -ne 0 ]
    then
        ERROR "Exec<su - ${local_user} ${CMMD} \"ssh ${local_user}@${remote_host} ${command}\" Failed."
        return ${ERR_SSH};
    else
        return 0;
    fi
}

#*************************************************************#
# Name:   fn_getEthMac                                        #
# Description:  get the NIC mac of the eth  name              #
# Inout:   nodelist user eth_name                             #
# Ouput:   ETH_MAC                                            #
# Author:                                                     #
# Date:    2009-11-09                                         #
#*************************************************************#
function fn_getEthMac()
{
    if [ $# -ne 3 ]
    then
        return 1;
    fi

    local local_user=$1
    local cfs_system=$2
    local target_eth=$3

    ETH_MAC=`fn_execRemoteCmd ${local_user} ${cfs_system} ifconfig ${target_eth}  | grep HWaddr | awk '{print $NF}'`
}

#*************************************************************#
# Name:   validateIP                                          #
# Description:  check whether the IP address is valid         #
# Inout:   IP_address                                         #
# Ouput:   255:invalid IP 1:validate                          #
# Author:                                                     #
# Date:    2009-11-09                                         #
#*************************************************************#
function fn_isValidIP()
{
    if [ $# -ne 1 ]
    then
        PRINT_LOG "INFO" "FUNC:fn_isValidIP Parameter is less than requirement"
        return ${ERROR_PARAM}
    fi

    local ip_addr=$1
    echo $ip_addr | awk '$0!~/0.0.0.0/ && $0!~/255.255.255.255/ \
        && $0~/^([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])\.([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])\.([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])\.([01]?[0-9][0-9]?|2[0-4][0-9]|25[0-5])$/ \
        {print "OK"}' | grep OK | grep -v grep >/dev/null 2>&1
    if [ $? -ne 0 ]
    then
        return ${ERR_IPREACH}
    fi

    return 0;
}

function fn_getProperties()
{
    if [ $# -ne 2 ]
    then
        ERROR "Incorrect propertie parameter numbers."
        return ${ERROR_PARAM}
    fi
    parameter=$1
    propertiers_file=$2
    para_value=`grep  "^${parameter}="  ${propertiers_file}|sort -n|awk -F'=' '{print $2}'`
    paralist_value=`egrep  "^${parameter}_[0-9]+="  ${propertiers_file}|sort -n|awk -F'=' '{print $2}'`
    if [ "$para_value" != "" ]
    then
        ARRAY=(${para_value})
    elif [ "$paralist_value" != "" ]
    then
        ARRAY=(${paralist_value})
    else
        ERROR "Properties(${parameter}) value is null."
        fn_check_status ${ERROR_PARAM}
    fi
}

function fn_validCount()
{
    para_count=$1
    array_size=$2

    if [ ${para_count} -ne ${array_size} ]
    then
        PRINT_LOG "WARN" "PARA_COUNT(${para_count}) is not equal to ARRAY_SIZE(${array_size})"
        exit 1
    fi
}

function fn_getLun()
{
    local lunid=$1
    local lunName
    lunName=`lsscsi | grep updisk |awk -F: '{print $NF}'|sed 's/]//g'|awk '{if(LUNID==$1) print $NF}' LUNID=${lunid}|cut -d"/" -f3`
    if [ "${lunName}" = "" ]
    then
        PRINT_LOG "WARN" "Can not get the lun name by lunid:${lunid}."
        exit ${ERR_GETLUN}
    fi
    echo ${lunName}
}

function fn_setupVxDisk()
{
    ${VXDISK} scandisks >/dev/null 2>&1
    for disk in $@
    do
        disk=`basename ${disk}`
        ${VXDISKUNSETUP} -C ${disk} >/dev/null 2>&1
        ${VXDISKSETUP} -if ${disk} format=cdsdisk >/dev/null 2>&1
        PRINT_LOG "WARN" "Failed to set up vx disk ${disk}."
        exit ${ERR_INITDISK};
    done
}

function fn_createDiskGroup()
{
    local disk_group=$1
    shift
    local vxdisk_list=$@
    disk_str=`echo ${vxdisk_list} | awk '{for(i=1;i<NF+1;i++) printf " %s%02s=%s",dg,i,$i}END{print""}' dg=${disk_group}`
    ${VXDG} destroy ${disk_group} >/dev/null 2>&1
    ${VXDG} init ${disk_group} ${disk_str}
    PRINT_LOG "WARN" "Failed to set up vx disk ${disk}."
    exit ${ERR_INITDG};
}

function fn_mountNFSDir()
{
    local nfsserver=$1
    local nfssvrdir=$2
    local nfsclidir=$3
    mkdir -p ${nfsclidir}
    nfsdir="`showmount -e ${nfsserver} |grep -v ${nfsserver}|awk '{print $1}'`"
    for dir in ${nfsdir}
    do
        if [ "${dir}" = "${nfssvrdir}" ]
        then
            mount -t nfs ${nfsserver}:${dir} ${nfsclidir} >/dev/null 2>&1
            break;
        fi
    done
    if [ "`df -h|grep ${nfsclidir}`" = "" ]
    then
        PRINT_LOG "WARN" "Mount ${nfsclidir} from ${nfsserver}:${nfssvrdir}............Failed."
        exit ${ERR_MOUNTNFS}
    else
        PRINT_LOG "INFO" "Mount ${nfsclidir} from ${nfsserver}:${nfssvrdir}............Success."
    fi
}

function fn_umountNFSDir()
{
    local nfsclidir=$1
    umount ${nfsclidir} >/dev/null 2>&1
    if  [ "`df -h|grep ${nfsclidir}`" != "" ]
    then
        PRINT_LOG "WARN" "Umount ${nfsclidir} ............Failed."
        exit ${ERR_MOUNTNFS}
    else
        PRINT_LOG "INFO" "Umount ${nfsclidir} ............Success."
    fi
}

function fn_checkResultFile()
{
    local RESULT_FILE=$1
    local RESULT_PATH

    if [ ! -f "${RESULT_FILE}" ]
    then
        RESULT_PATH=`dirname "${RESULT_FILE}"`
        if [ ! -d "${RESULT_PATH}" ]
        then
            mkdir -p "${RESULT_PATH}"
        fi
    fi

    rm -f "${RESULT_FILE}"
    touch "${RESULT_FILE}"
    if [ $? -ne 0 ]
    then
        ERROR "FUNC:fn_checkResultFile execute touch command fail."
        return ${ERR_EXECSYS}
    fi

    LOG "Success to create ${RESULT_FILE} file"
    return 0;
}

function fn_writeResultFile()
{
    local RESULT_FILE=$1
    local RESULT_NAME=$2
    local RESULT_VALUE=$3

    if [ $# -ne 3 ]
    then
        ERROR "[fn_writeResultFile]Parameter number is correct."
    fi

    echo -e  "${RESULT_NAME}=${RESULT_VALUE}" | tee -a ${RESULT_FILE}
    if [ $? -ne 0 ]
    then
        ERROR "Failed to write \"${RESULT_NAME}=${RESULT_VALUE}\" to ${RESULT_FILE}."
        return ${ERR_EXECSYS}
    else
        LOG "Success to write \"${RESULT_NAME}=${RESULT_VALUE}\" to ${RESULT_FILE}."
        return 0;
    fi
}


function fn_delete_upgradePatches()
{
    local UPGRADE_DIR=$1
    local UPGRADE_PARA=$2
    local PATCH_PKG_Name="tempvalue"

    LOG "Enter fn_delete_upgradePatches"
    local count=0
    while [ "${PATCH_PKG_Name}" != "" ]
    do
        PATCH_PKG_Name=`grep  "^PATCH_PKG_${count}="  ${UPGRADE_PARA}|sort -n|awk -F'=' '{print $2}'`
        PATCH_FUN_Name=`grep  "^PATCH_FUN_${count}="  ${UPGRADE_PARA}|sort -n|awk -F'=' '{print $2}'`
        if [ "${PATCH_FUN_Name}" != "" ] && [ "${PATCH_PKG_Name}" != ""  ]
        then
            LOG "upgrade patche ${UPGRADE_DIR}/${PATCH_PKG_Name} "
            [ -f ${UPGRADE_DIR}/${PATCH_PKG_Name} ] && rm -rf ${UPGRADE_DIR}/${PATCH_PKG_Name} > /dev/null 2>&1
            if [ $? -eq 0 ]
            then
                LOG "Success to delete the package ${UPGRADE_DIR}/${PATCH_PKG_Name}."
            else
                LOG "Failed to delete the package ${UPGRADE_DIR}/${PATCH_PKG_Name}."
            fi
            let count++
        fi
    done

    LOG "Exit fn_delete_upgradePatches"
    return 0;
}

#*************************************************************#
# Name:   fn_upgradePatches                                   #
# Description: upgrade  patches                               #
# Inout:                                                      #
# Ouput:                                                      #
# Author:                                                     #
# Date:    2010-05-26                                         #
#*************************************************************#
function fn_upgradePatches()
{
    local UPGRADE_DIR=$1
    local UPGRADE_PARA=$2
    local PATCH_PKG_Name="tempvalue"

    if ! [ -d ${UPGRADE_DIR} ]
    then
        ERROR "The upgrade patches directory is not exist!"
        return ${ERR_INST}
    fi

    if ! [ -f ${UPGRADE_PARA} ]
    then
        ERROR "The upgrade configuration template is not exist!"
        return ${ERR_INST}
    fi

    LOG "Start to upgrade Patches ... "

    local count=0
    while [ "x${PATCH_PKG_Name}" != "x" ]
    do
        PATCH_PKG_Name=`grep  "^PATCH_PKG_${count}="  ${UPGRADE_PARA}|sort -n|awk -F'=' '{print $2}'`
        PATCH_FUN_Name=`grep  "^PATCH_FUN_${count}="  ${UPGRADE_PARA}|sort -n|awk -F'=' '{print $2}'`
        if [ "x${PATCH_FUN_Name}" != "x" ] && [ "x${PATCH_PKG_Name}" != "x"  ]
        then
            ${PATCH_FUN_Name} "${UPGRADE_DIR}/${PATCH_PKG_Name}"
            if [ $? = 0 ]
            then
                let count++
            else
                ERROR "Exec <${PATCH_FUN_Name} ${UPGRADE_DIR}/${PATCH_PKG_Name}> failed,Upgrade patch failed."
                return ${ERR_INST}
            fi
        fi
    done

    LOG "Upgrade Patches successful."
}

function fn_executeSQL()
{
    local oraUser=$1
    local outfile=$2
    local sqlstat=$3

    local outfilename=sqlout$$.out

    if [ "$outfilename" = "" ]
    then
        return ${ERROR_PARAM}
    fi
    if [ "$sqlstat" = "" ]
    then
        return ${ERROR_PARAM}
    fi

    dbsql="/tmp/dbsql.sql"
    [ -f ${dbsql} ] && rm  ${dbsql}
    [ -f /tmp/$outfilename ] && rm /tmp/$outfilename
    cat << EOF > ${dbsql}
set pages 9999;
set long 90000;
set line 4000;
set linesize 30000;
set termout off;
set trimspool on
spool /tmp/$outfilename
$sqlstat
spool off
exit
EOF
    chmod 644 ${dbsql}
    su - ${oraUser} ${CMMD} "sqlplus -S / as SYSDBA  < /dev/null" > /tmp/test.txt
    error=`grep "ORA-12547: TNS:lost contact" /tmp/test.txt`
    if [ "x${error}" = "x" ]
    then
        su - ${oraUser} ${CMMD} "sqlplus -S / as SYSDBA @${dbsql}" 2>/dev/null
    else
        [ -f ${dbsql} ] && rm -f ${dbsql}
		return ${ERR_SERVICE}
    fi

    if [ -f /tmp/$outfilename ]
    then
        mv /tmp/$outfilename ${outfile}
    fi

    [ -f /tmp/test.txt ] && rm -f /tmp/test.txt
    [ -f ${dbsql} ] && rm  ${dbsql}
}

# Name        : fn_reboot_slot
# Description : invoke the hook script to reboot the slot
# Parameters  : $1 the slot number
function fn_reboot_slot()
{
    local SLOT=$1

    #Reboot the specifical slot ..
    PRINT_LOG "INFO" "Reboot the slot<${SLOT}> .."

    cd "${DLDEP_HOOK_PATH}"
    bash reboot_mac_on_slot ${SLOT}
    if [ $? != 0 ]
    then
        PRINT_LOG "WARN" "Reboot <${SLOT}> failure.."
        return ${ERR_REBOOT};
    fi

    PRINT_LOG "INFO" "Reboot the slot<${SLOT}> .. OK!!!"
    return 0;
}

# Name        : fn_get_mac
# Description : invoke a hook script to get mac address
# Parameters  : $1 slot number
#             : $2 get mac address of slot
function fn_get_mac()
{
    local SLOT=$1
    local MAC_ADDR_LIST

    local TMP_CONTENT

    #Getting mac of the specifical slot ..
    PRINT_LOG "INFO" "Get mac of the specifical slot<${SLOT}>.."

    cd "${DLDEP_HOOK_PATH}"
    TMP_CONTENT=`bash get_mac_of_base_from_slot "${SLOT}"`
    if [ -z "${TMP_CONTENT}" ]
    then
        PRINT_LOG "WARN" "Get mac addr of slot${SLOT} ..failure!!!"
        return ${ERR_GETMAC};
    else
        PRINT_LOG "INFO" "The mac addr is <${TMP_CONTENT}>"
    fi

    MAC_ADDR_LIST=`echo "${TMP_CONTENT}" | sed 's/:/-/g' | tr '[:upper:]' '[:lower:]'`

    IFS_TMP=$IFS
    IFS=' '
    MAC_ADDR_LIST=($MAC_ADDR_LIST)
    IFS=$IFS_TMP

    eval $2="(${MAC_ADDR_LIST[@]})"

    PRINT_LOG "INFO" "Get mac of the specifical slot<${SLOT}>.. OK!!!"
    return 0;
}

# Name        : fn_get_route_config
# Description : get the config from osmu
# Parameters  : $1 the absoluately path of config file
#             : $2 the index of host
#             : $3 get slot number
#             : $4 get smm card ip
function fn_get_route_config()
{
    #get config..
    local TMP_CONFIG_FILE=$1
    local INDEX_HOST_COUNT=$2

    local ACTION_TYPE
    local DESTINATION_IP
    local GATEWAY_IP
    local NETMASK

    local TMP_IP_ADDR
    local TMP_IP_ADDR_LIST

    PRINT_LOG "INFO" "Loading <${TMP_CONFIG_FILE}>.."

    ACTION_TYPE=`fn_get_value "${TMP_CONFIG_FILE}" "ACTION_TYPE_${INDEX_HOST_COUNT}"`
    DESTINATION_IP=`fn_get_value "${TMP_CONFIG_FILE}" "DESTINATION_IP_${INDEX_HOST_COUNT}"`
    GATEWAY_IP=`fn_get_value "${TMP_CONFIG_FILE}" "GATEWAY_IP_${INDEX_HOST_COUNT}"`
    NETMASK=`fn_get_value "${TMP_CONFIG_FILE}" "NETMASK_${INDEX_HOST_COUNT}"`

    if [ "${ACTION_TYPE}" != "add" ] && [ "${ACTION_TYPE}" != "delete" ] && [ "${ACTION_TYPE}" != "rollback" ]
    then
        PRINT_LOG "WARN" "The param ACTION_TYPE<${ACTION_TYPE}> should be <add|delete|rollback>!"
        return ${ERROR_PARAM};
    fi

    if [ -z "${DESTINATION_IP}" ] || [ -z "${GATEWAY_IP}" ]
    then
        PRINT_LOG "WARN" "The param DESTINATION_IP<${DESTINATION_IP}> or GATEWAY_IP<${GATEWAY_IP}> could not be found!"
        return ${ERROR_PARAM};
    fi

    TMP_IP_ADDR_LIST="${GATEWAY_IP}"
    if [ "${DESTINATION_IP}" != "default" ]
    then
        if [ -z "${NETMASK}" ]
        then
            PRINT_LOG "WARN" "The param NETMASK<${NETMASK}> could not be found!"
            return ${ERROR_PARAM};
        fi
        TMP_IP_ADDR_LIST="${TMP_IP_ADDR_LIST} ${DESTINATION_IP} ${NETMASK}"
    else
        PRINT_LOG "INFO" "The DESTINATION_IP<${DESTINATION_IP}>, it is a default route!"
    fi

    for TMP_IP_ADDR in ${TMP_IP_ADDR_LIST}
    do
        fn_validate_ip "${TMP_IP_ADDR}"
        fn_check_status $?
    done

    eval $3="${ACTION_TYPE}"
    eval $4="${DESTINATION_IP}"
    eval $5="${GATEWAY_IP}"
    eval $6="${NETMASK}"

    PRINT_LOG "INFO" "Loading <${TMP_CONFIG_FILE}>.. OK!!!"
    return 0;
}

fn_modify_route_file()
{
    local ACTION_TYPE=$1
    local DESTINATION_IP=$2
    local GATEWAY_IP=$3
    local NETMASK=$4
    local BOND_NIC=$5
    local TMP_CONTENT_FLAG

    PRINT_LOG "INFO" "Setting route files.."

    if [ "${DESTINATION_IP}" = "default" ] || [ "${DESTINATION_IP}" = "0.0.0.0" ]
    then
        grep -v "default" /etc/sysconfig/network/routes >/etc/sysconfig/network/routes.new 2>/dev/null
        mv /etc/sysconfig/network/routes.new /etc/sysconfig/network/routes 2>/dev/null
        if [ "${ACTION_TYPE}" = "add" ]
        then
                echo "default ${GATEWAY_IP} - ${BOND_NIC}" >>/etc/sysconfig/network/routes 2>/dev/null
        fi
    else
        grep -v "${DESTINATION_IP} ${GATEWAY_IP} ${NETMASK} ${BOND_NIC}" /etc/sysconfig/network/routes >/etc/sysconfig/network/routes.new 2>/dev/null
        mv /etc/sysconfig/network/routes.new /etc/sysconfig/network/routes 2>/dev/null
        if [ "${ACTION_TYPE}" = "add" ]
        then
                echo "${DESTINATION_IP} ${GATEWAY_IP} ${NETMASK} ${BOND_NIC}" >>/etc/sysconfig/network/routes 2>/dev/null
        fi
    fi
    PRINT_LOG "INFO" "Setting route files.. OK!!!"
}

function doCmd()
{
    eval $* > /dev/null
    if [ $? -eq 0 ]
    then
        LOG "Exec<$*> OK."
    else
        ERROR "Exec<$*> Fail."
    fi
}

function fn_isRootUser()
{
    local CURR_USER
    CURR_USER=`id | awk -F'(' '{print $2}' | awk -F')' '{print $1}'`
    if [ "$CURR_USER" = "root" ]
    then
        return 0;
    else
        return 1;
    fi
}

function fn_execSql4Postgres()
{
    local DBNAME=$1
    local STRING_SQL=$2
    local RESULT_FILE=$3

    local DBUSER="postgres"

    if [ "$#" -lt 2 ]
    then
        ERROR "Parameters error. "
        LOG "usage: $0 <postgres database name> <sql command> [<result file>]"
        LOG "==========${FILENAME} run end=========="
        exit ${ERROR_PARAM}
    fi

    fn_isRootUser
    if [ $? -ne 0 ]
    then
        ERROR "The user must be root."
        return 1;
    fi

    if [ -z "${RESULT_FILE}" ]
    then
        su - ${DBUSER} ${CMMD} "psql -d ${DBNAME} -t -c \"${STRING_SQL};\""
    else
        su - ${DBUSER} ${CMMD} "psql -d ${DBNAME} -t -c \"${STRING_SQL};\" " > "${RESULT_FILE}"
    fi

    if [ $? -eq 0 ]
    then
        LOG "Exec<su - ${DBUSER} ${CMMD} \"psql -d ${DBNAME} -t -c \"${STRING_SQL};\"\"> OK." >/dev/null 2>&1
        return 0;
    else
        LOG "Exec<su - ${DBUSER} ${CMMD} \"psql -d ${DBNAME} -t -c \"${STRING_SQL};\"\"> Fail."
        return 1;
    fi
}


function fn_LoginSMM_qi()
{
    local server=$1
    local user=$2
    local passwd=$3
    local tempfile="${ABSOLUATELY_PATH}/autossh_${RANDOM}.sh"
    local result
    shift
    shift
    shift
    local TMP_CONTENT_FILE=$1
    shift
    cmd=$@

    fn_Modify_known_hosts_file "${server}"

    cat << EOF > "${tempfile}"
#!/usr/bin/expect
set username "${user}"
set passwd "${passwd}"
set server "${server}"
set cmd "$@"
set EXT "exit"
spawn ssh \${username}@\${server} "[lindex \$argv 0]" ;
expect {
    "*(yes/no)?*"
    {
        send "yes\n"
        exp_continue
    }
    "*assword:*"
    {
        sleep 1
        send  "\${passwd}\r"
    }
}
sleep 2;
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*#"
    {
        send "\n"
    }
    "*>"
    {
        send "\n"
    }
}
expect {
    "*assword:"
    {
        exit 1;
    }
    "*#"
    {
        send "\${cmd}\n"
    }
    "*>"
    {
        send  "\${cmd}\n"
    }
}
expect {
    "*#"
    {
        send "\${EXT}\n"
    }
    "*Y/N]:"
    {
        send  "Y\n"
    }
}
expect {
    "*#"
    {
        send "\${EXT}\n"
    }
    "*>"
    {
        send "\${EXT}\n"
    }
}
EOF
    chmod 755 ${tempfile}
    echo "0" > ${tempfile}.result
    (${tempfile}||(echo $? > ${tempfile}.result)) | tee "${TMP_CONTENT_FILE}"
    result=`cat ${tempfile}.result`
    if [ "${result}" != 0 ]
    then
        PRINT_LOG "WARN" "Remote login ${server} failed."
    fi
    rm "${tempfile}".result
    rm "${tempfile}"
    return ${result}
}

#---  FUNCTION  --------------------------------------------------------------
#          NAME:  fn_get_ip_by_ifname
#   DESCRIPTION:
#    PARAMETERS:  $1 para name to assignment ip to
#                 $2 ifname
#       RETURNS:  0 success, 1 or other err_code failure
#-----------------------------------------------------------------------------
function fn_get_ip_by_ifname()
{
    local bout
    local ifname=$2

    if [ "" == "$ifname" ]; then
        PRINT_LOG "FATAL" "ifname can not be empty"
        return 1
    fi

    ifconfig $ifname 2> /dev/null | grep "inet addr" > /dev/null 2>&1
    if [ "${?}" -eq 0 ]
    then
        bout=`ifconfig $ifname | grep "inet addr" | cut -d : -f 2 |  cut -d ' ' -f 1`
    else
        PRINT_LOG "FATAL" "Get $ifname IP failure"
        return 1
    fi

    eval $1="\$bout"
    return 0
}    # --------  end of function _fn_get_OSMU_IP_  --------

#---  FUNCTION  --------------------------------------------------------------
#          NAME:  fn_set_LANG
#   DESCRIPTION:  (1) U must specify the $USER_HOME, or it will fail,
#                 for multi-user, so put into a circle
#                 (2) LANG must be get from the parafile, or will use default
#
#    PARAMETERS:  $1 $USER_HOME
#       RETURNS:  0 success, 1 or other err_code failure
#-----------------------------------------------------------------------------
function fn_set_LANG()
{
    local USER_HOME=$1
    if [ ! -e "$USER_HOME" ]; then
        PRINT_LOG "FATAL" "USER_HOME must exist"
        return ${ERROR_PATH_NOT_EXIST}
    fi

    local DOT_PROFILE=${USER_HOME}/.profile
    touch $DOT_PROFILE
    cp $DOT_PROFILE $DOT_PROFILE.bak

    # LANG
    if  [ "$LANG" != "" ] &&
        [ "$LC_CTYPE" != "" ] &&
        [ "$LC_MESSAGES" != "" ] &&
        [ "$LANG" != "NULL" ] &&
        [ "$LC_CTYPE" != "NULL" ] &&
        [ "$LC_MESSAGES" != "NULL" ]
    then

        sed -i -e "/LC_ALL/d"  \
            -e "/LANG/d"  -e "/LC_CTYPE/d" \
            -e "/LC_MESSAGES/d" $DOT_PROFILE.bak

        if [ 0 -ne $? ]; then
            PRINT_LOG "FATAL" "papameters LANG sed error"
            return $ERR_EXECSYS
        fi

        cat << EOF >> $DOT_PROFILE.bak
unset LC_ALL
export LANG="$LANG"
export LC_CTYPE="$LC_CTYPE"
export LC_MESSAGES="$LC_MESSAGES"
EOF
        if [ 0 -ne $? ]; then
            PRINT_LOG "FATAL" "papameters LANG set error"
            return $ERR_EXECSYS
        fi
    else
        PRINT_LOG "FATAL" "papameters of LANG error"
    fi


    cp $DOT_PROFILE.bak $DOT_PROFILE
    rm -f $DOT_PROFILE.bak

    return ${SUCCESS_CODE}
}

#---  FUNCTION  --------------------------------------------------------------
#          NAME:  fn_ooh
#   DESCRIPTION:  Usage: fn_ooh $? `$PS8` "Ur wanted err message"
#    PARAMETERS:  $1 function retval b4 fn_ooh
#                 $2 position, always <$FUNCNAME:$LINENO>
#                 $3 Error message
#       RETURNS:  0 success, 1 or other err_code failure
#-----------------------------------------------------------------------------
PS8="eval echo \$BASH_SOURCE\|\$LINENO\|\$FUNCNAME:"
ECHOFUNC="PRINT_LOG FATAL"

function fn_ooh()
{
    if [ "${1}" -ne 0 ]; then
        ${ECHOFUNC} "${2} [$1] ${*:3}"
    fi

    return ${1}
}


function date2days {
    echo "$*" | awk '{
        z=int((14-$2)/12); y=$1+4800-z; m=$2+12*z-3;
        j=int((153*m+2)/5)+$3+y*365+int(y/4)-int(y/100)+int(y/400)-2472633;
        print j
    }'
}


function days2date {
    echo "$1" | awk '{
        a=$1+2472632; b=int((4*a+3)/146097); c=int((-b*146097)/4)+a;
        d=int((4*c+3)/1461); e=int((-1461*d)/4)+c; m=int((5*e+2)/153);
        dd=-int((153*m+2)/5)+e+1; mm=int(-m/10)*12+m+3; yy=b*100+d-4800+int(m/10);
        printf ("%4d-%02d-%02d\n",yy,mm,dd)
    }'
}

#*************************************************************#
# Name:   get_maintainIP                                      #
# Description:  Get maintain IP address                       #
# Author:                                                     #
# Date:    2012-08-13                                         #
#*************************************************************#
function get_maintainIP()
{
    local MaintainIP
    local BONDIP=`ifconfig |grep bond0`
    local Base1_ETH=`eth_alias.sh |grep Base1 |awk '{print $1}'`
    local Base2_ETH=`eth_alias.sh |grep Base2 |awk '{print $1}'`
    local IP1=`ifconfig "${Base1_ETH}" |grep "inet addr" |awk -F '[ :]+' '{print $4}'`
    local IP2=`ifconfig "${Base2_ETH}" |grep "inet addr" |awk -F '[ :]+' '{print $4}'`

    if [ -n "${BONDIP}" ]
    then
        MaintainIP=`ifconfig bond0 |grep "inet addr" |awk -F '[ :]+' '{print $4}'`
    elif [ -n "${IP1}" ]
    then
        MaintainIP="${IP1}"
    else
        MaintainIP="${IP2}"
    fi

    eval $1=$MaintainIP
}



# Name        : fn_create_session_Key
# Description : create SSH Key for session use only
function fn_create_session_Key()
{
    local MAX_LOG_COUNT=12

    PRINT_LOG "INFO" "Create session key.. "
    mkdir -p "${TMP_BUFFER_PATH}/.ssh"

    ssh-keygen -t rsa -f "${TMP_BUFFER_PATH}/.ssh/id_rsa" -N "" > "${TMP_CONTENT_FILE}" 2>&1
    if [ $? != 0 ]
    then
        PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
        PRINT_LOG "WARN" "Exec <ssh-keygen -t rsa -f \"${TMP_BUFFER_PATH}/.ssh\" -N \"\"> ...failure!!!"
        return ${ERR_EXECSYS}
    fi
    PRINT_LOG "INFO" "Exec <ssh-keygen -t rsa -f \"${TMP_BUFFER_PATH}/.ssh\" -N \"\"> ...OK!!!"

    #set the pub key to itself.. for we will use it to set the rootkey later .
    cat "${TMP_BUFFER_PATH}/.ssh/id_rsa.pub" > "${TMP_BUFFER_PATH}/.ssh/authorized_keys"

    chmod 700 "${TMP_BUFFER_PATH}/.ssh"     > /dev/null 2>&1
    chmod 600 "${TMP_BUFFER_PATH}"/.ssh/*        > /dev/null 2>&1

    sed "s/root@.*/root@${RANDOM_INFO}/g" "${TMP_BUFFER_PATH}/.ssh/id_rsa.pub" >> "/root/.ssh/authorized_keys"

    #make sure the session key would not leave in the pxelinux.cfg forever ..
    ls -td "${DLDEP_PXECFG_PATH}"/* | grep "sshkey.*" | tail -n+${MAX_LOG_COUNT} | xargs -I '{}' -s 100000 rm -r '{}' > /dev/null 2>&1

    cd "${TMP_BUFFER_PATH}"
    tar cf sshkey${RANDOM_INFO}.tar .ssh
    cp -f sshkey${RANDOM_INFO}.tar "${DLDEP_PXECFG_PATH}/"
    if [ $? != 0 ]
    then
        PRINT_LOG "WARN" "Exec <cp -f sshkey${RANDOM_INFO}.tar \"${DLDEP_PXECFG_PATH}\"> ...failure!!!"
        return ${ERR_EXECSYS};
    fi
    PRINT_LOG "INFO" "Exec <cp -f sshkey${RANDOM_INFO}.tar \"${DLDEP_PXECFG_PATH}/\"> ...OK!!!"

    eval $1="sshkey${RANDOM_INFO}.tar"

    PRINT_LOG "INFO" "Create session key.. OK!!!"
    return 0
}



# Name        : fn_create_session_Key
# Description : create SSH Key for session use only
function fn_clean_session_key()
{
    local SSH_KEY=$1

    PRINT_LOG "INFO" "Clean session key.. "
    #clean the session key
    rm -f "${DLDEP_PXECFG_PATH}/${SSH_KEY}"
    sed -i -e "s#.*root@${RANDOM_INFO}##g" -e "/^$/d" /root/.ssh/authorized_keys

    PRINT_LOG "INFO" "Clean session key.. OK!!!"
}

function fn_clear_opasswd()
{
    local username=$1
	local opasswdfile=/etc/security/opasswd
	
	PRINT_LOG "INFO" "Clean ${username} from opasswd file"
	
	if [ -f ${opasswdfile} ]
	then
	    sed -i "/^${username}:/d" ${opasswdfile}
	fi
}


#*************************************************************#
# Name:   fn_write_operation_log                              #
# Description:  Write a message to osmumessage                #
# Author:                                                     #
# Date:    2013-11-06                                         #
#*************************************************************#
function fn_write_operation_log()
{
    local operation_ret="$1"
    local operation_name="$2"
	local operation_info="$3"
	local operation_obj="$4"
	
    if [ -z "${operation_ret}" -o -z "${operation_name}" ]
    then
        PRINT_LOG "FATAL" "operation_ret<${operation_ret}>,operation_name<${operation_name}> is null" | sed -r 's/(\ *\[.*\]){2}\ *//g'
        return ${ERROR_PARAM}
    fi
	
    local operation_code=`grep -m1 "^${operation_name}=" "${OPERATION_CODE_FILE}" | awk -F'=' '{print $2}' | awk '{print $1}'`
    if [ -z "${operation_code}" ]
    then
        grep -v "^[[:space:]]*#" "${OPERATION_CODE_FILE}" | grep -m1 "=${operation_name}[[:space:]]*" > /dev/null 2>&1
        if [ $? -ne 0 ]
        then
            PRINT_LOG "FATAL" "operation code <${operation_name}> does not exist in operation_code.inc" > /dev/null 2>&1
        fi
        operation_code="${operation_name}"
    fi

    # Avoid screen who am i to generate ":S.0"
    local operation_ip=`who am i | awk '{print $NF}' | sed -e 's/[()]//g' -e 's/:.*//g'`
    operation_ip=${operation_ip:-"127.0.0.1"}

    local strTime=`fn_getSysTime`
    local operation_user=`whoami`

    [ -d "${MGR_ROOT}"/tmp ] || mkdir -p "${MGR_ROOT}"/tmp
    echo "TIME=${strTime},OPERATION_USER=${operation_user},OPERATION_IP=${operation_ip},OPERATION_INFO=${operation_info},OPERATION_OBJ=${operation_obj},OPERATION_CODE=${operation_code},OPERATION_RESULT=${operation_ret}" >> "${LOGIN_MESSAGE}" 2>/dev/null

    return 0
}


#*********************************************************************#
# Name:        fn_wait_for_ssh                                        #
# Description: wait while can log in to the server in two minutes     #
# Parameters:  $1  --  the server ip address                          #
#              $2  --  Sleep times(Unit: Seconds)                     #
#              $3  --  Sleep Counts                                   #
#*********************************************************************#
function fn_wait_for_ssh()
{
    local server_ip=$1
    local sleep_time=$2
    local sleep_counts=$3
    local cur_count
    local SSH="ssh -o ConnectTimeout=1 -o StrictHostKeyChecking=no -o BatchMode=yes"
    
    PRINT_LOG "INFO" "Try to log in to the server<${server_ip}>..."
    
    if [ $# -lt 3 ]
    then
        PRINT_LOG "FATAL" "Parameters error: $0 $*"
        return ${ERROR_PARAM}
    fi
    
    fn_validate_ip "${server_ip}" || return $?    
    
    echo ${sleep_time} | egrep "^[0-9]+$" &> /dev/null
    if [ $? -ne 0 ] || [ "${sleep_time}" -le "0" ]
    then
        PRINT_LOG "FATAL" "The second parameter must be greater than 0"
        return ${ERROR_PARAM}
    fi
    
    echo ${sleep_counts} | egrep "^[0-9]+$" &> /dev/null
    if [ $? -ne 0 ] || [ ${sleep_counts} -le 0 ]
    then
        PRINT_LOG "FATAL" "The third parameter must be greater than 0"
        return ${ERROR_PARAM}
    fi

    for ((cur_count=0; cur_count<${sleep_counts}; cur_count++))
    do
        ${SSH} "${server_ip}" "exit 0" &> "${TMP_CONTENT_FILE}"
        if [ $? -eq 0 ]
        then
            PRINT_LOG "INFO" "Succeeded in logging in to the server<${server_ip}>."
            return ${SUCCESS_CODE}
        fi
        sleep ${sleep_time}
    done
    
    PRINT_FILE_TO_LOG "${TMP_CONTENT_FILE}"
    PRINT_LOG "FATAL" "Failed to log in to the server<${server_ip}>"
    return ${ERR_SSH}
}

function fn_syslog()
{
    local msg_info=$1
    local log_level=$2
    local username=`id -nu`
    
    if [ "$log_level" = "info" ]
    then
        log_level="local0.info"
    else
        log_level="local0.err"
    fi
    
    logger -t "${username}" -p "${log_level}" "${msg_info}"
    
    return 0
}
