# coding: utf-8
"""
生成前台网卡表格
"""
import copy
import json
import os

from easysuite import settings

from plugins.ROC.py_scripts.common.utils.add_increase_node_depends import AddIncreaseNodeDepends
from plugins.ROC.py_scripts.common.utils.config_controls import ConfigControls
from plugins.ROC.py_scripts.common.utils.loadjson.load_node_nics import LoadNodeNics
from plugins.ROC.py_scripts.common.utils.set_nic_index_usage import SetNicIndexUsage

from plugins.ROC.py_scripts.common.utils.parse_custom_json import ParseCustomJson
from .get_conditions_nics import GetConditionsNics
from .nics_business import NicsBusiness

USAGE_TO_NIC = {
    "inter_node_network": "eth0",
    "NCE_DataSync": "eth1",
    "southbound_network": "eth2, eth5",
    "northbound_network": "eth3",
    "HDmgmt_network": "eth4",
    "HAdisaster_network": "eth6"
}


class FulFillNodesNicsTable():
    """
    生成前台网卡表格
    """

    def __init__(self, kvs, ctrl_set, src_dir_path, custom_conditions):
        self.src_dir_path = src_dir_path
        self.nodes_nics_list = None
        self.ctrl_set = ctrl_set
        self.kvs = kvs
        self.old_controls = self.ctrl_set.get("controls", [])
        self.custom_conditions = custom_conditions
        self.expansion_param_dict = ParseCustomJson.get_lld_param_dict(src_dir_path,
                                                                       "expansion_param.json")
        self.org_nodes_nics_list = None
        self.addtion_node = [node.get("nodeName", "") for node in kvs.get("changed_node_vm_res", [])
                             if node.get("nodeName", "") not in kvs.get("old_node_list", [])]
        if not self.addtion_node:
            self.addtion_node = self.expansion_param_dict.get("additions_nodes", [])

    @staticmethod
    def get_node_ip_info(old_controls, node_name, nic, ip_type="IPv4", ipv6_float=False):
        """
        功能：对于已经填充过信息，重新网卡数据时需要从旧的数据中取出填充到新计算的元素上，避免重复填充
        :param old_controls:
        :param node_name:
        :param nic:
        :param ip_type:
        :param ipv6_float:
        :return:
        """
        prefix = "node"
        node_info = {"ip": "", "mask": "", "destgw": "", "dst_net": "", "ipv6": "", "prefix": "",
                     "destip": "", "destmask": "", "destipv6": "", "destprefix": "", "destgwv6": "",
                     "dst_netv6": ""}
        if not old_controls or len(old_controls) <= 3:
            return node_info
        nic_idx, old_idx = FulFillNodesNicsTable.get_nic_index(ip_type, ipv6_float, nic,
                                                               node_name, old_controls)
        for control in old_controls:
            fmt_para = (prefix, old_idx)
            sub_controls = control.get("controls")
            if not sub_controls or len(sub_controls) < nic_idx + 1:
                continue

            if old_idx and control.get("id") == "%s_vm_ip_title_%s" % fmt_para:
                if len(sub_controls) < nic_idx + 1:
                    continue
                node_info["ip"] = sub_controls[nic_idx].get("default")
            if old_idx and control.get("id") == "%s_vm_mask_title_%s" % fmt_para:
                node_info["mask"] = sub_controls[nic_idx].get("default")

            if old_idx and control.get("id") == "%s_vm_destip_title_%s" % fmt_para:
                node_info["destip"] = sub_controls[nic_idx].get("default")

            if old_idx and control.get("id") == "%s_vm_destmask_title_%s" % fmt_para:
                node_info["destmask"] = sub_controls[nic_idx].get("default")

            if old_idx and control.get("id") == "%s_vm_destgw_title_%s" % fmt_para:
                node_info["destgw"] = sub_controls[nic_idx].get("default")

            if old_idx and control.get("id") == "%s_vm_dst_net_title_%s" % fmt_para:
                node_info["dst_net"] = sub_controls[nic_idx].get("default")

        if ip_type == "IPv6":
            node_info["ipv6"] = node_info.pop("ip")
            node_info["prefix"] = node_info.pop("mask")
            node_info["destipv6"] = node_info.pop("destip")
            node_info["destprefix"] = node_info.pop("destmask")
            node_info["destgwv6"] = node_info.pop("destgw")
            node_info["dst_netv6"] = node_info.pop("dst_net")
        return node_info

    @staticmethod
    def get_nic_index(ip_type, ipv6_float, nic, node_name, old_controls):
        """
        获取原来填充的节点的索引和网卡的索引
        :param ip_type:
        :param ipv6_float:
        :param nic:
        :param node_name:
        :param old_controls:
        :return:
        """
        old_idx = ""
        nic_idx = 0
        prefix = "node"
        for control in old_controls:
            if control.get("default") == node_name:
                node_id = control.get("id")
                old_idx = node_id.replace("%s_vm_node_" % prefix, "")
            fmt_para = (prefix, old_idx)
            if old_idx and control.get("default") == "nic" and control.get(
                    "id") == "%s_vm_nic_title_%s" % fmt_para:
                nic_controls = control.get("controls")
                for nic_control in nic_controls:
                    nic_idx = nic_idx + 1
                    if nic_control.get("default") == nic and ip_type == "IPv4":
                        nic_idx = nic_idx - 1
                        break
                    if nic_control.get("default") == nic and ip_type == "IPv6" and \
                            not ipv6_float and "_float" not in nic_control.get("id") and \
                            "v6" in nic_control.get("id"):
                        nic_idx = nic_idx - 1
                        break
                    if ":" not in nic and nic_control.get("default") == nic and ip_type == "IPv6" \
                            and ipv6_float and "_float" in nic_control.get("id"):
                        nic_idx = nic_idx - 1
                        break
        return nic_idx, old_idx

    @staticmethod
    def nodes_nics_list_sort(nodes_nics_list):
        """
        功能：排序对网卡列表进行排序
        :param nodes_nics_list:
        :return:
        """
        nodes_nics_list = sorted(
            nodes_nics_list,
            key=lambda x: x.node_instance if not str(x.node_instance).startswith("NE_Mgr")
            else "NE_Mgr" + str(x.node_instance).replace("NE_Mgr", "").zfill(5))
        for node_nic in nodes_nics_list:
            node_nic.usage_nic_list = sorted(
                node_nic.usage_nic_list, key=lambda x: x.nic)
        return nodes_nics_list

    def fulfill_nics_table(self):
        """
        功能：填充界面虚拟节点网卡表
        :return:
        """
        kvs = self.kvs
        controls = self.old_controls
        # 如果是支持ipv6的场景就隐藏roc_model_node_nic表，显示roc_model_node_nic_ipv6表，反之亦然。
        if "ipv6" not in self.ctrl_set.get("id") and kvs.get("support_ipv6", "No") == "Yes":
            self.ctrl_set.update({"visible": False})
            controls.clear()
            return
        if "ipv6" in self.ctrl_set.get("id") and kvs.get("support_ipv6", "No") != "Yes":
            self.ctrl_set.update({"visible": False})
            controls.clear()
            return
        self.ctrl_set.update({"visible": True})

        new_controls = []
        self.ctrl_set["controls"] = new_controls
        # 对虚拟节点网卡数组按照字母序进行排序
        nodes_nics_list = self.nodes_nics_list_sort(self.nodes_nics_list)

        # 虚拟节点网卡表的节点序号是从kvs中获取的，如果是单管物理机场景没有虚拟节点资源表，
        # 就获取不到虚拟节点序号，需要提前设置节点的索引序号
        if len(nodes_nics_list) < 2 and kvs.get("is_single_scene", "") == "Yes":
            num = 1
            for node_nic in nodes_nics_list:
                node_name = node_nic.node_instance
                if node_name in ["VCSA", "VIM_01", "VIM_02", "VRM_01", "VRM_02"]:
                    continue
                node_idx_ctrl = ConfigControls.get_config(node_name, str(num))
                node_idx_ctrl.update({"visible": False})
                self.kvs.update({node_name: str(num)})
                new_controls.append(node_idx_ctrl)
                num += 1
        self.fulfill_nic_with_nodes_nics_list(nodes_nics_list, new_controls)

    def fulfill_nic_with_nodes_nics_list(self, nodes_nics_list, new_controls):
        """
        功能：遍历节点网卡列表，在遍历过程中逐个填充界面元素
        :param nodes_nics_list:
        :param new_controls:
        :return:
        """
        fc_nodes = [node.get("nodeName", "") for node in self.kvs.get("changed_node_vm_res", [])]
        # 扩容分析的agent节点根据ifit做相应调整
        is_expansion = False
        if self.kvs.get("action", "") in ["expansion", "roc_expansion"]:
            is_expansion = True
            self.filter_nodes_nics_list(nodes_nics_list)
        for node_nic in nodes_nics_list:
            node_name = node_nic.node_instance
            if self.kvs.get("is_support_lvs", "").lower() == "no" and "GW" in node_name \
                    and is_expansion:
                continue
            if self.kvs.get("action", "") in ["expansion", "roc_expansion"] and \
                    self.kvs.get("scene", "") != "C02" and node_name not in fc_nodes:
                continue
            node_idx = self.kvs.get(node_name)
            if node_name in ["VCSA", "VIM_01", "VIM_02", "VRM_01", "VRM_02"]:
                continue
            self.add_config_in_node(new_controls, node_nic, node_idx)

    def add_config_in_node(self, new_controls, node_nic, node_idx):
        """
        功能：在具体的网卡、IP、掩码、目标网段、目的掩码的外层有一个标签用来包含当前节点所有的网卡或者IP等。
        这样做的原因是一个节点的数据在界面的table元素中需要当做一行，然后名称是一列，网卡是一个合并的单元格（包含当前节点所有网卡）、
        IP是一个合并的单元格（包含当前节点所有IP），其他元素亦然。
        :param new_controls:
        :param node_nic:
        :param node_idx:
        :return:
        """
        fmt_para = ("node", node_idx)
        kvs = self.kvs
        custom_conditions = self.custom_conditions
        node_name = node_nic.node_instance
        node_name_opt = ""
        controls_config_dict = dict()
        if node_name in custom_conditions.get("opt_nodes", []):
            node_name_opt = node_name
        controls_config_dict["virtual_node_vm_node"] = \
            ConfigControls.get_config("%s_vm_node_%s" % fmt_para, node_name,
                                      depends_node_name=node_name_opt)
        # 填充controls_dict
        controls_dict = self.fulfill_nic_col_with_usage_nic_list(node_nic, node_idx)

        controls_config_dict["nic_node"] = \
            ConfigControls.get_config("%s_vm_nic_title_%s" % fmt_para, "nic",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("nic_controls")})
        controls_config_dict["iptype_node"] = \
            ConfigControls.get_config("%s_vm_iptype_title_%s" % fmt_para, "iptype",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("iptype_controls")})
        controls_config_dict["usage_node"] = \
            ConfigControls.get_config("%s_vm_usage_title_%s" % fmt_para, "usage",
                                      cfg_type="group_one_line", editable="true",
                                      other={"controls": controls_dict.get("usage_controls"),
                                             "visible": False})
        controls_config_dict["usage_description_node"] = \
            ConfigControls.get_config("%s_vm_usage_description_title_%s" % fmt_para, "usage",
                                      cfg_type="group_one_line", editable="true",
                                      other={"controls":
                                                 controls_dict.get("usage_description_controls"),
                                             "visible": False})
        controls_config_dict["ip_node"] = \
            ConfigControls.get_config("%s_vm_ip_title_%s" % fmt_para, "ip",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("ip_controls")})
        controls_config_dict["mask_node"] = \
            ConfigControls.get_config("%s_vm_mask_title_%s" % fmt_para, "mask",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("mask_controls")})
        controls_config_dict["destip_node"] = \
            ConfigControls.get_config("%s_vm_destip_title_%s" % fmt_para, "destip",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("destip_controls")})
        controls_config_dict["destmask_node"] = \
            ConfigControls.get_config("%s_vm_destmask_title_%s" % fmt_para, "destmask",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("destmask_controls")})
        controls_config_dict["destgw_node"] = \
            ConfigControls.get_config("%s_vm_destgw_title_%s" % fmt_para, "destgw",
                                      cfg_type="group_one_line", editable="true",
                                      depends_node_name=node_name_opt,
                                      other={"controls": controls_dict.get("destgw_controls")})
        controls_tmp = list()

        new_controls.append(controls_config_dict.get("virtual_node_vm_node", {}))
        controls_tmp.append(controls_config_dict.get("virtual_node_vm_node", {}))
        new_controls.append(controls_config_dict.get("nic_node", {}))
        controls_tmp.append(controls_config_dict.get("nic_node", {}))
        if kvs.get("support_ipv6", "No") == "Yes":
            new_controls.append(controls_config_dict.get("iptype_node", {}))
            controls_tmp.append(controls_config_dict.get("iptype_node", {}))

        if not self.kvs.get("action") in ["expansion", "roc_expansion"]:
            new_controls.append(controls_config_dict.get("usage_node", {}))
            controls_tmp.append(controls_config_dict.get("usage_node", {}))
        else:
            new_controls.append(controls_config_dict.get("usage_node", {}))

        new_controls.append(controls_config_dict.get("usage_description_node", {}))
        new_controls.append(controls_config_dict.get("ip_node", {}))
        controls_tmp.append(controls_config_dict.get("ip_node", {}))
        new_controls.append(controls_config_dict.get("mask_node", {}))
        controls_tmp.append(controls_config_dict.get("mask_node", {}))
        new_controls.append(controls_config_dict.get("destip_node", {}))
        controls_tmp.append(controls_config_dict.get("destip_node", {}))
        new_controls.append(controls_config_dict.get("destmask_node", {}))
        controls_tmp.append(controls_config_dict.get("destmask_node", {}))
        new_controls.append(controls_config_dict.get("destgw_node", {}))
        controls_tmp.append(controls_config_dict.get("destgw_node", {}))

        if len(self.nodes_nics_list) < 2 and self.kvs.get("is_single_scene", "") == "Yes":
            ip_config_single_node = \
                ConfigControls.get_config("%s_%s_%s" % (node_nic.node_instance, "maintenance",
                                                        "ip"), "127.0.0.1", cfg_type="ip",
                                          editable="true")

            ip_config_single_node.update({"visible": "false"})
            new_controls.append(ip_config_single_node)

        if self.kvs.get("action") in ["expansion", "roc_expansion"] \
                and node_name not in self.addtion_node:
            AddIncreaseNodeDepends.add_depends_to_configs(controls_tmp)

    def fulfill_nic_col_with_usage_nic_list(self, node_nic, node_idx):
        """
        功能：填充一个节点，根据业务条件设置元素的可编辑，是否必填属性
        :param node_nic:
        :param node_idx:
        :return:
        """
        controls_dict = {
            "nic_controls": [],
            "iptype_controls": [],
            "usage_controls": [],
            "usage_description_controls": [],
            "ip_controls": [],
            "mask_controls": [],
            "destgw_controls": [],
            "destip_controls": [],
            "destmask_controls": []
        }
        custom_conditions = self.custom_conditions
        node_name_opt = ""
        if node_nic.node_instance in custom_conditions.get("opt_nodes", []):
            node_name_opt = node_nic.node_instance
        nic_editable = "true"
        if self.kvs.get("scene") in ["C00"] or self.kvs.get("is_NFVI", "").lower() == "yes" or \
                (self.kvs.get("action", "") in ["expansion", "roc_expansion"] and
                 self.kvs.get("scene") != "C02"):
            nic_editable = "false"
        if self.kvs.get("is_single_scene", "NO").lower() == "yes":
            nic_editable = "false"
        for usage_nic in node_nic.usage_nic_list:
            nic = usage_nic.nic
            dest_editable = "true"
            dest_required = "true"
            if usage_nic.required == "false" or nic.startswith("eth4"):
                dest_editable = "false"
                dest_required = "false"
            if self.kvs.get("action", "") in ["expansion", "roc_expansion"]:
                dest_required = "false"
            if (nic and nic.startswith("eth0") and self.kvs.get("is_single_scene", "") != "Yes") \
                    or self.kvs.get("scene") in ["C02"] or \
                    self.kvs.get("is_single_scene", "") == "Yes":
                dest_required = "false"
            if self.kvs.get("scene") in ["C01", "C02"] and usage_nic.nic == "eth4":
                continue
            dest_show_dict = {
                "node_idx": node_idx,
                "dest_editable": dest_editable,
                "dest_required": dest_required,
                "nic_editable": nic_editable,
                "node_name_opt": node_name_opt
            }

            self.add_nic_row(node_nic, dest_show_dict, usage_nic, controls_dict)
            self.add_nic_row_with_float(node_nic, dest_show_dict, usage_nic, controls_dict)

        return controls_dict

    def add_nic_row_with_float(self, node_nic, dest_show_dict, usage_nic, controls_dict):
        """
        功能：添加节点的浮动网络信息
        :param node_nic:
        :param dest_show_dict:
        :param usage_nic:
        :param controls_dict:
        :return:
        """
        float_n = 1
        float_usage_nic_list = []

        nic = usage_nic.nic
        if hasattr(usage_nic, "float_usage_nic_list"):
            float_usage_nic_list = usage_nic.float_usage_nic_list
        float_usage = usage_nic.usage_type + "_float"
        for float_usage_nic in float_usage_nic_list:
            if float_usage_nic.usage in self.custom_conditions.get("analyze_float_nic_dict", []):
                if float_usage_nic.nic:
                    if float_usage_nic.nic.split(":"):
                        float_usage_nic.nic = \
                            float_usage_nic.nic.split(":")[0] + ":" + \
                            self.custom_conditions.get("analyze_float_nic_dict", []). \
                                get(float_usage_nic.usage, "")
            float_usage_nic.required = usage_nic.required
            float_dest_show_dict = copy.deepcopy(dest_show_dict)
            float_dest_show_dict["dest_editable"] = "false"
            float_dest_show_dict["dest_required"] = "false"
            float_dest_show_dict["dest_required"] = "false"
            if not usage_nic.is_ipv6:
                node_ip_info = \
                    self.get_node_ip_info(self.old_controls,
                                          node_nic.node_instance, float_usage_nic.nic, "IPv4")
                if self.kvs.get("scene") in ["C01", "C02"] and \
                        self.kvs.get("is_single_scene", "") == "Yes":
                    float_usage_nic.nic = float_usage_nic.nic.replace(":1", ":0")
                    node_ip_info = self.get_node_ip_info(
                        self.old_controls, node_nic.node_instance, nic, "IPv4")
                elif self.kvs.get("scene") in ["C00"] and \
                        self.kvs.get("is_single_scene", "") == "Yes":
                    float_usage_nic.nic = \
                        float_usage_nic.nic.replace("eth", "bond").replace(":1", ":0")
                else:
                    float_usage_nic.nic = float_usage_nic.nic
                nic_float = "%s_%s%d" % (
                    nic.split(":")[0].replace("bond", "eth"), float_usage, float_n)
                float_dest_show_dict["fmt_para"] = (
                    "node", nic_float, dest_show_dict.get("node_idx", ""))
                float_dest_show_dict["node_ip_info"] = node_ip_info
                # 浮动IP依赖的非浮动IP的信息
                float_dest_show_dict['depend_nic_info'] = dest_show_dict
                # float_usage_nic.float_index_usage
                self.fulfill_ip_config(node_nic, float_usage_nic, float_dest_show_dict,
                                       controls_dict)

            elif usage_nic.is_ipv6:
                node_ip_info = self.get_node_ip_info(self.old_controls,
                                                     node_nic.node_instance, nic, "IPv6",
                                                     ipv6_float=True)
                if self.kvs.get("scene") in ["C00"] and \
                        self.kvs.get("is_single_scene", "") == "Yes":
                    nic = nic.replace("eth", "bond")
                nic_float = "%s_%s%d" % (
                    nic.split(":")[0].replace("bond", "eth"), float_usage, float_n)
                fmt_para = ("node", nic_float, dest_show_dict.get("node_idx", ""))
                float_dest_show_dict["fmt_para"] = fmt_para
                float_dest_show_dict["node_ip_info"] = node_ip_info
                # 浮动IP依赖的非浮动IP的信息
                float_dest_show_dict['depend_nic_info'] = dest_show_dict
                # float_usage_nic.float_index_usage
                self.fulfill_ipv6_config(node_nic, float_usage_nic, float_dest_show_dict,
                                         controls_dict)
            float_n += 1

    def add_nic_row(self, node_nic, dest_show_dict, usage_nic, controls_dict):
        """
        功能：添加节点的固定网络信息
        :param node_nic:
        :param dest_show_dict:
        :param usage_nic:
        :param controls_dict:
        :return:
        """
        kvs = self.kvs
        old_controls = self.old_controls
        node_name = node_nic.node_instance
        nic = usage_nic.nic
        if not usage_nic.is_ipv6:
            if kvs.get("scene") in ["C00"] and kvs.get("is_single_scene", "") == "Yes":
                usage_nic.nic = nic.replace("eth", "bond")
                node_ip_info = self.get_node_ip_info(old_controls, node_name,
                                                     nic.replace("eth", "bond"), "IPv4")
            else:
                node_ip_info = self.get_node_ip_info(
                    old_controls, node_name, nic, "IPv4")
            fmt_para = ("node", nic.replace("bond", "eth"), dest_show_dict.get("node_idx", ""))
            dest_show_dict["fmt_para"] = fmt_para
            dest_show_dict["node_ip_info"] = node_ip_info
            self.fulfill_ip_config(node_nic, usage_nic, dest_show_dict, controls_dict)
        elif usage_nic.is_ipv6:
            node_ip_info = self.get_node_ip_info(
                old_controls, node_name, nic, "IPv6")
            if kvs.get("scene") in ["C00"] and kvs.get("is_single_scene", "") == "Yes":
                usage_nic.nic = nic.replace("eth", "bond")
            fmt_para = ("node", nic.replace("bond", "eth"), dest_show_dict.get("node_idx", ""))
            dest_show_dict["fmt_para"] = fmt_para
            dest_show_dict["node_ip_info"] = node_ip_info
            self.fulfill_ipv6_config(node_nic, usage_nic, dest_show_dict, controls_dict)

    def fulfill_ip_config(self, node_nic, usage_nic, dest_show_dict, controls_dict):
        """
        功能：填充Ipv4的网络信息
        :param node_nic:
        :param usage_nic:
        :param dest_show_dict:
        :param controls_dict:
        :return:
        """
        fmt_para = dest_show_dict.get("fmt_para", ())
        node_name_opt = dest_show_dict.get("node_name_opt", "")
        base_usage = ""
        if hasattr(usage_nic, "float_index_usage") and usage_nic.float_index_usage:
            base_usage = usage_nic.float_index_usage
        elif hasattr(usage_nic, "base_usage") and usage_nic.base_usage:
            base_usage = usage_nic.base_usage
        if self.kvs.get("action") in ["expansion", "roc_expansion"]:
            nic = self.get_eth_from_usage_type(usage_nic.usage_type, usage_nic.nic)
            if nic not in fmt_para[1]:
                nic_tag = fmt_para[1].split("_")
                nic_tag[0] = nic
                fmt_para = (fmt_para[0], "_".join(nic_tag), fmt_para[2])
        nic_config = \
            ConfigControls.get_config("%s_vm_%s_nic_%s" % fmt_para, usage_nic.nic,
                                      cfg_type="string", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("nic_editable", ""),
                                      depend_nic_info=dest_show_dict.get('depend_nic_info'),
                                      other={'regexp': '^[^ ]*$',
                                             'errinfo': 'netcard_name_error'})
        controls_dict.get("nic_controls").append(nic_config)
        ConfigControls.set_roc_ids(
            nic_config, usage_nic.usage, node_nic.node_instance, "nic", base_usage)

        controls_dict.get("iptype_controls").append(
            ConfigControls.get_config("%s_vm_%s_iptype_%s" % fmt_para, "IPv4", required="false",
                                      depends_node_name=node_name_opt))
        description = usage_nic.usage_description_en
        if isinstance(description, list):
            description = ",".join(description)
        if not description:
            description = usage_nic.usage
        controls_dict.get("usage_controls").append(
            ConfigControls.get_config("%s_vm_%s_usage_%s" % fmt_para, usage_nic.usage))
        controls_dict.get("usage_description_controls").append(
            ConfigControls.get_config("%s_vm_%s_usage_description_%s" % fmt_para, description,
                                      text=usage_nic.usage_description_en,
                                      text_zh=usage_nic.usage_description_zh,
                                      text_en=usage_nic.usage_description_en,
                                      depends_node_name=node_name_opt))
        ip_config = \
            ConfigControls.get_config("%s_vm_%s_ip_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("ip"),
                                      cfg_type="ip", editable="true",
                                      depends_node_name=node_name_opt,
                                      required=usage_nic.required)
        ConfigControls.set_roc_ids(
            ip_config, usage_nic.usage, node_nic.node_instance, "ip", base_usage)
        controls_dict.get("ip_controls").append(ip_config)

        mask_config = \
            ConfigControls.get_config("%s_vm_%s_mask_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("mask"),
                                      cfg_type="ip", editable="true",
                                      depends_node_name=node_name_opt,
                                      required=usage_nic.required, other={"isMask": "true"})
        ConfigControls.set_roc_ids(
            mask_config, usage_nic.usage, node_nic.node_instance, "mask", base_usage)
        controls_dict.get("mask_controls").append(mask_config)
        destip_config = \
            ConfigControls.get_config("%s_vm_%s_destip_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destip"),
                                      cfg_type="ip", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      required=dest_show_dict.get("dest_required", ""))
        ConfigControls.set_roc_ids(
            destip_config, usage_nic.usage, node_nic.node_instance, "destip", base_usage)
        controls_dict.get("destip_controls").append(destip_config)
        destmask_config = \
            ConfigControls.get_config("%s_vm_%s_destmask_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destmask"),
                                      cfg_type="ip", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      required=dest_show_dict.get("dest_required", ""),
                                      other={"isMask": "true"})
        ConfigControls.set_roc_ids(destmask_config, usage_nic.usage, node_nic.node_instance,
                                   "destmask", base_usage)
        controls_dict.get("destmask_controls").append(destmask_config)
        destgw_config = \
            ConfigControls.get_config("%s_vm_%s_destgw_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destgw"),
                                      cfg_type="ip", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      required=dest_show_dict.get("dest_required", ""))
        ConfigControls.set_roc_ids(
            destgw_config, usage_nic.usage, node_nic.node_instance, "destgw", base_usage)
        controls_dict.get("destgw_controls").append(destgw_config)

    @staticmethod
    def fulfill_ipv6_config(node_nic, usage_nic, dest_show_dict, controls_dict):
        """
        功能：填充ipv6的网络信息
        :param node_nic:
        :param usage_nic:
        :param dest_show_dict:
        :param controls_dict:
        :return:
        """
        node_name_opt = dest_show_dict.get("node_name_opt", "")
        fmt_para = dest_show_dict.get("fmt_para", ())
        base_usage = ""
        if hasattr(usage_nic, "float_index_usage") and usage_nic.float_index_usage:
            base_usage = usage_nic.float_index_usage
        elif hasattr(usage_nic, "base_usage") and usage_nic.base_usage:
            base_usage = usage_nic.base_usage
        description = usage_nic.usage_description_en
        if not description:
            description = usage_nic.usage
        if isinstance(description, list):
            description = ",".join(description)
        nic_config = ConfigControls.get_config("%s_vm_%s_nicv6_%s" % fmt_para, usage_nic.nic,
                                               cfg_type="string",
                                               editable=dest_show_dict.get("nic_editable", ""),
                                               required=usage_nic.required,
                                               depends_node_name=node_name_opt,
                                               other={'regexp': '^[^ ]*$',
                                                      'errinfo': 'netcard_name_error'})
        ConfigControls.set_roc_ids(
            nic_config, usage_nic.usage, node_nic.node_instance, "nicv6", base_usage)
        controls_dict.get("nic_controls").append(nic_config)
        controls_dict.get("iptype_controls").append(
            ConfigControls.get_config("%s_vm_%s_iptypev6_%s" % fmt_para, "IPv6", cfg_type="string",
                                      required="false", editable="false",
                                      depends_node_name=node_name_opt))

        controls_dict.get("usage_controls").append(
            ConfigControls.get_config("%s_vm_%s_usagev6_%s" % fmt_para, usage_nic.usage))
        controls_dict.get("usage_description_controls").append(
            ConfigControls.get_config("%s_vm_%s_usagev6_description_%s" % fmt_para, description,
                                      text=description,
                                      text_zh=usage_nic.usage_description_zh,
                                      text_en=usage_nic.usage_description_en,
                                      depends_node_name=node_name_opt))
        ip_config = ConfigControls.get_config("%s_vm_%s_ipv6_%s" % fmt_para,
                                              dest_show_dict.get("node_ip_info", {}).get("ipv6"),
                                              cfg_type="ip", depends_node_name=node_name_opt,
                                              ext_type="v6", editable="true",
                                              required=usage_nic.required)
        ConfigControls.set_roc_ids(
            ip_config, usage_nic.usage, node_nic.node_instance, "ipv6", base_usage)
        controls_dict.get("ip_controls").append(ip_config)

        mask_config = \
            ConfigControls.get_config("%s_vm_%s_prefix_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("prefix"),
                                      ext_type="v6", depends_node_name=node_name_opt,
                                      cfg_type="ip", required=usage_nic.required,
                                      editable="true", other={"isMask": "true"})
        ConfigControls.set_roc_ids(
            mask_config, usage_nic.usage, node_nic.node_instance, "prefix", base_usage)
        controls_dict.get("mask_controls").append(mask_config)

        destip_config = \
            ConfigControls.get_config("%s_vm_%s_destipv6_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destipv6"),
                                      cfg_type="ip", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      required="false", ext_type="v6")
        ConfigControls.set_roc_ids(destip_config, usage_nic.usage, node_nic.node_instance,
                                   "destipv6", base_usage)
        controls_dict.get("destip_controls").append(destip_config)

        destmask_config = \
            ConfigControls.get_config("%s_vm_%s_destprefix_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destprefix"),
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      cfg_type="ip", required="false", ext_type="v6",
                                      depends_node_name=node_name_opt, other={"isMask": "true"})
        ConfigControls.set_roc_ids(destmask_config, usage_nic.usage, node_nic.node_instance,
                                   "destprefix",
                                   base_usage)
        controls_dict.get("destmask_controls").append(destmask_config)

        destgw_config = \
            ConfigControls.get_config("%s_vm_%s_destgwv6_%s" % fmt_para,
                                      dest_show_dict.get("node_ip_info", {}).get("destgwv6"),
                                      cfg_type="ip", depends_node_name=node_name_opt,
                                      editable=dest_show_dict.get("dest_editable", ""),
                                      required="false", ext_type="v6")
        ConfigControls.set_roc_ids(destgw_config, usage_nic.usage, node_nic.node_instance,
                                   "destgwv6", base_usage)
        controls_dict.get("destgw_controls").append(destgw_config)

    def exe(self):
        """
        填充前台网卡表格入口
        :return:
        """
        # 填充低阶详情网卡列表
        mgr_network = self.kvs.get("mgr_network", "")

        # 导入网卡的json数据，转换为对象列表
        nodes_nics_list = LoadNodeNics.load_nodes_nics_json(
            self.src_dir_path, mgr_network)

        # 获取包含所有条件的的网卡列表
        get_contions_nics = GetConditionsNics(
            nodes_nics_list, self.kvs, self.custom_conditions)
        self.nodes_nics_list = get_contions_nics.exe()

        if self.kvs.get("action", "") in ["expansion", "roc_expansion"]:
            task_id = self.kvs.get("task_id")
            org_node_nics_path = os.path.join(settings.TEMP_DIR, "source_info", str(task_id),
                                              "org_product_info",
                                              "org_node_nics_list.json")
            if os.path.exists(org_node_nics_path):
                nic_list = json.load(open(org_node_nics_path, 'r', encoding="utf8"))
                self.org_nodes_nics_list = LoadNodeNics.generate_node_nics_data(nic_list)
                if self.kvs.get("is_single_scene", "").lower() != "yes":
                    self.nodes_nics_list = NicsBusiness.merge_org_nic_list(self.nodes_nics_list,
                                                                           self.org_nodes_nics_list)
                else:
                    self.nodes_nics_list = self.org_nodes_nics_list
        # 对浮动网卡添加生成key的编号
        set_nic_index_usage = SetNicIndexUsage(self.kvs, self.nodes_nics_list)
        set_nic_index_usage.exe()

        # 填充虚拟节点网络表格
        self.fulfill_nics_table()
        self.change_to_network_plane_single()

    def filter_nodes_nics_list(self, nodes_nics_list):
        """
        不支持ifit 去掉nodes_nics_list分析agent节点的eth2网卡
        :param nodes_nics_list:
        :return:
        """
        if self.kvs.get("support_ifit", "") == "Yes" \
                or self.kvs.get("mgr_network", "") == "double_network":
            return
        for node_nics in nodes_nics_list:
            if "Analyzer_Collector_Agent" in node_nics.node_instance:
                usage_nic_list_tmp = copy.deepcopy(node_nics.usage_nic_list)
                for usage_nic in usage_nic_list_tmp:
                    if usage_nic.nic and usage_nic.nic.startswith("eth2"):
                        node_nics.usage_nic_list.remove(usage_nic)

    def change_to_network_plane_single(self):
        """
        增加网络平面配置
        :return:
        """
        if self.ctrl_set.get("name") != "network_plane_config_single":
            return

        eth0 = list()
        eth1 = list()
        eth2 = list()
        eth3 = list()

        for node_id in self.ctrl_set["controls"]:
            if node_id.get("id").find("nic_title") >= 0:
                for key in node_id["controls"]:
                    if key.get("default") == "eth0":
                        eth0.append(key.get("id"))
                    if key.get("default") == "eth1":
                        eth1.append(key.get("id"))
                    if key.get("default") == "eth2":
                        eth2.append(key.get("id"))
                    if key.get("default") == "eth3":
                        eth3.append(key.get("id"))

        del self.ctrl_set["controls"]
        controls = list()

        controls = FulFillNodesNicsTable.get_north_eth_and_network_plane(controls, eth0)
        controls = FulFillNodesNicsTable.get_alarm_eth_and_network_plane(controls, eth1)
        controls = FulFillNodesNicsTable.get_south_eth_and_network_plane(controls, eth2)
        controls = FulFillNodesNicsTable.get_hard_eth_and_network_plane(controls, eth3)

        self.ctrl_set["controls"] = controls

    @staticmethod
    def get_north_eth_and_network_plane(controls, eth):
        """
        增加北向的网络平面
        :param controls:
        :return:
        """
        if eth:
            netcard_north_lab = {
                'id': 'netcard_north_lab',
                'name': 'netcard_north_lab',
                'type': 'labelfield',
                'default': '客户端/北向网络',
                'default_zh': '客户端/北向网络',
                'default_en': 'Client-side/Northbound Network',
                'required': 'true'
            }
            controls.append(netcard_north_lab)

            count_nic = 0
            for eth0 in eth:
                node_vm_eth0_nic_2 = {
                    'id': 'netcard_north_single',
                    'name': 'netcard_north_single',
                    'type': 'string',
                    'copyTo': eth0,
                    'default': 'eth0',
                    'required': 'true',
                    'visible': "false"
                }
                if count_nic == len(eth) - 1:
                    node_vm_eth0_nic_2['visible'] = "true"
                count_nic = count_nic + 1
                controls.append(node_vm_eth0_nic_2)
        return controls

    @staticmethod
    def get_alarm_eth_and_network_plane(controls, eth):
        """
        增加北向的网络平面
        :param controls:
        :return:
        """
        if eth:
            netcard_alarm_lab = {
                'id': 'netcard_alarm_lab',
                'name': 'netcard_alarm_lab',
                'type': 'labelfield',
                'default': '内部通信网络',
                'default_zh': '内部通信网络',
                'default_en': 'Internal communication network',
                'required': 'true'
            }
            controls.append(netcard_alarm_lab)

            count_nic = 0

            for eth1 in eth:
                node_vm_eth1_nic_2 = {
                    'id': 'netcard_alarm_single',
                    'name': 'netcard_alarm_single',
                    'type': 'string',
                    'copyTo': eth1,
                    'default': 'eth1',
                    'required': 'true',
                    'visible': "false"
                }
                if count_nic == len(eth) - 1:
                    node_vm_eth1_nic_2['visible'] = "true"
                count_nic = count_nic + 1
                controls.append(node_vm_eth1_nic_2)
        return controls

    @staticmethod
    def get_south_eth_and_network_plane(controls, eth):
        """
        增加北向的网络平面
        :param controls:
        :return:
        """
        if eth:
            netcard_south_lab = {
                'id': 'netcard_south_lab',
                'name': 'netcard_south_lab',
                'type': 'labelfield',
                'default': '南向网络',
                'default_zh': '南向网络',
                'default_en': 'Southbound Network',
                'required': 'true'
            }
            controls.append(netcard_south_lab)

            count_nic = 0

            for eth2 in eth:
                node_vm_eth2_nic_2 = {
                    'id': 'netcard_south_single',
                    'name': 'netcard_south_single',
                    'type': 'string',
                    'copyTo': eth2,
                    'default': 'eth2',
                    'required': 'true',
                    'visible': "false"
                }
                if count_nic == len(eth) - 1:
                    node_vm_eth2_nic_2['visible'] = "true"
                count_nic = count_nic + 1
                controls.append(node_vm_eth2_nic_2)
        return controls

    @staticmethod
    def get_hard_eth_and_network_plane(controls, eth):
        """
        增加硬件管理网络平面
        :param controls:
        :return:
        """
        if eth:
            netcard_alarm_lab = {
                'id': 'netcard_hard_lab',
                'name': 'netcard_hard_lab',
                'type': 'labelfield',
                'default': '硬件管理网络',
                'default_zh': '硬件管理网络',
                'default_en': 'Hardware Management Network',
                'required': 'true'
            }
            controls.append(netcard_alarm_lab)

            count_nic = 0

            for eth3 in eth:
                node_vm_eth3_nic_2 = {
                    'id': 'netcard_hard_single',
                    'name': 'netcard_hard_single',
                    'type': 'string',
                    'copyTo': eth3,
                    'default': 'eth3',
                    'required': 'true',
                    'visible': "false"
                }
                if count_nic == len(eth) - 1:
                    node_vm_eth3_nic_2['visible'] = "true"
                count_nic = count_nic + 1
                controls.append(node_vm_eth3_nic_2)
        return controls

    @staticmethod
    def get_eth_from_usage_type(usage_type, nic):
        """
        根据用途反推网卡
        :param usage_type: 用途类型
        :param nic: 实际网卡
        :return:
        """
        if usage_type == "southbound_network":
            return nic.split(":")[0] if nic.split(":")[0] \
                                        in USAGE_TO_NIC.get("southbound_network") else "eth2"
        return USAGE_TO_NIC.get(usage_type, "")
