#! /bin/bash
# ********************************************** ***************************** #
# File Name: log_config.sh
# Description: 修改rsyslog和logrotate的配置
# **************************************************************************** #

# 变量定义
# rsyslog配置文件名
RSYSLOG_CONF_FILENAME="pangeaSASRsyslog.conf"
# rsyslog配置文件路径
SYSLOG_CONFIG_DIR="/etc/rsyslog.d"
# shell脚本日志打印的文件
LOG_FILE="/var/log/pangea_internal_upgrade/mediaproto/sas_rpm_log"
# 脚本日志打印前缀
LOG_PREFIX="[PANGEA SAS]"
# logrorate配置文件名
LOGRORATE_CONF_FILENAME="pangeaSASLogRorate.conf"
# logrotate配置文件路径
LOGRORATE_CONFIG_DIR="/etc/logrotate.d"

# **************************************************************************** #
# Function Name: force_check_pre_step
# Description: 检查上一步是否返回0，失败直接退出
# Parameter: $1 当前操作的描述
# **************************************************************************** #
function force_check_pre_step()
{
    if [ $? -ne 0 ]
    then
        print_log "[ERR] $1 failed."
        exit 1
    fi
    print_log "[INFO] $1 success."
    return 0
}

# **************************************************************************** #
# Function Name: exec_and_retry
# Description: 重试 $2 次 函数 $1
# Parameter: $1 回调函数 $2 重试次数 $3 当前操作的描述
# **************************************************************************** #
function exec_and_retry()
{
    local i=0
    local ret=0
    while [ $i -le $2 ]
    do
        let "i++"
        eval "$1"
        ret=$?
        # 如果执行成功就退出循环
        if [ $ret -eq 0 ]
        then
            print_log "[INFO] $3 success."
            return 0
        fi
        print_log "[WARN] $3 exec failed, ret: $ret, retry..."
    done
    print_log "[ERR] $3 tried $2 times failed, exit."
}

# **************************************************************************** #
# Function Name: print_log
# Description: 此shell脚本打印日志
# Parameter: $1 要打印的字符串内容
# **************************************************************************** #
function print_log()
{
    if [ -e ${LOG_FILE} ]; then
        # 打印到 脚本日志文件中
        echo "${LOG_PREFIX}"$1 >> ${LOG_FILE}
    fi

    # 打印屏显
    echo "$1"
}

# **************************************************************************** #
# Function Name: set_syslog_config
# Description: 修改rsyslog配置
# **************************************************************************** #
function set_syslog_config()
{
    print_log "[INFO] start set syslog config."
    # 安装rpm时的配置文件路径
    local rsyslog_conf_filepath=$(readlink -f ${RSYSLOG_CONF_FILENAME})
    local rsyslogN=$(systemctl -l | grep "rsyslog.service" -c)
    # 判断是否有rsyslog服务
    if [ ${rsyslogN} -lt 1 ]
    then
        print_log "[ERR] rsyslog.service not install."
        return 1
    fi

    if [ -e $SYSLOG_CONFIG_DIR ]
    then
        # 将配置文件拷贝到/etc/rsyslog.d/
        cp -rf ${rsyslog_conf_filepath} ${SYSLOG_CONFIG_DIR}
        force_check_pre_step "cp ${rsyslog_conf_filepath} to ${SYSLOG_CONFIG_DIR}"
        # 修改权限
        chmod 640 "${SYSLOG_CONFIG_DIR}/${RSYSLOG_CONF_FILENAME}"
        # 重启rsyslog服务， 重试5次
        exec_and_retry "systemctl restart rsyslog" 5 "start rsyslog service."
    else
        print_log "[ERR] rsyslog config dir: ${SYSLOG_CONFIG_DIR} not exist."
        return 1
    fi
    return 0
}

# **************************************************************************** #
# Function Name: set_logrotate_config
# Description: 修改logrotate配置
# **************************************************************************** #
function set_logrotate_config()
{
    print_log "[INFO] start set logrotate config."
    # 安装rpm时的配置文件路径
    local logrotate_conf_filepath=$(readlink -f ${LOGRORATE_CONF_FILENAME})
    local crondN=$(systemctl -l | grep "crond.service" -c)

    # 判断是否有crond服务
    if [ ${crondN} -lt 1 ]
    then
        print_log "[ERR] crond.services not install."
        return 1
    fi

    if [ -e $LOGRORATE_CONFIG_DIR ]
    then
        # 将配置文件拷贝到/etc/logrotate.d
        cp -rf ${logrotate_conf_filepath} ${LOGRORATE_CONFIG_DIR}
        force_check_pre_step "cp logrotate ${logrotate_conf_filepath} to ${LOGRORATE_CONFIG_DIR}"
        # 修改权限
        chmod 640 "${LOGRORATE_CONFIG_DIR}/${LOGRORATE_CONF_FILENAME}"
        # 重启crond服务，重试5次
        exec_and_retry "systemctl restart crond" 5 "start logrotate service"
        # 执行logrotate命令让配置生效
        exec_and_retry "logrotate -vf ${LOGRORATE_CONFIG_DIR}/${LOGRORATE_CONF_FILENAME} > /dev/null 2>&1" 5 "active logrotate"
    else
        print_log "[ERR] logrotate config dir: ${LOGRORATE_CONFIG_DIR} not exist."
        return 1
    fi
    return 0
}

# **************************************************************************** #
# Function Name: add_config
# Description: 增加rsyslog和logrotate配置
# **************************************************************************** #
function add_config()
{
    print_log "[INFO] start to add config."
    # 修改rsyslog配置
    set_syslog_config
    force_check_pre_step "set syslog config"
    # 修改logrotate配置
    set_logrotate_config
    force_check_pre_step "set logrotate config"
    print_log "[INFO] add config success."
    return 0
}

# **************************************************************************** #
# Function Name: remove_syslog_config
# Description: 删除rsyslog配置
# **************************************************************************** #
function remove_syslog_config()
{
    if [ -e $SYSLOG_CONFIG_DIR ]
    then
        # 将配置文件删除
        rm -rf "${SYSLOG_CONFIG_DIR}/${RSYSLOG_CONF_FILENAME}"
        # 重启rsyslog服务
        force_check_pre_step "rm -rf ${SYSLOG_CONFIG_DIR}/${RSYSLOG_CONF_FILENAME}"
    else
        print_log "[ERR] rsyslog config dir: ${SYSLOG_CONFIG_DIR} not exist."
        return 1
    fi

    return 0
}

# **************************************************************************** #
# Function Name: remove_logrotate_config
# Description: 删除logrotate配置
# **************************************************************************** #
function remove_logrotate_config()
{
    if [ -e $LOGRORATE_CONFIG_DIR ]
    then
        # 删除配置
        rm -rf "${LOGRORATE_CONFIG_DIR}/${LOGRORATE_CONF_FILENAME}"
        # 重启crond服务
        force_check_pre_step "rm -rf ${LOGRORATE_CONFIG_DIR}/${LOGRORATE_CONF_FILENAME}"
    else
        print_log "[ERR] rsyslog config dir: ${LOGRORATE_CONFIG_DIR} not exist."
        return 1
    fi

    return 0
}

# **************************************************************************** #
# Function Name: remove_config
# Description: 删除rsyslog和logrotate配置
# **************************************************************************** #
function remove_config()
{
    print_log "[INFO] start to remove config."
    # 修改rsyslog配置
    remove_syslog_config
    force_check_pre_step "remove rsyslog config"
    # 修改logrotate配置
    remove_logrotate_config
    force_check_pre_step "remove logrotate config"
    print_log "[INFO] remove config success."
    return 0
}

# **************************************************************************** #
# Description: 程序入口
# **************************************************************************** #
while getopts ":ar" opt
do
    case "${opt}" in
        a)
            # 增加配置
            add_config
            exit 0
            ;;
        r)
            # 删除配置
            remove_config
            exit 0
            ;;
        *)
            exit 1
            ;;
    esac
done
