#!/bin/bash

ROOT_DIR="$(dirname $(readlink -f $0))"
LOG_DIR="/var/log/pangea_internal_upgrade/mediaproto"
OPERATE_LOG="${LOG_DIR}/operate_log"
PROGRESS_FILE="${LOG_DIR}/raid_fw_upgrade_progress"

RET_CODE_OK=0
RET_CODE_FAIL=1
RET_CODE_NOT_MATCH=2
RET_CODE_NO_SUCH_DEVICE=4

EVENT_TYPE=$1

declare -A EventTable
EventTable[Firmware,getver]=raid_fw_getver
EventTable[Firmware,upgrade]=raid_fw_upgrade
EventTable[Firmware,getupgraderate]=raid_getupgraderate

declare -A ResultPrint
ResultPrint[getver,NOT_MATCH]="NOT_MATCH"
ResultPrint[getver,GET_FAIL]="GET_FAIL"

function raid_result_print()
{
    str=${ResultPrint[${EVENT_TYPE},$1]}
    if [[ $1 == "success" ]]
    then
        echo "success"
    elif [[ -n "${str}" ]]
    then
        echo "${str}"
    else
        echo "failure"
    fi
}

# ־
function operate_log()
{
    echo "[$(date '+%Y-%m-%d %H:%M:%S')][RAID][$1]" >> "${OPERATE_LOG}"
}

function architecture_check()
{
    echo "${cfg_raid_pkg_name}" | grep -i $(uname -m) >> /dev/null
    if [[ $? == ${RET_CODE_OK} ]]
    then
        return ${RET_CODE_OK}
    else
        raid_result_print "NOT_MATCH"
        operate_log "The RAID FW:${cfg_raid_pkg_name} is not support architecture:$(uname -m)."
        return ${RET_CODE_NOT_MATCH}
    fi
}

function read_version_xml()
{
    if [[ ! -f "${ROOT_DIR}/version.xml" ]];
    then
        operate_log "The RAID card cfg file:${ROOT_DIR}/version.xml is not found."
        raid_result_print "GET_FAIL"
        return ${RET_CODE_FAIL}
    fi

    dos2unix ${ROOT_DIR}/version.xml &> /dev/null

    cfg_tool=$(cat "${ROOT_DIR}/version.xml" |grep -i "<Tool>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}' |sed 's/;/ /g')
    cfg_package_type=$(cat "${ROOT_DIR}/version.xml" |grep -i "<PackageType>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_file=$(cat "${ROOT_DIR}/version.xml" |grep -i "<FileName>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_version=$(cat "${ROOT_DIR}/version.xml" |grep -i "<Version>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_ctrl=$(cat "${ROOT_DIR}/version.xml" |grep -i "<RaidCtrlModel>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}' |sed 's/;/ /g')
    cfg_raid_ctrl_info=$(cat "${ROOT_DIR}/version.xml" |grep -i "<RaidCtrlInfo>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}' |sed 's/;/ /g')
    cfg_raid_pkg_name=$(cat "${ROOT_DIR}/version.xml" |grep -i "<PackageName>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    return ${RET_CODE_OK}
}

function raid_model_match_arcconf()
{
    bus_num=$(cat tmp_raid.txt |grep -i "PCI Address" |awk 'BEGIN{FS=": "}{print $2}' |awk 'BEGIN{FS=":"}{print $2}')
    device_num=$(cat tmp_raid.txt |grep -i "PCI Address" |awk 'BEGIN{FS=": "}{print $2}' |awk 'BEGIN{FS=":"}{print $3}')
    function_num=$(cat tmp_raid.txt |grep -i "PCI Address" |awk 'BEGIN{FS=": "}{print $2}' |awk 'BEGIN{FS=":"}{print $4}')

    #get base info
    lspci -s "${bus_num}:${device_num}.${function_num}" -xx > base_info_luxor.txt
    vendor_id=$(sed -n '2p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $3}')$(sed -n '2p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $2}')
    device_id=$(sed -n '2p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $5}')$(sed -n '2p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $4}')
    sub_vid=$(sed -n '4p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $15}')$(sed -n '4p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $14}')
    sub_did=$(sed -n '4p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $17}')$(sed -n '4p' base_info_luxor.txt |awk 'BEGIN{FS=" "}{print $16}')

    rm -rf base_info_luxor.txt
    #check device info
    if [[ ${cfg_raid_ctrl_info} = "0x${vendor_id}.0x${device_id}.0x${sub_vid}.0x${sub_did}" ]]
    then
        return ${RET_CODE_OK}
    else
        operate_log "The RAID card: 0x${vendor_id}.0x${device_id}.0x${sub_vid}.0x${sub_did} is not match."
        return ${RET_CODE_NOT_MATCH}
    fi
}

function raid_model_match_storcli()
{
    #get base info
    vendor_id=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "^Vendor Id" |awk '{print $4}')
    device_id=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "^Device Id" |awk '{print $4}')
    sub_vid=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "SubVendor Id" |awk '{print $4}')
    sub_did=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "SubDevice Id" |awk '{print $4}')

    device_id_front=$(echo ${device_id} |cut -c 1-2 )
    device_id_end=$(echo ${device_id} |cut -c 3-4 )

    #check device info
    if [[ ${cfg_raid_ctrl_info} = "${vendor_id}.${device_id_front}00${device_id_end}.${sub_vid}.${sub_did}" ]]
    then
        return ${RET_CODE_OK}
    else
        operate_log "The RAID card:${vendor_id}.${device_id_front}00${device_id_end}.${sub_vid}.${sub_did} is not match."
        return ${RET_CODE_NOT_MATCH}
    fi
}

function raid_model_match_hiraid()
{
    #get base info
    vendor_id=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "^Vendor Id" |awk '{print $4}')
    device_id=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "^Device Id" |awk '{print $4}')
    sub_vid=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "SubVendor Id" |awk '{print $4}')
    sub_did=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "SubDevice Id" |awk '{print $4}')

    #check device info
    if [[ ${cfg_raid_ctrl_info} = "${vendor_id}.${device_id}.${sub_vid}" ]]
    then
        operate_log "The RAID card:${vendor_id}.${device_id}.${sub_vid}.${sub_did} check OK."
        return ${RET_CODE_OK}
    else
        operate_log "The RAID card:${vendor_id}.${device_id}.${sub_vid}.${sub_did} is not match."
        return ${RET_CODE_NOT_MATCH}
    fi
}

function hiraid_product_type_check
{
    #ȷϻǷΪӲ˾Ӳ
    local support_products=(STL6SPCX STL6SPCY STL6SPCY55 STL6SPCY44 STL6SPCH01 STL6SPCH02 STL6SPCY91 STL6SPCY92)
    product_type=$(dmidecode -t 2 |grep "Product Name" -i | awk -F ': ' '{print $2}')
    for product in ${support_products[@]}
    do
        if [ "${product_type}" == "${product}" ];
        then
            return ${RET_CODE_OK}
        fi
    done
    return ${RET_CODE_FAIL}
}

function raid_event_handler()
{
    if [[ -f "${ROOT_DIR}/${cfg_tool}" ]];
    then
        chmod +x "${ROOT_DIR}/${cfg_tool}"
    else
        operate_log "The RAID card tool:${ROOT_DIR}/${cfg_tool} not found."
        raid_result_print "GET_FAIL"
        return ${RET_CODE_FAIL}
    fi

    if [[ -f "${ROOT_DIR}/upgrade.sh" ]];
    then
        dos2unix ${ROOT_DIR}/upgrade.sh &> /dev/null
        chmod +x ${ROOT_DIR}/upgrade.sh
    else
        operate_log "The RAID card upgrade script:${ROOT_DIR}/upgrade.sh is not found."
        raid_result_print "GET_FAIL"
        return ${RET_CODE_FAIL}
    fi

    if [[ ${EVENT_TYPE} == "getupgraderate" ]];
    then
        max_time=6
        raid_getupgraderate ${max_time}
        if [[ $? != ${RET_CODE_OK} ]];
        then
            return ${RET_CODE_FAIL}
        else
            return ${RET_CODE_OK}
        fi
    fi

    raid_ctrl_count=0
    case ${cfg_raid_ctrl} in
        LSI3004|LSI3108|LSI3408IMR|LSI3416IMR|LSI3508|LSI3516|LSI9440-8i|LSI9460-8i|LSI9361-8i|LSI9560-8i)
            num_ctrl=$(${ROOT_DIR}/${cfg_tool} show ctrlcount |grep -i "Controller Count" |awk 'BEGIN{FS=" "}{print $4}')
            for ((i=0; i<num_ctrl; i++))
            do
                ${ROOT_DIR}/${cfg_tool} /c$i show all > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
                raid_model_match_storcli
                if [[ $? == ${RET_CODE_OK} ]]
                then
                    let raid_ctrl_count=$raid_ctrl_count+1
                    ${1} "storcli" $i
                    ret=$?
                    if [[ ${ret} != ${RET_CODE_OK} ]];
                    then
                        rm ${ROOT_DIR}/tmp_raid.txt -rf
                        return ${ret}
                    fi
                fi
                rm ${ROOT_DIR}/tmp_raid.txt -rf
            done
        ;;
        SmartRAID3152-8i|SmartHBA2100-8i)
            num_ctrl=$(${ROOT_DIR}/${cfg_tool} list |grep -i "Controllers found" |awk 'BEGIN{FS=" "}{print $3}')
            for ((i=1; i<=num_ctrl; i++))
            do
                ${ROOT_DIR}/${cfg_tool} getconfig $i ad > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
                raid_model_match_arcconf
                if [[ $? == ${RET_CODE_OK} ]]
                then
                    let raid_ctrl_count=$raid_ctrl_count+1
                    ${1} "arcconf" $i
                    ret=$?
                    if [[ ${ret} != ${RET_CODE_OK} ]];
                    then
                        rm ${ROOT_DIR}/tmp_raid.txt -rf
                        return ${ret}
                    fi
                fi
                rm ${ROOT_DIR}/tmp_raid.txt -rf
            done
        ;;
        HIRAID|HIRAID_HBA)
            num_ctrl=$(${ROOT_DIR}/${cfg_tool} show allctrl | grep -i "Controllers Number" | awk '{print $4}')
            hiraid_product_type_check
            #˾ϴHIRAIDHIRAIDĳ.HIRAIDΪ˾ϻ򶫺4ƥɹаװ
            if [ $? == ${RET_CODE_OK} -a ${num_ctrl} == 0 ]
            then
                raid_result_print "NOT_MATCH"
                operate_log "The HIRAID card is not found and no need install hiraid firmware."
                return ${RET_CODE_NO_SUCH_DEVICE}
            fi
            for ((i=0; i<num_ctrl; i++))
            do
                ${ROOT_DIR}/${cfg_tool} c$i show status > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
                raid_model_match_hiraid
                if [[ $? == ${RET_CODE_OK} ]]
                then
                    let raid_ctrl_count=$raid_ctrl_count+1
                    ${1} "hiraid" $i
                    ret=$?
                    if [[ ${ret} != ${RET_CODE_OK} ]];
                    then
                        rm ${ROOT_DIR}/tmp_raid.txt -rf
                        return ${ret}
                    fi
                fi
                rm ${ROOT_DIR}/tmp_raid.txt -rf
            done
        ;;
        *)
            operate_log "The card[${cfg_raid_ctrl}] upgrade is not supported."
            raid_result_print "NOT_MATCH"
            return ${RET_CODE_NOT_MATCH}
        ;;
        esac

    if [[ ${raid_ctrl_count} == 0 ]];
    then
        raid_result_print "NOT_MATCH"
        operate_log "The RAID card ${cfg_raid_ctrl} is not found in the current environment."
        return ${RET_CODE_NOT_MATCH}
    fi
    return ${RET_CODE_OK}
}

function raid_fw_getver()
{
    if [[ $1 = "storcli" ]];
    then
        result=$(cat ${ROOT_DIR}/tmp_raid.txt|grep -iE "^Firmware Version|^FW Version")
        ret=$?
        ver="$2:$(echo "${result}" | awk '{print $4}')"
    elif [[ $1 = "arcconf" ]];
    then
        result=$(cat ${ROOT_DIR}/tmp_raid.txt|grep "Firmware")
        ret=$?
        ver="$2:$(echo "${result}" | awk '{print $3}')"
    elif [[ $1 = "hiraid" ]];
    then
        cur_fw=$(cat ${ROOT_DIR}/tmp_raid.txt | grep "Current FW" | awk -F\| '{print $2}' | awk '{gsub(/ /,"")}1')
        ret=$?
        if [[ ${ret} != ${RET_CODE_OK} ]];
        then
            raid_result_print "GET_FAIL"
            operate_log "Get Current FW failure."
            return ${RET_CODE_FAIL}
        fi
        cur_ver=$(cat ${ROOT_DIR}/tmp_raid.txt | grep "FW Version${cur_fw}" | awk -F\| '{print $2}' | awk '{gsub(/ /,"")}1')
        ret=$?
        ver="$2:${cur_ver}"
    else
        raid_result_print "GET_FAIL"
        operate_log "The raid card tool:${1} is not supported."
        return ${RET_CODE_FAIL}
    fi
    if [[ ${ret} != ${RET_CODE_OK} ]];
    then
        raid_result_print "GET_FAIL"
        operate_log "Get FW version failure."
        return ${RET_CODE_FAIL}
    else
        echo "${ver}"
        return ${RET_CODE_OK}
    fi
}

function raid_fw_upgrade()
{
    if [[ ! -f "${ROOT_DIR}/${cfg_raid_file}" ]];
    then
        operate_log "The RAID card firmware file:${ROOT_DIR}/${cfg_raid_file} is not found."
        raid_result_print "GET_FAIL"
        return ${RET_CODE_FAIL}
    fi

    type strings &> /dev/null
    if [[ $? == ${RET_CODE_OK} ]];
    then
        if [[ $1 = "storcli" ]];
        then
            file_ver=$(strings ${ROOT_DIR}/${cfg_raid_file}|grep "VERSIONS="|awk 'BEGIN{FS="APP_"}{print $2}'|awk 'BEGIN{FS=","}{print $1}')
            if [[ ${cfg_raid_version} != ${file_ver} ]];
            then
                operate_log "The firmware file version is inconsistent with that specified in the version.xml file."
                raid_result_print "GET_FAIL"
                return ${RET_CODE_FAIL}
            fi
        elif [[ $1 = "arcconf" ]];
        then
            file_ver=$(strings ${ROOT_DIR}/${cfg_raid_file}|grep "FW="|awk 'BEGIN{FS="FW="}{print $2}')
            if [[ ! (${cfg_raid_version} =~ ${file_ver}) ]];
            then
                operate_log "The firmware file version is inconsistent with that specified in the version.xml file."
                raid_result_print "GET_FAIL"
                return ${RET_CODE_FAIL}
            fi
        elif [[ $1 = "hiraid" ]];
        then
            : # do nothing
        else
            operate_log "The raid card tool:${1} is not supported."
            raid_result_print "GET_FAIL"
            return ${RET_CODE_FAIL}
        fi
    fi

    echo -e "Start time: $(date '+%Y-%m-%d %H:%M:%S')\nProgress: 0\nEnd time: $(date '+%Y-%m-%d %H:%M:%S')" > ${PROGRESS_FILE}
    setsid sh ${ROOT_DIR}/upgrade.sh "${1}" "${ROOT_DIR}/${cfg_tool}" "$2" "${ROOT_DIR}/${cfg_raid_file}" "${PROGRESS_FILE}" 1>/dev/null 2>&1 &
    if [[ $? == ${RET_CODE_OK} ]];
    then
        raid_result_print "success"
        return ${RET_CODE_OK}
    else
        raid_result_print "GET_FAIL"
        operate_log "The FW script:upgrade.sh exec failure."
        return ${RET_CODE_FAIL}
    fi
}

function raid_getupgraderate()
{
    if [[ ! -f "${PROGRESS_FILE}" ]];
    then
        operate_log "The RAID card upgrade progress file:${PROGRESS_FILE} is not found."
        raid_result_print "GET_FAIL"
        return ${RET_CODE_FAIL}
    fi
    progress_rate=$(cat ${PROGRESS_FILE}|grep "Progress"|awk '{print $NF}')
    if [[ ${progress_rate} = "-1" ]];
    then
        raid_result_print "GET_FAIL"
        operate_log "Get raid FW upgrade rate failure."
        return ${RET_CODE_FAIL}
    elif [[ ${progress_rate} = "100" ]];
    then
        raid_result_print "success"
        return ${RET_CODE_OK}
    fi

    cur_time=$(date '+%Y-%m-%d %H:%M:%S')
    cur_seconds=$(date --date="$cur_time" +%s)
    start_time=$(cat ${PROGRESS_FILE}|grep "Start time"|awk '{print $(NF-1) " " $NF}')
    start_seconds=$(date --date="$start_time" +%s)

    elapse=$((cur_seconds-start_seconds))
    if [[ $elapse -ge ${1} ]];
    then
        progress_rate=99
     else
        progress_rate=$(awk 'BEGIN{printf "%d","'${elapse}'"/"'${1}'"*100}')
    fi
    echo "upgrading ${progress_rate}"
    return ${RET_CODE_OK}
}

# main
function main()
{
    if [ ! -d ${LOG_DIR} ]; then
        mkdir ${LOG_DIR} -p
        chmod 0750 ${LOG_DIR} -R
    fi
    read_version_xml
    if [[ $? != ${RET_CODE_OK} ]];
    then
        return ${RET_CODE_FAIL}
    fi

    architecture_check
    if [[ $? != ${RET_CODE_OK} ]];
    then
        return ${RET_CODE_NOT_MATCH}
    fi

    event=${EventTable[${cfg_package_type},${EVENT_TYPE}]}
    if [[ ${event} = "" ]];
    then
        operate_log "Unsupported operation:${cfg_package_type} ${EVENT_TYPE}."
        return ${RET_CODE_FAIL}
    fi

    raid_event_handler $event
    return $?
}

main "$1"
exit $?
