#!/bin/bash
COMPILE_KERNEL_REL=5.10.0-60.67.0.107.ule3.aarch64
ROOT_DIR="$(dirname $(readlink -f $0))"
LOG_DIR="/var/log/pangea_internal_upgrade/mediaproto"
PROGRESS_FILE="${ROOT_DIR}/raid_upgrade_progress"
LOG_FILE="${LOG_DIR}/raid_install_log"
CUR_EULER_OS_VERSION=$(uname -r | awk -F '.' '{print $(NF-1)"."$NF}')
CUR_KERNEL_REL=$(uname -r)

STATUS_OK=0
STATUS_FAIL=1
STATUS_NOT_MATCH=2
STATUS_NO_SUCH_DEVICE=4

EVENT_TYPE=$1

declare -A EventTable
EventTable[Driver,getver]=raid_driver_getver
EventTable[Driver,upgrade]=raid_driver_upgrade
EventTable[Driver,getupgraderate]=raid_getupgraderate

declare -A ResultPrint
ResultPrint[getver,NOT_MATCH]="NOT_MATCH"
ResultPrint[getver,GET_FAIL]="GET_FAIL"

declare -A RaidHandle
RaidHandle[Driver,megaraid_sas]=handle_megaraid_sas
RaidHandle[Driver,mpt3sas]=handle_mpt3sas
RaidHandle[Driver,smartpqi]=handle_smartpqi
RaidHandle[Driver,hiraid]=handle_hiraid

function raid_result_print()
{
    str=${ResultPrint[${EVENT_TYPE},$1]}
    if [[ $1 == "success" ]]
    then
        echo "success"
    elif [[ -n "${str}" ]]
    then
        echo "${str}"
    else
        echo "failure"
    fi
}

# 操作日志
function operate_log()
{
    echo "[$(date '+%Y-%m-%d %H:%M:%S')][RAID][$1]" >> "${LOG_FILE}"
}

function os_version_check()
{
    is_kernel_match="true"
    if [[ "${COMPILE_KERNEL_REL}" =~ "eulerosv2r9" ]]; then
        if [ "${COMPILE_KERNEL_REL}" != "${CUR_KERNEL_REL}" ]; then
            is_kernel_match="false"
        fi
    else
        # 有明确结论，欧拉R10及以后的版本保证R版本内核态兼容性，即只匹配大版本即可
        local compile_euler_version=$(echo ${COMPILE_KERNEL_REL} | awk -F '.' '{print $(NF-1)"."$NF}')
        if [ "${compile_euler_version}" != "${CUR_EULER_OS_VERSION}" -o ! "${CUR_EULER_OS_VERSION}" ]; then
            is_kernel_match="false"
        fi
    fi
    
    if [ "${is_kernel_match}" != "true" ]; then
        raid_result_print "NOT_MATCH"
        operate_log "Rpm Compile Kernel Ver is ${COMPILE_KERNEL_REL}, not matches Current Kernel Ver ${CUR_KERNEL_REL}"
        exit ${STATUS_NOT_MATCH}
    else
        operate_log "Current Kernel Ver is ${CUR_KERNEL_REL}, Rpm Compile Kernel Ver is ${COMPILE_KERNEL_REL}"
        operate_log "RAID driver kernel version check PASS !"
    fi
    return ${STATUS_OK}
}

function read_version_xml()
{
    if [[ ! -f "${ROOT_DIR}/version.xml" ]];
    then
        operate_log "The RAID card cfg file:${ROOT_DIR}/version.xml is not found."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    dos2unix ${ROOT_DIR}/version.xml &> /dev/null

    cfg_tool=$(cat "${ROOT_DIR}/version.xml" |grep -i "<Tool>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}' |sed 's/;/ /g')
    cfg_package_type=$(cat "${ROOT_DIR}/version.xml" |grep -i "<PackageType>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_file=$(cat "${ROOT_DIR}/version.xml" |grep -i "<FileName>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_version=$(cat "${ROOT_DIR}/version.xml" |grep -i "<Version>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')
    cfg_raid_driver=$(cat "${ROOT_DIR}/version.xml" |grep -i "<RpmName>"|awk 'BEGIN{FS=">"}{print $2}'|awk 'BEGIN{FS="<"}{print $1}')

    return ${STATUS_OK}
}

function raid_driver_match()
{
    #get driver info
    driver_name=$(cat ${ROOT_DIR}/tmp_raid.txt |grep -i "Driver Name" |awk '{print $4}')

    #Check whether the driver is matched
    if [[ ${cfg_raid_driver} == ${driver_name} ]];
    then
        return ${STATUS_OK}
    else
        return ${STATUS_NOT_MATCH}
    fi
}

function handle_megaraid_sas()
{
    num_ctrl=$(${ROOT_DIR}/${cfg_tool} show ctrlcount |grep -i "Controller Count" |awk 'BEGIN{FS=" "}{print $4}')
    for ((i=0; i<num_ctrl; i++))
    do
        # RAID卡驱动查询或升级只需要一次
        ${ROOT_DIR}/${cfg_tool} /c$i show all > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        raid_driver_match
        ret=$?
        if [[ $ret == ${STATUS_OK} ]];
        then
            ${1}
            ret=$?
            rm ${ROOT_DIR}/tmp_raid.txt -rf
            return $ret
        else
            rm ${ROOT_DIR}/tmp_raid.txt -rf
            continue
        fi
    done

    operate_log "The driver[${cfg_raid_driver}] not match the environment."
    raid_result_print "NOT_MATCH"
    return ${STATUS_NOT_MATCH}
}

function handle_mpt3sas()
{
    # 检查是否存在3008RAID卡
    num_ctrl=$(lspci | grep -i LSI | grep 3008 | wc -l)
    if [[ ${num_ctrl} != 0 ]];
    then
        echo "Driver Name = mpt3sas" > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        driver_ver=$(modinfo mpt3sas 2>/dev/null | grep -iw version | awk '{print $NF}')
        if [[ -n ${driver_ver} ]];
        then
            echo "Driver Version = ${driver_ver}" >> ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        fi

        ${1}
        ret=$?
        rm ${ROOT_DIR}/tmp_raid.txt -rf
        return $ret
    fi

    operate_log "The driver[${cfg_raid_driver}] not match the environment."
    raid_result_print "NOT_MATCH"
    return ${STATUS_NOT_MATCH}
}

function handle_smartpqi()
{
    num_ctrl=$(${ROOT_DIR}/${cfg_tool} list |grep -i "Controllers found" |awk 'BEGIN{FS=" "}{print $3}')
    for ((i=1; i<=num_ctrl; i++))
    do
        ${ROOT_DIR}/${cfg_tool} getconfig $i ad > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        raid_driver_match
        ret=$?
        if [[ $ret == ${STATUS_OK} ]];
        then
            ${1}
            ret=$?
            rm ${ROOT_DIR}/tmp_raid.txt -rf
            return $ret
        else
            rm ${ROOT_DIR}/tmp_raid.txt -rf
            continue
        fi
    done

    operate_log "The driver[${cfg_raid_driver}] not match the environment."
    raid_result_print "NOT_MATCH"
    return ${STATUS_NOT_MATCH}
}

function hiraid_product_type_check
{
    #确认环境是否为东海硬件或者司南硬件
    local support_products=(STL6SPCX STL6SPCY STL6SPCY55 STL6SPCY44 STL6SPCH01 STL6SPCH02 STL6SPCY91 STL6SPCY92)
    product_type=$(dmidecode -t 2 |grep "Product Name" -i | awk -F ': ' '{print $2}')
    for product in ${support_products[@]}
    do
        if [ "${product_type}" == "${product}" ];
        then
            return ${STATUS_OK}
        fi
    done
    return ${STATUS_FAIL}
}

# hiraid/hiraid_hba
function handle_hiraid()
{
    # 3758：1880RAID卡 3858:1880HBA卡
    num_ctrl=$(lspci | grep -Ei "Device\s+3[78]58" | wc -l)
    if [[ ${num_ctrl} != 0 ]];
    then
        echo "Driver Name = hiraid" > ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        driver_ver=$(modinfo hiraid 2>/dev/null | grep -iw version | awk '{print $NF}')
        if [[ -n ${driver_ver} ]];
        then
            echo "Driver Version = ${driver_ver}" >> ${ROOT_DIR}/tmp_raid.txt 2>/dev/null
        fi

        ${1}
        ret=$?
        rm ${ROOT_DIR}/tmp_raid.txt -rf
        return $ret
    else
        #东海和司南存在有HIRAID卡和无HIRAID卡的场景,当环境无HIRAID卡但环境为司南或东海，返回4，代表匹配包成功但不进行安装升级
        hiraid_product_type_check
        if [[ $? == ${STATUS_OK} ]]
        then
            operate_log "The HIRAID card is not found and no need install hiraid driver."
            raid_result_print "NOT_MATCH"
            return ${STATUS_NO_SUCH_DEVICE}
        fi
    fi

    operate_log "The driver[${cfg_raid_driver}] not match the environment."
    raid_result_print "NOT_MATCH"
    return ${STATUS_NOT_MATCH}
}

function raid_event_handler()
{
    if [[ -f "${ROOT_DIR}/${cfg_tool}" ]];
    then
        chmod +x "${ROOT_DIR}/${cfg_tool}"
    elif [[ ${cfg_raid_driver} == "mpt3sas" ]];
    then
        # No tools needed for mpt3sas
        :
    else
        operate_log "The RAID card tool:${ROOT_DIR}/${cfg_tool} not found."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    if [[ -f "${ROOT_DIR}/upgrade.sh" ]];
    then
        dos2unix ${ROOT_DIR}/upgrade.sh &> /dev/null
        chmod +x ${ROOT_DIR}/upgrade.sh
    else
        operate_log "The RAID card upgrade script:${ROOT_DIR}/upgrade.sh is not found."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    if [[ ${EVENT_TYPE} = "getupgraderate" ]];
    then
        max_time=90
        raid_getupgraderate ${max_time}
        if [[ $? != ${STATUS_OK} ]];
        then
            return ${STATUS_FAIL}
        else
            return ${STATUS_OK}
        fi
    fi

    local handle=${RaidHandle[${cfg_package_type},${cfg_raid_driver}]}
    if [ -z "${handle}" ]; then
        operate_log "The ${cfg_package_type}[${cfg_raid_driver}] upgrade is not supported."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    ${handle} $1
    return $?
}

function raid_getupgraderate()
{
    if [[ ! -f "${PROGRESS_FILE}" ]];
    then
        operate_log "The RAID card upgrade progress file:${PROGRESS_FILE} is not found."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi
    progress_rate=$(cat ${PROGRESS_FILE}|grep "Progress"|awk '{print $NF}')
    if [[ ${progress_rate} = "-1" ]];
    then
        raid_result_print "GET_FAIL"
        return ${STATUS_OK}
    elif [[ ${progress_rate} = "100" ]];
    then
        raid_result_print "success"
        return ${STATUS_OK}
    fi

    cur_time=$(date '+%Y-%m-%d %H:%M:%S')
    cur_seconds=$(date --date="$cur_time" +%s)
    start_time=$(cat ${PROGRESS_FILE}|grep "Start time"|awk '{print $(NF-1) " " $NF}')
    start_seconds=$(date --date="$start_time" +%s)

    elapse=$((cur_seconds-start_seconds))
    if [[ $elapse -ge ${1} ]];
    then
        progress_rate=99
     else
        progress_rate=$(awk 'BEGIN{printf "%d","'${elapse}'"/"'${1}'"*100}')
    fi
    echo "upgrading ${progress_rate}"
    return ${STATUS_OK}
}

function raid_driver_getver()
{
    ver=$(cat ${ROOT_DIR}/tmp_raid.txt|grep -E "Driver Version|Driver\s+:")
    if [[ $? != ${STATUS_OK} ]];
    then
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    else
        echo "${ver}" | awk '{print $4}'
        return ${STATUS_OK}
    fi
}

function raid_driver_upgrade()
{
    if [[ ! -f "${ROOT_DIR}/${cfg_raid_file}" ]];
    then
        operate_log "The RAID card driver file:${ROOT_DIR}/${cfg_raid_file} is not found."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi
    driver_name=$(cat ${ROOT_DIR}/tmp_raid.txt|grep -i "Driver Name"|awk '{print $4}')
    echo "${cfg_raid_file}"|grep -i "${driver_name}.*.rpm" > /dev/null
    if [[ $? != ${STATUS_OK} ]];
    then
        operate_log "The raid controller driver mismatched(${driver_name})."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    if [[ ! (${cfg_raid_file} =~ ${cfg_raid_version}) ]];
    then
        operate_log "The driver file version is inconsistent with that specified in the version.xml file."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    echo -e "Start time: $(date '+%Y-%m-%d %H:%M:%S')\nProgress: 0\nEnd time: $(date '+%Y-%m-%d %H:%M:%S')" > ${PROGRESS_FILE}
    setsid sh ${ROOT_DIR}/upgrade.sh "${ROOT_DIR}/${cfg_raid_file}" "${PROGRESS_FILE}" 1>/dev/null 2>&1 &
    if [[ $? == ${STATUS_OK} ]];
    then
        raid_result_print "success"
        return ${STATUS_OK}
    else
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi
}

# main函数
function main()
{
    if [ ! -d ${LOG_DIR} ]; then
        mkdir ${LOG_DIR} -p
        chmod 0750 ${LOG_DIR} -R
    fi

    read_version_xml
    if [[ $? != ${STATUS_OK} ]];
    then
        return ${STATUS_FAIL}
    fi

    os_version_check
    if [[ $? != ${STATUS_OK} ]];
    then
        return ${STATUS_NOT_MATCH}
    fi

    event=${EventTable[${cfg_package_type},${EVENT_TYPE}]}
    if [[ ${event} = "" ]];
    then
        operate_log "Unsupported operation:${cfg_package_type} ${EVENT_TYPE}."
        raid_result_print "GET_FAIL"
        return ${STATUS_FAIL}
    fi

    raid_event_handler $event
    return $?
}

main "$1"
exit $?
